"""Helper routines for the Pathé Movie skill."""
from pathlib import Path
import difflib
import json
import urllib.parse
import urllib.request

API_BASE = "https://www.pathe.nl/api"
HEADERS = {
    "User-Agent": "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7)"
                 " AppleWebKit/537.36 (KHTML, like Gecko)"
                 " Chrome/138.0.0.0 Safari/537.36",
    "Accept": "application/json",
}
FILLER_WORDS = {"the", "a", "an", "of", "in", "on", "for", "and"}
CONFIG_PATH = Path(__file__).resolve().parents[1] / "config" / "pathe_movie_config.json"
CONFIG = json.loads(CONFIG_PATH.read_text())
APPROVED_CINEMAS = CONFIG.get("approvedCinemas", [])
PRIMARY_CINEMA = CONFIG.get("primaryCinema") or (APPROVED_CINEMAS[0] if APPROVED_CINEMAS else None)


def _fetch_json(path, query_args=None):
    url = f"{API_BASE}{path}"
    if query_args:
        url += "?" + urllib.parse.urlencode(query_args)
    req = urllib.request.Request(url, headers=HEADERS)
    with urllib.request.urlopen(req) as resp:
        return json.load(resp)


def normalize_query(title):
    parts = [p for p in title.split() if p.lower() not in FILLER_WORDS]
    return " ".join(parts)


def search_movies(name):
    filtered = normalize_query(name)
    return _fetch_json("/search/full", {"q": filtered})


def best_match(name, candidates):
    titles = [movie.get("title", "") for movie in candidates]
    matches = difflib.get_close_matches(name, titles, n=3, cutoff=0.5)
    return [movie for movie in candidates if movie.get("title") in matches]


def get_show(slug):
    return _fetch_json(f"/show/{slug}", {"language": "nl"})


def get_cinemas(slug):
    return _fetch_json(f"/show/{slug}/cinemas", {"language": "nl"})


def get_cinema_details(cinema):
    return _fetch_json(f"/cinema/{cinema}", {"language": "nl"})


def get_showtimes(slug, cinema):
    return _fetch_json(f"/show/{slug}/showtimes/{cinema}", {"language": "en"})


def poster_url(movie_result):
    poster = movie_result.get("poster") or movie_result.get("posterPath")
    if poster:
        return poster.get("lg") or poster.get("md")


def clean_result(movie, query):
    return {
        "slug": movie.get("slug"),
        "title": movie.get("title"),
        "poster": poster_url(movie),
        "rating": movie.get("contentRating"),
    }


def preferred_cinemas(available, include_secondary=False):
    filtered = [cin for cin in available if cin in APPROVED_CINEMAS]
    if not filtered:
        return APPROVED_CINEMAS if include_secondary else ([PRIMARY_CINEMA] if PRIMARY_CINEMA else [])
    if PRIMARY_CINEMA in filtered:
        return [PRIMARY_CINEMA]
    return filtered if include_secondary else filtered[:1]
