/**
 * storage.test.js - 存储模块单元测试
 */

import { describe, it, before, after } from 'node:test';
import assert from 'node:assert/strict';
import { existsSync, mkdirSync, rmSync } from 'fs';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';

import storage, { sanitizeInput } from '../scripts/storage.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

// 测试用的临时目录
const TEST_DIR = join(__dirname, 'test-temp');
const ORIGINAL_DATA_DIR = join(__dirname, '..', 'data');
const ORIGINAL_CACHE_DIR = join(__dirname, '..', '.cache');

describe('storage 模块', () => {
  before(() => {
    // 创建测试目录
    if (!existsSync(TEST_DIR)) {
      mkdirSync(TEST_DIR, { recursive: true });
    }
  });
  
  after(() => {
    // 清理测试文件
    if (existsSync(TEST_DIR)) {
      rmSync(TEST_DIR, { recursive: true, force: true });
    }
  });
  
  describe('sanitizeInput', () => {
    it('应该清理空输入', () => {
      assert.strictEqual(sanitizeInput(null), '');
      assert.strictEqual(sanitizeInput(undefined), '');
      assert.strictEqual(sanitizeInput(''), '');
    });
    
    it('应该限制输入长度', () => {
      const longInput = 'a'.repeat(2000);
      const sanitized = sanitizeInput(longInput, 100);
      assert.strictEqual(sanitized.length, 100);
    });
    
    it('应该移除控制字符', () => {
      const input = 'hello\x00\x01\x02world';
      const sanitized = sanitizeInput(input);
      assert.strictEqual(sanitized, 'helloworld');
    });
    
    it('应该保留正常字符', () => {
      const input = 'Hello, World! 123';
      const sanitized = sanitizeInput(input);
      assert.strictEqual(sanitized, input);
    });
  });
  
  describe('initializeVault', () => {
    it('应该成功初始化密码库', () => {
      const result = storage.initializeVault('test-master-password');
      assert.strictEqual(result.success, true);
      assert(existsSync(join(__dirname, '..', 'data', 'vault.enc')));
    });
    
    it('应该使用加密的缓存', () => {
      storage.initializeVault('test-master-password');
      const cacheFile = join(__dirname, '..', '.cache', 'key.enc');
      assert(existsSync(cacheFile));
      
      // 缓存文件应该是加密的二进制数据，不是纯文本 JSON
      const cacheContent = storage.loadConfig(); // 只是验证文件存在
      // 实际验证需要读取文件并检查格式
    });
  });
  
  describe('getDecryptionKey', () => {
    it('应该在没有缓存时返回 no_cache', () => {
      // 使用唯一测试 ID
      const testId = 'test-gdk-' + Date.now();
      const testPassword = 'password-' + testId;
      
      // 先初始化
      storage.initializeVault(testPassword);
      // 锁定清除缓存
      storage.lockVault();
      
      // 没有缓存时应该返回 no_cache
      const result = storage.getDecryptionKey('any-password');
      assert.strictEqual(result.locked, true);
      assert.strictEqual(result.reason, 'no_cache');
    });
    
    it('应该接受正确的主密码', () => {
      // 使用唯一测试 ID
      const testId = 'test-gdk2-' + Date.now();
      const testPassword = 'password-' + testId;
      
      // 先初始化
      storage.initializeVault(testPassword);
      // 锁定清除缓存
      storage.lockVault();
      
      // 用正确密码解锁
      const result = storage.getDecryptionKey(testPassword);
      assert.strictEqual(result.locked, false, '应该成功解锁');
      assert(Buffer.isBuffer(result.key), '应该返回 Buffer 类型的密钥');
    });
  });
  
  describe('loadVault/saveVault', () => {
    it('应该能够加载和保存密码库', () => {
      // 使用唯一测试 ID
      const testId = 'test-lsv-' + Date.now();
      const testPassword = 'password-' + testId;
      
      // 初始化和获取密钥
      storage.initializeVault(testPassword);
      storage.lockVault(); // 清除缓存
      const auth = storage.getDecryptionKey(testPassword);
      assert(!auth.locked, '应该成功获取密钥');
      
      const vault = storage.loadVault(auth.key);
      assert(vault.entries);
      assert.strictEqual(vault.version, 1);
      
      vault.entries.push({
        id: 'test-entry',
        name: 'test',
        type: 'password',
        password: 'test123',
        tags: [],
        notes: '',
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString()
      });
      
      storage.saveVault(vault, auth.key);
      
      // 重新加载验证
      const vault2 = storage.loadVault(auth.key);
      assert.strictEqual(vault2.entries.length, 1);
      assert.strictEqual(vault2.entries[0].name, 'test');
    });
    
    it('应该创建版本历史', () => {
      // 使用唯一测试 ID
      const testId = 'test-history-' + Date.now();
      const testPassword = 'password-' + testId;
      
      // 初始化和获取密钥
      storage.initializeVault(testPassword);
      storage.lockVault(); // 清除缓存
      const auth = storage.getDecryptionKey(testPassword);
      assert(!auth.locked, '应该成功获取密钥');
      
      const vault = storage.loadVault(auth.key);
      vault.entries.push({
        id: 'test-entry',
        name: 'test',
        type: 'password',
        password: 'test123',
        tags: [],
        notes: '',
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString()
      });
      
      storage.saveVault(vault, auth.key);
      
      // 检查历史文件
      const history = storage.listHistory();
      assert(history.length >= 0); // 可能为 0 如果是第一次保存
    });
  });
  
  describe('lockVault', () => {
    it('应该清除缓存文件', () => {
      storage.initializeVault('test-password');
      const cacheFile = join(__dirname, '..', '.cache', 'key.enc');
      
      assert(existsSync(cacheFile));
      
      storage.lockVault();
      
      assert(!existsSync(cacheFile));
    });
  });
  
  describe('restoreVault', () => {
    it('应该拒绝不存在的文件', () => {
      const result = storage.restoreVault('/nonexistent/path.enc');
      assert.strictEqual(result.success, false);
    });
    
    it('应该拒绝无效格式的文件', () => {
      // 创建一个无效文件
      const invalidFile = join(TEST_DIR, 'invalid.enc');
      storage.saveConfig({ test: true }); // 创建一个小文件
      
      const result = storage.restoreVault(invalidFile);
      assert.strictEqual(result.success, false);
    });
  });
});
