/**
 * generator.test.js - 密码生成模块单元测试
 */

import { describe, it } from 'node:test';
import assert from 'node:assert/strict';

import generator from '../scripts/generator.js';

describe('generator 模块', () => {
  describe('generatePassword', () => {
    it('应该生成默认长度的密码', () => {
      const password = generator.generatePassword();
      assert.strictEqual(password.length, 32);
    });
    
    it('应该生成指定长度的密码', () => {
      const password = generator.generatePassword({ length: 16 });
      assert.strictEqual(password.length, 16);
    });
    
    it('应该包含大写字母', () => {
      const password = generator.generatePassword({ 
        length: 32, 
        includeUppercase: true,
        includeNumbers: false,
        includeSymbols: false
      });
      assert.match(password, /[A-Z]/);
    });
    
    it('应该包含数字', () => {
      const password = generator.generatePassword({ 
        length: 32, 
        includeUppercase: false,
        includeNumbers: true,
        includeSymbols: false
      });
      assert.match(password, /[0-9]/);
    });
    
    it('应该包含符号', () => {
      const password = generator.generatePassword({ 
        length: 32, 
        includeUppercase: false,
        includeNumbers: false,
        includeSymbols: true
      });
      assert.match(password, /[^a-zA-Z0-9]/);
    });
    
    it('应该排除易混淆字符', () => {
      const password = generator.generatePassword({ 
        length: 32, 
        excludeAmbiguous: true
      });
      assert.doesNotMatch(password, /[0O1lI|]/);
    });
    
    it('每次生成的密码应该不同', () => {
      const passwords = new Set();
      for (let i = 0; i < 10; i++) {
        passwords.add(generator.generatePassword());
      }
      assert.strictEqual(passwords.size, 10);
    });
  });
  
  describe('generateMnemonicPassword', () => {
    it('应该生成默认 4 个单词的助记密码', () => {
      const password = generator.generateMnemonicPassword();
      const words = password.split('-');
      assert.strictEqual(words.length, 4);
    });
    
    it('应该生成指定数量单词的助记密码', () => {
      const password = generator.generateMnemonicPassword(6);
      const words = password.split('-');
      assert.strictEqual(words.length, 6);
    });
    
    it('应该使用自定义词库', () => {
      const customWords = ['apple', 'banana', 'cherry', 'date'];
      const password = generator.generateMnemonicPassword(4, customWords);
      const words = password.split('-');
      words.forEach(word => {
        assert(customWords.includes(word), `单词 "${word}" 不在自定义词库中`);
      });
    });
  });
  
  describe('checkStrength', () => {
    it('应该识别弱密码', () => {
      const result = generator.checkStrength('123456');
      assert.strictEqual(result.level, 'weak');
    });
    
    it('应该识别强密码', () => {
      const result = generator.checkStrength('xK9#mP2$vL5@nQ8!');
      assert(result.score >= 6);
      assert(['strong', 'very-strong'].includes(result.level));
    });
    
    it('应该检测重复字符', () => {
      const result = generator.checkStrength('aaaaaaaa');
      assert(result.feedback.some(f => f.includes('重复')));
    });
    
    it('应该检测连续数字', () => {
      const result = generator.checkStrength('12345678');
      assert(result.feedback.some(f => f.includes('连续数字')));
    });
    
    it('应该检测键盘模式', () => {
      const result = generator.checkStrength('qwerty123');
      assert(result.feedback.some(f => f.includes('键盘模式')));
    });
    
    it('应该检测常见密码', () => {
      const result = generator.checkStrength('password123');
      assert(result.feedback.some(f => f.includes('常见密码')));
    });
    
    it('应该返回正确的特征检测', () => {
      const result = generator.checkStrength('Abc123!@#');
      assert.strictEqual(result.hasLowercase, true);
      assert.strictEqual(result.hasUppercase, true);
      assert.strictEqual(result.hasNumbers, true);
      assert.strictEqual(result.hasSymbols, true);
    });
  });
  
  describe('generateApiKey', () => {
    it('应该生成带前缀的 API Key', () => {
      const apiKey = generator.generateApiKey('sk-', 32);
      assert(apiKey.startsWith('sk-'));
      assert.strictEqual(apiKey.length, 35); // prefix(3) + random(32)
    });
    
    it('应该生成不带符号的 API Key', () => {
      const apiKey = generator.generateApiKey('ghp_', 36);
      assert(apiKey.startsWith('ghp_'));
      assert.match(apiKey.substring(4), /^[a-zA-Z0-9]+$/);
    });
  });
});
