/**
 * crypto.test.js - 加密模块单元测试
 */

import { describe, it } from 'node:test';
import assert from 'node:assert/strict';
import { randomBytes } from 'crypto';

import crypto from '../scripts/crypto.js';

describe('crypto 模块', () => {
  describe('deriveKey', () => {
    it('应该从相同密码派生相同密钥', () => {
      const password = 'test-password-123';
      const { key: key1, salt } = crypto.deriveKey(password);
      const { key: key2 } = crypto.deriveKey(password, salt);
      
      assert.strictEqual(key1.toString('hex'), key2.toString('hex'));
    });
    
    it('应该从不同盐值派生不同密钥', () => {
      const password = 'test-password-123';
      const { key: key1 } = crypto.deriveKey(password);
      const { key: key2 } = crypto.deriveKey(password);
      
      assert.notStrictEqual(key1.toString('hex'), key2.toString('hex'));
    });
  });
  
  describe('encrypt/decrypt', () => {
    it('应该能够加密并解密数据', () => {
      const key = randomBytes(32);
      const data = 'Hello, World!';
      
      const { encrypted, iv, authTag } = crypto.encrypt(key, data);
      const decrypted = crypto.decrypt(key, encrypted, iv, authTag);
      
      assert.strictEqual(decrypted.toString('utf8'), data);
    });
    
    it('应该拒绝错误的 authTag', () => {
      const key = randomBytes(32);
      const data = 'Hello, World!';
      
      const { encrypted, iv, authTag } = crypto.encrypt(key, data);
      const wrongTag = randomBytes(16);
      
      assert.throws(() => {
        crypto.decrypt(key, encrypted, iv, wrongTag);
      });
    });
  });
  
  describe('encryptAndPack/unpackAndDecrypt', () => {
    it('应该能够打包并解包数据', () => {
      const key = randomBytes(32);
      const data = { message: 'Hello', count: 42 };
      
      const packed = crypto.encryptAndPack(key, data);
      const unpacked = crypto.unpackAndDecrypt(packed, key);
      
      assert.deepStrictEqual(unpacked, data);
    });
    
    it('应该支持 Buffer 和 hex 字符串两种密钥类型', () => {
      const key = randomBytes(32);
      const data = { test: 'data' };
      
      const packed = crypto.encryptAndPack(key, data);
      
      // 使用 Buffer 解密
      const result1 = crypto.unpackAndDecrypt(packed, key);
      assert.deepStrictEqual(result1, data);
      
      // 使用 hex 字符串解密
      const result2 = crypto.unpackAndDecrypt(packed, key.toString('hex'));
      assert.deepStrictEqual(result2, data);
    });
    
    it('应该拒绝无效的密钥类型', () => {
      const key = randomBytes(32);
      const data = { test: 'data' };
      const packed = crypto.encryptAndPack(key, data);
      
      assert.throws(() => {
        crypto.unpackAndDecrypt(packed, 123);
      });
    });
  });
  
  describe('deriveCacheKey', () => {
    it('应该派生与普通密钥不同的缓存密钥', () => {
      const password = 'master-password';
      const { key: normalKey } = crypto.deriveKey(password);
      const { key: cacheKey } = crypto.deriveCacheKey(password);
      
      // 缓存密钥应该与普通密钥不同（因为使用了不同的密码后缀）
      assert.notStrictEqual(normalKey.toString('hex'), cacheKey.toString('hex'));
    });
    
    it('应该从相同密码派生相同的缓存密钥（使用相同盐值）', () => {
      const password = 'master-password';
      const { key: cacheKey1, salt } = crypto.deriveCacheKey(password);
      const { key: cacheKey2 } = crypto.deriveCacheKey(password, salt);
      
      assert.strictEqual(cacheKey1.toString('hex'), cacheKey2.toString('hex'));
    });
  });
  
  describe('secureWipe', () => {
    it('应该清除 Buffer 内容', () => {
      const buffer = Buffer.from('sensitive data');
      const originalHex = buffer.toString('hex');
      
      crypto.secureWipe(buffer);
      
      assert.strictEqual(buffer.toString('hex'), '00'.repeat(buffer.length));
      assert.notStrictEqual(buffer.toString('hex'), originalHex);
    });
  });
  
  describe('randomIntUnbiased', () => {
    it('应该返回范围内的随机整数', () => {
      for (let i = 0; i < 100; i++) {
        const result = crypto.randomIntUnbiased(10);
        assert(result >= 0 && result < 10);
      }
    });
    
    it('应该拒绝非正数', () => {
      assert.throws(() => crypto.randomIntUnbiased(0));
      assert.throws(() => crypto.randomIntUnbiased(-1));
    });
  });
  
  describe('randomHex', () => {
    it('应该生成指定长度的十六进制字符串', () => {
      const hex = crypto.randomHex(16);
      assert.strictEqual(hex.length, 32); // 16 bytes = 32 hex chars
      assert.match(hex, /^[0-9a-f]+$/);
    });
  });
});
