/**
 * storage.js - Storage Module
 * 
 * Manages password vault file read/write and version history
 */

import { readFileSync, writeFileSync, existsSync, mkdirSync, readdirSync, unlinkSync, appendFileSync } from 'fs';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';
import crypto from './crypto.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

// Default paths
const DEFAULT_DATA_DIR = join(__dirname, '..', 'data');
const DEFAULT_CACHE_DIR = join(__dirname, '..', '.cache');
const DEFAULT_LOGS_DIR = join(__dirname, '..', '.logs');
const VAULT_FILE = join(DEFAULT_DATA_DIR, 'vault.enc');
const CACHE_FILE = join(DEFAULT_CACHE_DIR, 'key.enc');
const CONFIG_FILE = join(__dirname, '..', 'config.json');
const DETECTION_LOG = join(DEFAULT_LOGS_DIR, 'detection.jsonl');

// Memory cache (avoid repeated decryption)
let vaultCache = null;
let vaultCacheKeyHex = null;

// Input validation constants
const MAX_INPUT_LENGTH = 1024;
const MAX_NAME_LENGTH = 100;
const MAX_NOTES_LENGTH = 1000;
const MAX_TAGS_COUNT = 20;
const MAX_TAG_LENGTH = 50;

// Default configuration
const DEFAULT_CONFIG = {
  cacheTimeout: 172800,          // 48 hours (seconds)
  maxHistoryVersions: 3,         // Number of historical versions to retain
  auditLogLevel: 'all',          // all/sensitive/none
  autoDetect: {
    enabled: true,
    sensitivityThreshold: 'medium',
    askBeforeSave: true
  },
  requireConfirm: {
    delete: true,
    deleteAll: true,
    export: true,
    backup: true,
    restore: true
  },
  generator: {
    defaultLength: 32,
    includeUppercase: true,
    includeNumbers: true,
    includeSymbols: true
  }
};

/**
 * Ensure directories exist
 */
function ensureDirectories() {
  [DEFAULT_DATA_DIR, DEFAULT_CACHE_DIR, DEFAULT_LOGS_DIR].forEach(dir => {
    if (!existsSync(dir)) {
      mkdirSync(dir, { recursive: true });
    }
  });
}

/**
 * Sanitize and validate user input
 * @param {string} input - User input
 * @param {number} maxLength - Maximum length
 * @returns {string} - Sanitized input
 */
export function sanitizeInput(input, maxLength = MAX_INPUT_LENGTH) {
  if (!input) {
    return '';
  }
  
  // Convert to string
  let result = String(input);
  
  // Limit length
  if (result.length > maxLength) {
    result = result.substring(0, maxLength);
  }
  
  // Remove control characters (except newline and tab)
  result = result.replace(/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/g, '');
  
  return result;
}

/**
 * Load configuration
 */
export function loadConfig() {
  if (existsSync(CONFIG_FILE)) {
    const saved = JSON.parse(readFileSync(CONFIG_FILE, 'utf8'));
    return { ...DEFAULT_CONFIG, ...saved };
  }
  return DEFAULT_CONFIG;
}

/**
 * Save configuration
 */
export function saveConfig(config) {
  ensureDirectories();
  writeFileSync(CONFIG_FILE, JSON.stringify(config, null, 2), 'utf8');
}

/**
 * Get vault status
 */
export function getVaultStatus() {
  const config = loadConfig();
  
  if (!existsSync(VAULT_FILE)) {
    return { exists: false, initialized: false };
  }
  
  if (!existsSync(CACHE_FILE)) {
    return { exists: true, locked: true, initialized: true };
  }
  
  // Check if cache is expired
  try {
    const cacheData = JSON.parse(readFileSync(CACHE_FILE, 'utf8'));
    const now = Date.now();
    const expiresAt = new Date(cacheData.expiresAt).getTime();
    
    if (now > expiresAt) {
      return { 
        exists: true, 
        locked: true, 
        initialized: true,
        expired: true,
        lastUsed: cacheData.lastUsedAt
      };
    }
    
    // Update last used time
    cacheData.lastUsedAt = new Date().toISOString();
    cacheData.expiresAt = new Date(now + config.cacheTimeout * 1000).toISOString();
    writeFileSync(CACHE_FILE, JSON.stringify(cacheData), 'utf8');
    
    return {
      exists: true,
      locked: false,
      initialized: true,
      unlocked: true,
      lastUsed: cacheData.lastUsedAt,
      expiresAt: cacheData.expiresAt
    };
  } catch (e) {
    return { exists: true, locked: true, initialized: true, error: true };
  }
}

/**
 * Initialize vault
 */
export function initializeVault(masterPassword) {
  ensureDirectories();
  
  const { key, salt } = crypto.deriveKey(masterPassword);
  
  const vaultData = {
    version: 1,
    createdAt: new Date().toISOString(),
    updatedAt: new Date().toISOString(),
    salt: salt.toString('hex'),
    entries: [],
    history: []
  };
  
  // Encrypt and save
  const packed = crypto.encryptAndPack(key, vaultData);
  writeFileSync(VAULT_FILE, packed);
  
  // Cache key (encrypted version using master password)
  cacheKey(key, masterPassword);
  
  // Clear key from memory
  crypto.secureWipe(key);
  
  return { success: true, message: 'Vault initialized' };
}

/**
 * Cache decryption key (encrypted version)
 * @param {Buffer} key - Derived encryption key
 * @param {string} masterPassword - Master password (used to derive cache encryption key)
 */
function cacheKey(key, masterPassword) {
  const config = loadConfig();
  
  // Derive cache encryption key from master password
  const { key: cacheEncryptionKey } = crypto.deriveCacheKey(masterPassword);
  
  // Encrypt key data
  const keyData = {
    key: key.toString('hex'),
    createdAt: new Date().toISOString(),
    lastUsedAt: new Date().toISOString(),
    expiresAt: new Date(Date.now() + config.cacheTimeout * 1000).toISOString()
  };
  
  const encrypted = crypto.encryptAndPack(cacheEncryptionKey, keyData);
  writeFileSync(CACHE_FILE, encrypted, 'utf8');
}

/**
 * Get decryption key
 */
export function getDecryptionKey(masterPassword = null) {
  if (!existsSync(CACHE_FILE)) {
    return { locked: true, reason: 'no_cache' };
  }
  
  // If no master password provided and cache exists, try to decrypt cache
  if (!masterPassword) {
    try {
      // Need master password to decrypt cache, so return locked when no password
      return { locked: true, reason: 'need_password_for_cache' };
    } catch (e) {
      return { locked: true, reason: 'read_cache_failed' };
    }
  }
  
  // If master password provided, verify and decrypt cache
  try {
    // Read salt from vault
    let vaultSalt = null;
    if (existsSync(VAULT_FILE)) {
      try {
        const vaultPacked = readFileSync(VAULT_FILE);
        // Try to derive key with provided password and decrypt to verify
        const { key: testKey, salt: derivedSalt } = crypto.deriveKey(masterPassword);
        const vaultData = crypto.unpackAndDecrypt(vaultPacked, testKey);
        vaultSalt = Buffer.from(vaultData.salt, 'hex');
        
        // Verify salt matches
        if (!vaultSalt.equals(derivedSalt)) {
          // Salt doesn't match, but password might be correct (could be old version)
          // Continue with derived key
        }
      } catch (e) {
        // Decryption failed, wrong password
        return { locked: true, reason: 'wrong_password' };
      }
    }
    
    // Decrypt cache with master password
    const { key: cacheEncryptionKey } = crypto.deriveCacheKey(masterPassword);
    const cachePacked = readFileSync(CACHE_FILE);
    const cacheData = crypto.unpackAndDecrypt(cachePacked, cacheEncryptionKey);
    
    const now = Date.now();
    const expiresAt = new Date(cacheData.expiresAt).getTime();
    
    if (now > expiresAt) {
      // Cache expired, clear cache file
      unlinkSync(CACHE_FILE);
      return { locked: true, reason: 'expired', lastUsed: cacheData.lastUsedAt };
    }
    
    // Update cache time and re-encrypt save
    cacheData.lastUsedAt = new Date().toISOString();
    cacheData.expiresAt = new Date(now + loadConfig().cacheTimeout * 1000).toISOString();
    
    const { key: newCacheEncryptionKey } = crypto.deriveCacheKey(masterPassword);
    const newCachePacked = crypto.encryptAndPack(newCacheEncryptionKey, cacheData);
    writeFileSync(CACHE_FILE, newCachePacked, 'utf8');
    
    // Get actual key from cache
    const key = Buffer.from(cacheData.key, 'hex');
    
    return {
      locked: false,
      key,
      expiresAt: cacheData.expiresAt
    };
  } catch (e) {
    return { locked: true, reason: 'wrong_password' };
  }
}

/**
 * Load and decrypt vault (with caching)
 * @param {Buffer} decryptionKey - Decryption key
 */
export function loadVault(decryptionKey) {
  const keyHex = decryptionKey.toString('hex');
  
  // Check cache
  if (vaultCache && vaultCacheKeyHex === keyHex) {
    return vaultCache;
  }
  
  const packed = readFileSync(VAULT_FILE);
  const vaultData = crypto.unpackAndDecrypt(packed, decryptionKey);
  
  // Update cache
  vaultCache = vaultData;
  vaultCacheKeyHex = keyHex;
  
  return vaultData;
}

/**
 * Clear vault cache (use when security requires)
 */
export function clearVaultCache() {
  vaultCache = null;
  vaultCacheKeyHex = null;
}

/**
 * Save vault (create version history)
 */
export function saveVault(vaultData, decryptionKey) {
  const config = loadConfig();
  
  // Save current version to history
  if (existsSync(VAULT_FILE)) {
    const timestamp = Date.now();
    const historyFile = join(DEFAULT_DATA_DIR, `vault.${timestamp}.enc`);
    
    // Copy old file to history
    const oldPacked = readFileSync(VAULT_FILE);
    writeFileSync(historyFile, oldPacked);
    
    vaultData.history = vaultData.history || [];
    vaultData.history.push({
      version: vaultData.history.length + 1,
      timestamp: new Date().toISOString(),
      file: historyFile
    });
    
    // Keep only the most recent N versions
    while (vaultData.history.length > config.maxHistoryVersions) {
      const old = vaultData.history.shift();
      if (existsSync(old.file)) {
        unlinkSync(old.file);
      }
    }
  }
  
  vaultData.updatedAt = new Date().toISOString();
  
  // Encrypt and save
  const packed = crypto.encryptAndPack(decryptionKey, vaultData);
  writeFileSync(VAULT_FILE, packed);
  
  // Update cache
  vaultCache = vaultData;
  vaultCacheKeyHex = decryptionKey.toString('hex');
}

/**
 * Lock vault (clear cache)
 */
export function lockVault() {
  if (existsSync(CACHE_FILE)) {
    unlinkSync(CACHE_FILE);
  }
  return { locked: true };
}

/**
 * Log detection event
 */
export function logDetection(type, action, result, extra = {}) {
  ensureDirectories();
  
  const logEntry = {
    timestamp: new Date().toISOString(),
    type,
    action,
    result,
    source: extra.source || 'password_manager',
    count: extra.count || 1
  };
  
  const line = JSON.stringify(logEntry) + '\n';
  
  // Use async append write (avoid blocking)
  try {
    appendFileSync(DETECTION_LOG, line, 'utf8');
  } catch (e) {
    // Log write failure doesn't affect main flow
    console.error('Warning: Detection log write failed', e.message);
  }
}

/**
 * Read detection log
 */
export function getDetectionLog(since = null) {
  if (!existsSync(DETECTION_LOG)) {
    return [];
  }
  
  const content = readFileSync(DETECTION_LOG, 'utf8');
  const lines = content.trim().split('\n').filter(l => l);
  
  return lines.map(line => JSON.parse(line)).filter(entry => {
    if (!since) return true;
    return new Date(entry.timestamp) >= new Date(since);
  });
}

/**
 * Backup vault
 */
export function backupVault(outputPath, decryptionKey) {
  const packed = readFileSync(VAULT_FILE);
  writeFileSync(outputPath, packed);
  return { success: true, path: outputPath };
}

/**
 * Restore vault
 */
export function restoreVault(inputPath) {
  if (!existsSync(inputPath)) {
    return { success: false, error: 'File does not exist' };
  }
  
  try {
    const packed = readFileSync(inputPath);
    
    // Validate format: try to unpack with empty key (verify format only, not content)
    // Will throw exception if format is wrong
    try {
      // Read first 48 bytes (salt 16 + iv 16 + authTag 16)
      if (packed.length < 48) {
        return { success: false, error: 'Invalid backup file format (file too small)' };
      }
      // Format validation passed, don't actually decrypt
    } catch (e) {
      return { success: false, error: 'Invalid backup file format' };
    }
    
    writeFileSync(VAULT_FILE, packed);
    
    // Clear cache, require re-entering master password
    lockVault();
    
    // Clear vault cache
    clearVaultCache();
    
    return { success: true };
  } catch (e) {
    return { success: false, error: `Restore failed: ${e.message}` };
  }
}

/**
 * List historical versions
 */
export function listHistory() {
  if (!existsSync(VAULT_FILE)) {
    return [];
  }
  
  const files = readdirSync(DEFAULT_DATA_DIR)
    .filter(f => f.startsWith('vault.') && f.endsWith('.enc'))
    .map(f => {
      const match = f.match(/vault\.(\d+)\.enc/);
      return match ? { file: f, timestamp: parseInt(match[1]) } : null;
    })
    .filter(Boolean)
    .sort((a, b) => b.timestamp - a.timestamp);
  
  return files.map((f, i) => ({
    version: i + 1,
    timestamp: new Date(f.timestamp).toISOString(),
    file: f.file
  }));
}

/**
 * Rollback to specified version
 */
export function rollbackToVersion(version, decryptionKey) {
  const history = listHistory();
  const target = history[version - 1];
  
  if (!target) {
    return { success: false, error: 'Version does not exist' };
  }
  
  const targetFile = join(DEFAULT_DATA_DIR, target.file);
  if (!existsSync(targetFile)) {
    return { success: false, error: 'Version file does not exist' };
  }
  
  const packed = readFileSync(targetFile);
  writeFileSync(VAULT_FILE, packed);
  
  return { success: true, version };
}

export default {
  loadConfig,
  saveConfig,
  getVaultStatus,
  initializeVault,
  getDecryptionKey,
  loadVault,
  saveVault,
  lockVault,
  logDetection,
  getDetectionLog,
  backupVault,
  restoreVault,
  listHistory,
  rollbackToVersion,
  paths: {
    VAULT_FILE,
    CACHE_FILE,
    CONFIG_FILE,
    DETECTION_LOG
  }
};
