#!/usr/bin/env node

import process from 'node:process';

import {
  API_KEY_ENV,
  DEFAULT_BASE_URL,
  DEFAULT_BETA_HEADER,
  buildHeaders,
  die,
  loadRequestJson,
  normaliseDomain,
  parseCli,
  postJson,
  printJson,
  printMarkdownSearchResults,
  toInt,
  toBool,
} from './_lib.mjs';

function printHelp() {
  // eslint-disable-next-line no-console
  console.log(`Parallel Search API CLI

Usage:
  node parallel-search.mjs --objective "..." --query "..." [options]
  node parallel-search.mjs --request request.json
  echo '{"objective":"...","search_queries":["..."]}' | node parallel-search.mjs

Required:
  --objective TEXT                 Natural language objective (what you want).

Common options:
  --query TEXT                     Search query (repeatable).
  --mode one-shot|agentic          Defaults to one-shot.
  --max-results N                  1–20.

Source policy:
  --include-domain example.com     Allow only these domains (repeatable, max 10).
  --exclude-domain example.com     Block these domains (repeatable, max 10).
  --after-date YYYY-MM-DD          Filter results published on/after this date.

Excerpts:
  --excerpts                        Force excerpts=true.
  --no-excerpts                     Set excerpts=false.
  --excerpt-max-chars N             Per result excerpt budget.
  --excerpt-max-total-chars N       Total excerpt budget across results.

Freshness:
  --fetch-max-age-seconds N         Request content no older than N seconds.

Advanced:
  --request path/to/request.json    Use a full JSON request body.
  --request-json '{...}'            Same but inline.
  --base-url https://api.parallel.ai
  --beta-header search-extract-2025-10-10

Output:
  --format json|md                  Default: json
  --compact                         Minified JSON output.
  --dry-run                         Print request JSON and exit.

Env:
  ${API_KEY_ENV}                    Parallel API key (required).
  PARALLEL_BASE_URL                 Override API base URL.
  PARALLEL_BETA_HEADER              Override beta header value.
`);
}

async function main() {
  const args = parseCli(process.argv, {
    multi: ['query', 'include-domain', 'exclude-domain'],
    booleans: ['help', 'excerpts', 'compact', 'dry-run'],
    aliases: { '-h': '--help' },
  });

  if (args.help) {
    printHelp();
    process.exit(0);
  }

  const apiKey = process.env[API_KEY_ENV];
  const baseUrl = String(args['base-url'] ?? DEFAULT_BASE_URL).replace(/\/+$/, '');
  const betaHeader = String(args['beta-header'] ?? DEFAULT_BETA_HEADER);

  const requestFromJson = await loadRequestJson({
    requestPath: args.request,
    requestJson: args['request-json'],
    allowStdin: true,
  });

  /** @type {any} */
  let body;

  if (requestFromJson) {
    body = requestFromJson;
  } else {
    if (!args.objective) throw new Error('--objective is required (or provide --request / stdin JSON)');

    body = {
      objective: String(args.objective),
    };

    if (args.query) {
      body.search_queries = Array.isArray(args.query) ? args.query.map(String) : [String(args.query)];
    }

    if (args.mode) {
      body.mode = String(args.mode);
    }

    if (args['max-results'] !== undefined) {
      body.max_results = toInt(args['max-results'], { name: '--max-results', min: 1, max: 20 });
    }

    const includeDomainsRaw = args['include-domain'] ?? [];
    const excludeDomainsRaw = args['exclude-domain'] ?? [];

    const includeDomains = (Array.isArray(includeDomainsRaw) ? includeDomainsRaw : [includeDomainsRaw])
      .map(normaliseDomain)
      .filter(Boolean);

    const excludeDomains = (Array.isArray(excludeDomainsRaw) ? excludeDomainsRaw : [excludeDomainsRaw])
      .map(normaliseDomain)
      .filter(Boolean);

    if (includeDomains.length || excludeDomains.length || args['after-date']) {
      body.source_policy = {};
      if (includeDomains.length) body.source_policy.include_domains = includeDomains;
      if (excludeDomains.length) body.source_policy.exclude_domains = excludeDomains;
      if (args['after-date']) body.source_policy.after_date = String(args['after-date']);
    }

    const excerptsExplicit = args.excerpts !== undefined
      ? toBool(args.excerpts, { name: '--excerpts' })
      : undefined;
    const excerptMaxChars = args['excerpt-max-chars'];
    const excerptMaxTotal = args['excerpt-max-total-chars'];

    const excerptSettingsProvided = excerptMaxChars !== undefined || excerptMaxTotal !== undefined;

    if (excerptsExplicit === false) {
      body.excerpts = false;
    } else if (excerptSettingsProvided) {
      body.excerpts = {
        ...(excerptMaxChars !== undefined
          ? { max_chars_per_result: toInt(excerptMaxChars, { name: '--excerpt-max-chars', min: 1 }) }
          : {}),
        ...(excerptMaxTotal !== undefined
          ? { max_chars_total: toInt(excerptMaxTotal, { name: '--excerpt-max-total-chars', min: 1 }) }
          : {}),
      };
    } else if (excerptsExplicit === true) {
      body.excerpts = true;
    }

    if (args['fetch-max-age-seconds'] !== undefined) {
      body.fetch_policy = {
        max_age_seconds: toInt(args['fetch-max-age-seconds'], { name: '--fetch-max-age-seconds', min: 0 }),
      };
    }
  }

  if (args['dry-run']) {
    printJson(body, { pretty: !args.compact });
    process.exit(0);
  }

  const headers = buildHeaders({ apiKey, betaHeader });
  const endpoint = `${baseUrl}/v1beta/search`;

  const res = await postJson(endpoint, {
    headers,
    body,
    timeoutMs: 120_000,
  });

  const format = String(args.format ?? 'json').toLowerCase();
  if (format === 'md' || format === 'markdown') {
    printMarkdownSearchResults(res);
  } else {
    printJson(res, { pretty: !args.compact });
  }
}

main().catch((err) => {
  const message = err?.message ? String(err.message) : String(err);

  // API error (non-2xx)
  if (typeof err?.status === 'number') {
    die(message, 3);
  }

  // Arg parsing / validation errors
  if (message.startsWith('Unknown') || message.includes('--') || message.includes('required')) {
    die(message, 2);
  }

  // Network/timeouts/etc
  die(message, 1);
});
