#!/usr/bin/env node

import process from 'node:process';

import {
  API_KEY_ENV,
  DEFAULT_BASE_URL,
  DEFAULT_BETA_HEADER,
  buildHeaders,
  die,
  loadRequestJson,
  normaliseDomain,
  parseCli,
  postJson,
  printJson,
  printMarkdownExtractResults,
  printMarkdownSearchResults,
  toBool,
  toInt,
} from './_lib.mjs';

function printHelp() {
  // eslint-disable-next-line no-console
  console.log(`Parallel Search → Extract pipeline

Usage:
  node parallel-search-extract.mjs --objective "..." --query "..." --top 3 [options]

Search options:
  --objective TEXT
  --query TEXT (repeatable)
  --mode one-shot|agentic
  --max-results N (1–20)
  --include-domain example.com (repeatable)
  --exclude-domain example.com (repeatable)
  --after-date YYYY-MM-DD
  --excerpt-max-chars N
  --excerpt-max-total-chars N
  --fetch-max-age-seconds N

Extract options:
  --top N (1–10)                   How many search results to extract.
  --extract-objective TEXT          Defaults to the search objective.
  --excerpts / --no-excerpts
  --full-content / --no-full-content
  --excerpts-max-chars N
  --excerpts-max-total-chars N
  --full-max-chars N
  --extract-fetch-max-age-seconds N
  --extract-fetch-timeout-seconds N
  --disable-cache-fallback

Advanced:
  --search-request request.json     Full Search request body (JSON).
  --extract-request request.json    Full Extract request body (JSON).
  --base-url https://api.parallel.ai
  --beta-header search-extract-2025-10-10

Output:
  --format json|md                  Default: json
  --compact                         Minified JSON output.
  --dry-run                         Print request JSON and exit.

Env:
  ${API_KEY_ENV}                    Parallel API key (required).
`);
}

async function main() {
  const args = parseCli(process.argv, {
    multi: ['query', 'include-domain', 'exclude-domain'],
    booleans: ['help', 'excerpts', 'full-content', 'disable-cache-fallback', 'compact', 'dry-run'],
    aliases: { '-h': '--help' },
  });

  if (args.help) {
    printHelp();
    process.exit(0);
  }

  const apiKey = process.env[API_KEY_ENV];
  const baseUrl = String(args['base-url'] ?? DEFAULT_BASE_URL).replace(/\/+$/, '');
  const betaHeader = String(args['beta-header'] ?? DEFAULT_BETA_HEADER);

  const searchRequestFromJson = await loadRequestJson({
    requestPath: args['search-request'],
    requestJson: args['search-request-json'],
    allowStdin: false,
  });

  const extractRequestFromJson = await loadRequestJson({
    requestPath: args['extract-request'],
    requestJson: args['extract-request-json'],
    allowStdin: false,
  });

  const topN = toInt(args.top ?? 3, { name: '--top', min: 1, max: 10 });

  /** @type {any} */
  let searchBody;

  if (searchRequestFromJson) {
    searchBody = searchRequestFromJson;
  } else {
    if (!args.objective) throw new Error('--objective is required (or provide --search-request)');

    searchBody = {
      objective: String(args.objective),
    };

    if (args.query) {
      searchBody.search_queries = Array.isArray(args.query) ? args.query.map(String) : [String(args.query)];
    }

    if (args.mode) searchBody.mode = String(args.mode);

    if (args['max-results'] !== undefined) {
      searchBody.max_results = toInt(args['max-results'], { name: '--max-results', min: 1, max: 20 });
    }

    const includeDomainsRaw = args['include-domain'] ?? [];
    const excludeDomainsRaw = args['exclude-domain'] ?? [];

    const includeDomains = (Array.isArray(includeDomainsRaw) ? includeDomainsRaw : [includeDomainsRaw])
      .map(normaliseDomain)
      .filter(Boolean);

    const excludeDomains = (Array.isArray(excludeDomainsRaw) ? excludeDomainsRaw : [excludeDomainsRaw])
      .map(normaliseDomain)
      .filter(Boolean);

    if (includeDomains.length || excludeDomains.length || args['after-date']) {
      searchBody.source_policy = {};
      if (includeDomains.length) searchBody.source_policy.include_domains = includeDomains;
      if (excludeDomains.length) searchBody.source_policy.exclude_domains = excludeDomains;
      if (args['after-date']) searchBody.source_policy.after_date = String(args['after-date']);
    }

    const excerptMaxChars = args['excerpt-max-chars'];
    const excerptMaxTotal = args['excerpt-max-total-chars'];
    const excerptsExplicit = args.excerpts !== undefined ? toBool(args.excerpts, { name: '--excerpts' }) : undefined;

    const excerptSettingsProvided = excerptMaxChars !== undefined || excerptMaxTotal !== undefined;

    if (excerptsExplicit === false) {
      searchBody.excerpts = false;
    } else if (excerptSettingsProvided) {
      searchBody.excerpts = {
        ...(excerptMaxChars !== undefined
          ? { max_chars_per_result: toInt(excerptMaxChars, { name: '--excerpt-max-chars', min: 1 }) }
          : {}),
        ...(excerptMaxTotal !== undefined
          ? { max_chars_total: toInt(excerptMaxTotal, { name: '--excerpt-max-total-chars', min: 1 }) }
          : {}),
      };
    } else if (excerptsExplicit === true) {
      searchBody.excerpts = true;
    }

    if (args['fetch-max-age-seconds'] !== undefined) {
      searchBody.fetch_policy = {
        max_age_seconds: toInt(args['fetch-max-age-seconds'], { name: '--fetch-max-age-seconds', min: 0 }),
      };
    }
  }

  const headers = buildHeaders({ apiKey, betaHeader });
  const searchEndpoint = `${baseUrl}/v1beta/search`;
  const extractEndpoint = `${baseUrl}/v1beta/extract`;

  if (args['dry-run']) {
    const combined = {
      search: { endpoint: searchEndpoint, body: searchBody },
      extract: { endpoint: extractEndpoint, body: extractRequestFromJson ?? '(built after search)' },
    };
    printJson(combined, { pretty: !args.compact });
    process.exit(0);
  }

  const searchRes = await postJson(searchEndpoint, { headers, body: searchBody, timeoutMs: 120_000 });

  const urls = Array.isArray(searchRes?.results)
    ? searchRes.results
        .map((r) => r?.url)
        .filter(Boolean)
        .slice(0, topN)
        .map(String)
    : [];

  if (!urls.length) {
    throw new Error('Search returned no usable URLs to extract');
  }

  /** @type {any} */
  let extractBody;

  if (extractRequestFromJson) {
    extractBody = extractRequestFromJson;
    // Ensure urls are set to the selected search results unless the user explicitly provided their own.
    if (!Array.isArray(extractBody.urls) || !extractBody.urls.length) extractBody.urls = urls;
  } else {
    extractBody = {
      urls,
      objective: String(args['extract-objective'] ?? searchBody?.objective ?? ''),
    };

    // Reuse search queries if present.
    if (Array.isArray(searchBody?.search_queries) && searchBody.search_queries.length) {
      extractBody.search_queries = searchBody.search_queries;
    }

    const excerptsExplicit = args.excerpts !== undefined ? toBool(args.excerpts, { name: '--excerpts' }) : undefined;
    const fullExplicit = args['full-content'] !== undefined
      ? toBool(args['full-content'], { name: '--full-content' })
      : undefined;

    const excerptsMaxChars = args['excerpts-max-chars'];
    const excerptsMaxTotal = args['excerpts-max-total-chars'];
    const fullMaxChars = args['full-max-chars'];

    const excerptsSettingsProvided = excerptsMaxChars !== undefined || excerptsMaxTotal !== undefined;

    if (excerptsExplicit === false) {
      extractBody.excerpts = false;
    } else if (excerptsSettingsProvided) {
      extractBody.excerpts = {
        ...(excerptsMaxChars !== undefined
          ? { max_chars_per_result: toInt(excerptsMaxChars, { name: '--excerpts-max-chars', min: 1 }) }
          : {}),
        ...(excerptsMaxTotal !== undefined
          ? { max_chars_total: toInt(excerptsMaxTotal, { name: '--excerpts-max-total-chars', min: 1 }) }
          : {}),
      };
    } else if (excerptsExplicit === true) {
      extractBody.excerpts = true;
    }

    if (fullExplicit === false) {
      extractBody.full_content = false;
    } else if (fullMaxChars !== undefined) {
      extractBody.full_content = {
        max_chars_per_result: toInt(fullMaxChars, { name: '--full-max-chars', min: 1 }),
      };
    } else if (fullExplicit === true) {
      extractBody.full_content = true;
    }

    const fetchMaxAge = args['extract-fetch-max-age-seconds'];
    const fetchTimeout = args['extract-fetch-timeout-seconds'];
    const disableCacheFallback = args['disable-cache-fallback'];

    if (fetchMaxAge !== undefined || fetchTimeout !== undefined || disableCacheFallback !== undefined) {
      extractBody.fetch_policy = {
        ...(fetchMaxAge !== undefined
          ? { max_age_seconds: toInt(fetchMaxAge, { name: '--extract-fetch-max-age-seconds', min: 600 }) }
          : {}),
        ...(fetchTimeout !== undefined
          ? { timeout_seconds: toInt(fetchTimeout, { name: '--extract-fetch-timeout-seconds', min: 1 }) }
          : {}),
        ...(disableCacheFallback !== undefined
          ? { disable_cache_fallback: toBool(disableCacheFallback, { name: '--disable-cache-fallback' }) }
          : {}),
      };
    }
  }

  const extractRes = await postJson(extractEndpoint, { headers, body: extractBody, timeoutMs: 180_000 });

  const combined = {
    picked_urls: urls,
    search: searchRes,
    extract: extractRes,
  };

  const format = String(args.format ?? 'json').toLowerCase();
  if (format === 'md' || format === 'markdown') {
    printMarkdownSearchResults(searchRes);
    // eslint-disable-next-line no-console
    console.log('\n---\n');
    printMarkdownExtractResults(extractRes);
  } else {
    printJson(combined, { pretty: !args.compact });
  }
}

main().catch((err) => {
  const message = err?.message ? String(err.message) : String(err);

  // API error (non-2xx)
  if (typeof err?.status === 'number') {
    die(message, 3);
  }

  // Arg parsing / validation errors
  if (message.startsWith('Unknown') || message.includes('--') || message.includes('required')) {
    die(message, 2);
  }

  // Network/timeouts/etc
  die(message, 1);
});
