#!/usr/bin/env node

import process from 'node:process';

import {
  API_KEY_ENV,
  DEFAULT_BASE_URL,
  DEFAULT_BETA_HEADER,
  buildHeaders,
  die,
  loadRequestJson,
  parseCli,
  postJson,
  printJson,
  printMarkdownExtractResults,
  toBool,
  toInt,
} from './_lib.mjs';

function printHelp() {
  // eslint-disable-next-line no-console
  console.log(`Parallel Extract API CLI

Usage:
  node parallel-extract.mjs --url "https://..." [--url "https://..."] [options]
  node parallel-extract.mjs --request request.json
  echo '{"urls":["https://..."],"excerpts":true}' | node parallel-extract.mjs

Required:
  --url URL                        URL to extract (repeatable, batches of 10).

Common options:
  --objective TEXT                 Optional objective to focus excerpts.
  --query TEXT                     Optional search query (repeatable).

Excerpts:
  --excerpts                        Force excerpts=true.
  --no-excerpts                     Set excerpts=false.
  --excerpts-max-chars N            Per URL excerpt budget.
  --excerpts-max-total-chars N      Total excerpt budget across URLs.

Full content:
  --full-content                    Force full_content=true.
  --no-full-content                 Set full_content=false.
  --full-max-chars N                Per URL full content max chars.

Fetch policy (live crawling):
  --fetch-max-age-seconds N         When set, min 600.
  --fetch-timeout-seconds N         Crawl timeout (API-side).
  --disable-cache-fallback          If true, fail rather than returning cached.

Advanced:
  --request path/to/request.json    Use a full JSON request body.
  --request-json '{...}'            Same but inline.
  --base-url https://api.parallel.ai
  --beta-header search-extract-2025-10-10

Output:
  --format json|md                  Default: json
  --compact                         Minified JSON output.
  --dry-run                         Print request JSON and exit.

Env:
  ${API_KEY_ENV}                    Parallel API key (required).
  PARALLEL_BASE_URL                 Override API base URL.
  PARALLEL_BETA_HEADER              Override beta header value.
`);
}

function chunk(array, size) {
  const out = [];
  for (let i = 0; i < array.length; i += size) out.push(array.slice(i, i + size));
  return out;
}

async function main() {
  const args = parseCli(process.argv, {
    multi: ['url', 'query'],
    booleans: ['help', 'excerpts', 'full-content', 'disable-cache-fallback', 'compact', 'dry-run'],
    aliases: { '-h': '--help' },
  });

  if (args.help) {
    printHelp();
    process.exit(0);
  }

  const apiKey = process.env[API_KEY_ENV];
  const baseUrl = String(args['base-url'] ?? DEFAULT_BASE_URL).replace(/\/+$/, '');
  const betaHeader = String(args['beta-header'] ?? DEFAULT_BETA_HEADER);

  const requestFromJson = await loadRequestJson({
    requestPath: args.request,
    requestJson: args['request-json'],
    allowStdin: true,
  });

  /** @type {any} */
  let body;

  if (requestFromJson) {
    body = requestFromJson;
  } else {
    const urls = args.url
      ? (Array.isArray(args.url) ? args.url : [args.url]).map(String).filter(Boolean)
      : [];

    if (!urls.length) throw new Error('--url is required (or provide --request / stdin JSON)');

    body = {
      urls,
    };

    if (args.objective) body.objective = String(args.objective);

    if (args.query) {
      body.search_queries = Array.isArray(args.query) ? args.query.map(String) : [String(args.query)];
    }

    // Excerpts: boolean or settings object.
    const excerptsExplicit = args.excerpts !== undefined
      ? toBool(args.excerpts, { name: '--excerpts' })
      : undefined;
    const excerptsMaxChars = args['excerpts-max-chars'];
    const excerptsMaxTotal = args['excerpts-max-total-chars'];
    const excerptsSettingsProvided = excerptsMaxChars !== undefined || excerptsMaxTotal !== undefined;

    if (excerptsExplicit === false) {
      body.excerpts = false;
    } else if (excerptsSettingsProvided) {
      body.excerpts = {
        ...(excerptsMaxChars !== undefined
          ? { max_chars_per_result: toInt(excerptsMaxChars, { name: '--excerpts-max-chars', min: 1 }) }
          : {}),
        ...(excerptsMaxTotal !== undefined
          ? { max_chars_total: toInt(excerptsMaxTotal, { name: '--excerpts-max-total-chars', min: 1 }) }
          : {}),
      };
    } else if (excerptsExplicit === true) {
      body.excerpts = true;
    }

    // Full content: boolean or settings object.
    const fullExplicit = args['full-content'] !== undefined
      ? toBool(args['full-content'], { name: '--full-content' })
      : undefined;
    const fullMaxChars = args['full-max-chars'];

    if (fullExplicit === false) {
      body.full_content = false;
    } else if (fullMaxChars !== undefined) {
      body.full_content = {
        max_chars_per_result: toInt(fullMaxChars, { name: '--full-max-chars', min: 1 }),
      };
    } else if (fullExplicit === true) {
      body.full_content = true;
    }

    // Fetch policy
    const fetchMaxAge = args['fetch-max-age-seconds'];
    const fetchTimeout = args['fetch-timeout-seconds'];
    const disableCacheFallback = args['disable-cache-fallback'];

    if (fetchMaxAge !== undefined || fetchTimeout !== undefined || disableCacheFallback !== undefined) {
      body.fetch_policy = {
        ...(fetchMaxAge !== undefined
          ? { max_age_seconds: toInt(fetchMaxAge, { name: '--fetch-max-age-seconds', min: 600 }) }
          : {}),
        ...(fetchTimeout !== undefined
          ? { timeout_seconds: toInt(fetchTimeout, { name: '--fetch-timeout-seconds', min: 1 }) }
          : {}),
        ...(disableCacheFallback !== undefined
          ? { disable_cache_fallback: toBool(disableCacheFallback, { name: '--disable-cache-fallback' }) }
          : {}),
      };
    }
  }

  if (args['dry-run']) {
    printJson(body, { pretty: !args.compact });
    process.exit(0);
  }

  const headers = buildHeaders({ apiKey, betaHeader });
  const endpoint = `${baseUrl}/v1beta/extract`;

  // Extract API only supports up to 10 URLs per request.
  const urls = Array.isArray(body?.urls) ? body.urls.map(String).filter(Boolean) : null;

  if (!urls) throw new Error('Request must include urls[]');
  if (urls.length <= 10) {
    const res = await postJson(endpoint, { headers, body, timeoutMs: 180_000 });

    const format = String(args.format ?? 'json').toLowerCase();
    if (format === 'md' || format === 'markdown') {
      printMarkdownExtractResults(res);
    } else {
      printJson(res, { pretty: !args.compact });
    }
    return;
  }

  // Batch mode: chunk into groups of 10 and merge.
  const batches = chunk(urls, 10);
  const combined = {
    extract_id: null,
    results: [],
    errors: [],
    warnings: null,
    usage: [],
    batches: batches.length,
  };

  for (const urlsBatch of batches) {
    const bodyBatch = { ...body, urls: urlsBatch };
    // eslint-disable-next-line no-await-in-loop
    const res = await postJson(endpoint, { headers, body: bodyBatch, timeoutMs: 180_000 });

    if (combined.extract_id === null && res?.extract_id) combined.extract_id = res.extract_id;
    if (Array.isArray(res?.results)) combined.results.push(...res.results);
    if (Array.isArray(res?.errors)) combined.errors.push(...res.errors);
    if (res?.warnings && !combined.warnings) combined.warnings = res.warnings;
    if (Array.isArray(res?.usage)) combined.usage.push(...res.usage);
  }

  const format = String(args.format ?? 'json').toLowerCase();
  if (format === 'md' || format === 'markdown') {
    printMarkdownExtractResults(combined);
  } else {
    printJson(combined, { pretty: !args.compact });
  }
}

main().catch((err) => {
  const message = err?.message ? String(err.message) : String(err);

  // API error (non-2xx)
  if (typeof err?.status === 'number') {
    die(message, 3);
  }

  // Arg parsing / validation errors
  if (message.startsWith('Unknown') || message.includes('--') || message.includes('required')) {
    die(message, 2);
  }

  // Network/timeouts/etc
  die(message, 1);
});
