// Shared helpers for the Parallel Search/Extract OpenClaw skill.
//
// Design goals:
// - Zero external dependencies
// - Stable CLI + JSON passthrough (future-proof against API changes)
// - Good error messages + deterministic exit codes

import fs from 'node:fs/promises';
import process from 'node:process';

export const DEFAULT_BASE_URL = process.env.PARALLEL_BASE_URL ?? 'https://api.parallel.ai';
export const DEFAULT_BETA_HEADER = process.env.PARALLEL_BETA_HEADER ?? 'search-extract-2025-10-10';
export const API_KEY_ENV = 'PARALLEL_API_KEY';

export function die(message, code = 1) {
  // eslint-disable-next-line no-console
  console.error(message);
  process.exit(code);
}

export function toInt(value, { name, min, max } = {}) {
  if (value === undefined || value === null || value === '') return undefined;
  const n = Number.parseInt(String(value), 10);
  if (!Number.isFinite(n)) throw new Error(`${name ?? 'value'} must be an integer`);
  if (min !== undefined && n < min) throw new Error(`${name ?? 'value'} must be >= ${min}`);
  if (max !== undefined && n > max) throw new Error(`${name ?? 'value'} must be <= ${max}`);
  return n;
}

export function toBool(value, { name } = {}) {
  if (typeof value === 'boolean') return value;
  if (value === undefined) return undefined;
  const s = String(value).trim().toLowerCase();
  if (['true', '1', 'yes', 'y', 'on'].includes(s)) return true;
  if (['false', '0', 'no', 'n', 'off'].includes(s)) return false;
  throw new Error(`${name ?? 'value'} must be a boolean`);
}

export function normaliseDomain(input) {
  // Parallel recommends apex domains without scheme or subdomain prefixes.
  // We accept slightly messy input and normalise.
  if (!input) return input;
  let s = String(input).trim();
  s = s.replace(/^https?:\/\//i, '');
  s = s.split('/')[0] ?? s;
  s = s.replace(/^www\./i, '');
  return s;
}

export function parseCli(argv, {
  multi = [],
  booleans = [],
  aliases = {},
} = {}) {
  const multiSet = new Set(multi);
  const boolSet = new Set(booleans);

  /** @type {Record<string, any>} */
  const out = { _: [] };

  for (let i = 2; i < argv.length; i += 1) {
    const token = argv[i];

    if (token === '--') {
      out._.push(...argv.slice(i + 1));
      break;
    }

    if (!token.startsWith('-')) {
      out._.push(token);
      continue;
    }

    // Short flags are allowed only via aliases (e.g. -q -> --query)
    if (token.startsWith('-') && !token.startsWith('--')) {
      const aliased = aliases[token];
      if (!aliased) throw new Error(`Unknown short flag: ${token}`);
      // Rewrite and re-process as long flag.
      argv.splice(i, 1, aliased);
      i -= 1;
      continue;
    }

    // Long flags
    const [rawKey, inlineValue] = token.split('=', 2);
    let key = rawKey.replace(/^--/, '');

    // Support --no-flag for booleans.
    if (key.startsWith('no-')) {
      key = key.slice(3);
      out[key] = false;
      continue;
    }

    const next = argv[i + 1];
    const hasNextValue = next !== undefined && !next.startsWith('-');

    let value;
    if (inlineValue !== undefined) {
      value = inlineValue;
    } else if (boolSet.has(key)) {
      // Plain --flag toggles true.
      value = true;
    } else if (hasNextValue) {
      value = next;
      i += 1;
    } else {
      // If not explicitly boolean but no value provided, treat as true.
      value = true;
    }

    if (multiSet.has(key)) {
      if (!Array.isArray(out[key])) out[key] = [];
      out[key].push(value);
    } else {
      out[key] = value;
    }
  }

  return out;
}

export async function readTextFile(filePath) {
  try {
    return await fs.readFile(filePath, 'utf8');
  } catch (err) {
    throw new Error(`Failed to read file: ${filePath} (${/** @type {any} */ (err)?.message ?? err})`);
  }
}

export async function readStdinText() {
  // Read stdin only if it isn't a TTY.
  if (process.stdin.isTTY) return '';

  const chunks = [];
  for await (const chunk of process.stdin) {
    chunks.push(Buffer.from(chunk));
  }
  return Buffer.concat(chunks).toString('utf8');
}

export function safeJsonParse(text, { name = 'JSON' } = {}) {
  try {
    return JSON.parse(text);
  } catch (err) {
    throw new Error(`Failed to parse ${name}: ${/** @type {any} */ (err)?.message ?? err}`);
  }
}

export async function loadRequestJson({ requestPath, requestJson, allowStdin = true } = {}) {
  if (requestPath) {
    const raw = await readTextFile(requestPath);
    return safeJsonParse(raw, { name: `JSON file ${requestPath}` });
  }
  if (requestJson) {
    return safeJsonParse(String(requestJson), { name: '--request-json' });
  }
  if (allowStdin) {
    const stdin = await readStdinText();
    if (stdin.trim()) return safeJsonParse(stdin, { name: 'stdin JSON' });
  }
  return null;
}

export function buildHeaders({ apiKey, betaHeader } = {}) {
  if (!apiKey) throw new Error(`${API_KEY_ENV} is required (set env or OpenClaw skills config)`);
  return {
    'Content-Type': 'application/json',
    'x-api-key': apiKey,
    'parallel-beta': betaHeader ?? DEFAULT_BETA_HEADER,
  };
}

export async function postJson(url, { headers, body, timeoutMs = 120_000 } = {}) {
  const controller = new AbortController();
  const timeout = setTimeout(() => controller.abort(), timeoutMs);

  try {
    const res = await fetch(url, {
      method: 'POST',
      headers,
      body: JSON.stringify(body),
      signal: controller.signal,
    });

    const text = await res.text();
    const contentType = res.headers.get('content-type') ?? '';

    let parsed;
    if (contentType.includes('application/json') && text.trim()) {
      try {
        parsed = JSON.parse(text);
      } catch {
        // Leave as text.
      }
    }

    if (!res.ok) {
      const message = parsed
        ? JSON.stringify(parsed, null, 2)
        : text || `HTTP ${res.status} ${res.statusText}`;
      const err = new Error(message);
      // @ts-ignore
      err.status = res.status;
      // @ts-ignore
      err.responseBody = parsed ?? text;
      throw err;
    }

    // Prefer parsed JSON but fall back to raw string.
    return parsed ?? (text.trim() ? text : null);
  } finally {
    clearTimeout(timeout);
  }
}

export function printJson(value, { pretty = true } = {}) {
  const text = pretty ? JSON.stringify(value, null, 2) : JSON.stringify(value);
  // eslint-disable-next-line no-console
  console.log(text);
}

export function printMarkdownSearchResults(searchResponse, { maxExcerptChars = 500 } = {}) {
  const results = searchResponse?.results ?? [];
  // eslint-disable-next-line no-console
  console.log(`# Search results\n`);
  for (const r of results) {
    const title = r?.title ?? '(no title)';
    const url = r?.url ?? '(no url)';
    const date = r?.publish_date ?? 'unknown';
    const excerpt = Array.isArray(r?.excerpts) && r.excerpts.length ? String(r.excerpts[0]) : '';
    const snippet = excerpt.length > maxExcerptChars ? `${excerpt.slice(0, maxExcerptChars)}…` : excerpt;

    // eslint-disable-next-line no-console
    console.log(`- **${title}** (${date})\n  - ${url}`);
    if (snippet.trim()) {
      // eslint-disable-next-line no-console
      console.log(`  - ${snippet.replace(/\s+/g, ' ').trim()}`);
    }
  }
}

export function printMarkdownExtractResults(extractResponse, { maxExcerptChars = 800 } = {}) {
  const results = extractResponse?.results ?? [];
  // eslint-disable-next-line no-console
  console.log(`# Extract results\n`);
  for (const r of results) {
    const title = r?.title ?? '(no title)';
    const url = r?.url ?? '(no url)';
    const date = r?.publish_date ?? 'unknown';
    // eslint-disable-next-line no-console
    console.log(`## ${title}\n- ${url}\n- publish_date: ${date}\n`);

    if (Array.isArray(r?.excerpts) && r.excerpts.length) {
      const excerpt = String(r.excerpts[0]);
      const snippet = excerpt.length > maxExcerptChars ? `${excerpt.slice(0, maxExcerptChars)}…` : excerpt;
      // eslint-disable-next-line no-console
      console.log(`**Excerpt (truncated):**\n\n${snippet}\n`);
    }

    if (typeof r?.full_content === 'string' && r.full_content.trim()) {
      // eslint-disable-next-line no-console
      console.log(`**Full content:**\n\n${r.full_content}\n`);
    }
  }
}
