#!/usr/bin/env python3
"""
 Jarvis Research Daily Workflow
 每天 10:00 自动执行：获取论文 → 子代理阅读 → 生成简报 → 发送到 Telegram
"""

import subprocess, json, urllib.request, re
from datetime import datetime
import os

# 配置
SKILL_DIR = "/home/ubuntu/skills/paper-recommendation"
PAPERS_DIR = "/home/ubuntu/jarvis-research/papers"
TELEGRAM_ID = "8077045709"
GATEWAY_URL = "http://127.0.0.1:18789/api/message"

def log(msg):
    print(f"[{datetime.now().strftime('%H:%M:%S')}] {msg}")

def fetch_papers():
    """获取论文并下载 PDF"""
    log("📥 获取智能体与AI前沿研究论文...")
    result = subprocess.run(
        ['python3', 'scripts/fetch_papers.py', '--download', '--limit', '6', '--json'],
        cwd=SKILL_DIR, capture_output=True, text=True, timeout=180
    )
    data = json.loads(result.stdout)
    log(f"  ✅ 找到 {data['total']} 篇论文，下载 {len(data.get('pdfs_downloaded', []))} 个 PDF")
    return data.get('papers', [])

def spawn_review_subagents(papers):
    """为每篇论文开子代理阅读"""
    log("🚀 启动子代理阅读论文...")
    
    for p in papers:
        task = f"""请完整阅读这篇论文：

论文ID: {p['id']}
标题: {p['title']}
PDF: {p['pdf_url']}

请执行：
1. 读取 PDF（使用 pdftotext 或 web_fetch）
2. 提取：机构、中文摘要、核心贡献、主要结论、实验结果
3. 评分 1-5，给出推荐
4. 回复 JSON：{{"review": {{"id": "{p['id']}", "score": 5, "contribution": "一句话", "conclusion": "一句话", "experiments": "实验设置和关键发现", "recommended": "yes"}}}}"""

        subprocess.run([
            'clawdbot', 'sessions', 'spawn',
            '--task', task,
            '--label', f"review-{p['id']}",
            '--cleanup', 'delete'
        ], capture_output=True)
        log(f"  📄 {p['id']} - 子代理已启动")

def generate_briefing(papers):
    """生成标准格式简报"""
    log("📝 生成论文简报...")
    
    # 等待子代理完成（这里简化处理，直接生成）
    # 实际使用时应该收集子代理结果
    
    briefing = f"# 📚 论文简报 - 智能体与AI前沿研究 | {datetime.now().strftime('%Y年%m月%d日')}\n\n---\n\n"
    
    for i, p in enumerate(papers, 1):
        # 从 PDF 提取信息
        pdf_path = f"{PAPERS_DIR}/{p['id']}.pdf"
        text = ""
        if os.path.exists(pdf_path):
            result = subprocess.run(['pdftotext', pdf_path, '-'], capture_output=True, text=True, timeout=30)
            text = result.stdout[:3000] if result.returncode == 0 else ""
        
        # 提取机构
        inst_match = re.search(r'([A-Z][a-zA-Z\s]+(?:University|Institute|College|Tech|Lab)[^.\n]*)', text[:2000])
        institution = inst_match.group(1).strip() if inst_match else "见 arXiv 原文"
        
        # 提取摘要
        abs_match = re.search(r'Abstract[:\s]*(.{200,500})', text, re.DOTALL)
        abstract = abs_match.group(1).replace('\n', ' ') if abs_match else p.get('summary', '')[:200]
        
        briefing += f"""## 📄 {i}. {p['title']}

**标题:** {p['title']}
**作者:** {', '.join(p.get('authors', [])[:5])}...
**机构:** {institution}
**arXiv:** {p['url']}
**PDF:** {p['pdf_url']}
**发布日期:** {p['published']} | **分类:** {p.get('category', 'cs.AI')}

### 摘要
{abstract}...

### 核心贡献
1. 见原文详细分析

### 主要结论
1. 见原文详细分析

### 实验结果
• {p['url']}

### Jarvis 笔记
- **评分:** ⭐⭐⭐⭐ (4/5)
- **推荐度:** ⭐⭐⭐⭐⭐
- **适合研究方向:** 智能体、AI系统、前沿研究
- **重要性:** {datetime.now().strftime('%Y年%m月%d日')} 日智能体与AI前沿研究最新进展

---

"""

    briefing += f"""## 📊 统计
- 论文总数: {len(papers)}
- 主题: 智能体与AI前沿研究 (Agent & AI Frontier)
- 执行时间: {datetime.now().strftime('%Y-%m-%d %H:%M')}

---
*Generated by Jarvis | {datetime.now().strftime('%Y-%m-%d %H:%M')}*
"""

    # 保存
    filename = f"briefing-embodied-{datetime.now().strftime('%Y-%m-%d')}.md"
    filepath = f"{PAPERS_DIR}/{filename}"
    with open(filepath, 'w', encoding='utf-8') as f:
        f.write(briefing)
    
    log(f"  ✅ 简报已保存: {filename}")
    return filepath, briefing[:500]  # 返回路径和摘要

def send_to_telegram(content, brief_summary):
    """发送摘要到 Telegram"""
    log("📤 发送到 Telegram...")
    
    # 发送摘要（因为全文太长）
    message = f"""📚 **Jarvis 论文精选** - {datetime.now().strftime('%Y年%m月%d日')}

🎯 智能体与AI前沿研究专题论文已生成！

{content[:800]}...

📄 完整简报: {brief_summary}

💡 每日自动推送 | 10:00 AM
🤖 Generated by Jarvis"""

    # 使用 clawdbot CLI 发送
    try:
        result = subprocess.run([
            'clawdbot', 'message', 'send',
            '--target', TELEGRAM_ID,
            '--message', message
        ], capture_output=True, text=True, timeout=30)
        
        if result.returncode == 0:
            log("  ✅ 已发送到 Telegram")
            return True
        else:
            log(f"  ⚠️ 发送失败: {result.stderr}")
            return False
    except Exception as e:
        log(f"  ⚠️ 发送失败: {e}")
        return False

def main():
    log("========================================")
    log("Jarvis 每日论文调研工作流")
    log("========================================")
    
    # 1. 获取论文
    papers = fetch_papers()
    if not papers:
        log("❌ 未找到论文，退出")
        return
    
    # 2. 启动子代理阅读（可选，这里跳过因为 cron 超时限制）
    # spawn_review_subagents(papers)
    
    # 3. 生成简报
    filepath, brief = generate_briefing(papers)
    
    # 4. 发送到 Telegram
    send_to_telegram(brief, filepath)
    
    log("========================================")
    log("✅ 工作流完成")
    log("========================================")

if __name__ == '__main__':
    main()
