#!/usr/bin/env bash
set -euo pipefail
DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
# shellcheck source=common.sh
source "$DIR/common.sh"

# ============================================================================
# Pancake API Script
# ============================================================================
# Unified script for interacting with Pancake platform APIs.
#
# Usage: bash scripts/pancake.sh <command> [args...]
#
# Commands are organized by domain:
#   - pages: Page management
#   - conversations: Conversation management
#   - messages: Message operations
#   - customers: Customer management
#   - statistics: Analytics and statistics
#   - tags: Tag management
#   - posts: Post management
#   - users: User/staff management
#   - call-logs: SIP call logs
#   - export: Data export
#   - upload: File upload
#   - chat-plugin: Chat plugin operations
# ============================================================================

cmd="${1:-}"
shift || true

case "$cmd" in

  # ==========================================================================
  # PAGES
  # ==========================================================================

  # List all pages for the authenticated user
  # Usage: pancake.sh pages-list
  pages-list)
    require_env USER_ACCESS_TOKEN
    pancake_request_user GET "/api/v1/pages"
    ;;

  # Generate page access token
  # Usage: pancake.sh pages-generate-token PAGE_ID
  pages-generate-token)
    require_env USER_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    pancake_request_user POST "/api/v1/pages/${page_id}/generate_page_access_token"
    ;;

  # ==========================================================================
  # CONVERSATIONS
  # ==========================================================================

  # List conversations for a page
  # Usage: pancake.sh conversations-list PAGE_ID [QUERY_STRING]
  # Example: pancake.sh conversations-list 123456 "?tags=1,2&type=INBOX"
  conversations-list)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    qs="${2:-}"
    if [[ -n "$qs" && "$qs" != "?"* ]]; then
      qs="?${qs}"
    fi
    pancake_request_page GET "/api/public_api/v2/pages/${page_id}/conversations${qs}"
    ;;

  # Add or remove a tag from a conversation
  # Usage: echo '{"action":"add","tag_id":"123"}' | pancake.sh conversations-tag PAGE_ID CONVERSATION_ID
  conversations-tag)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    conversation_id="${2:?CONVERSATION_ID required}"
    body="$(cat)"
    pancake_request_page POST "/api/public_api/v1/pages/${page_id}/conversations/${conversation_id}/tags" "$body"
    ;;

  # Assign staff to a conversation
  # Usage: echo '{"assignee_ids":["user1","user2"]}' | pancake.sh conversations-assign PAGE_ID CONVERSATION_ID
  conversations-assign)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    conversation_id="${2:?CONVERSATION_ID required}"
    body="$(cat)"
    pancake_request_page POST "/api/public_api/v1/pages/${page_id}/conversations/${conversation_id}/assign" "$body"
    ;;

  # Mark conversation as read
  # Usage: pancake.sh conversations-read PAGE_ID CONVERSATION_ID
  conversations-read)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    conversation_id="${2:?CONVERSATION_ID required}"
    pancake_request_page POST "/api/public_api/v1/pages/${page_id}/conversations/${conversation_id}/read"
    ;;

  # Mark conversation as unread
  # Usage: pancake.sh conversations-unread PAGE_ID CONVERSATION_ID
  conversations-unread)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    conversation_id="${2:?CONVERSATION_ID required}"
    pancake_request_page POST "/api/public_api/v1/pages/${page_id}/conversations/${conversation_id}/unread"
    ;;

  # ==========================================================================
  # MESSAGES
  # ==========================================================================

  # Get messages from a conversation
  # Usage: pancake.sh messages-list PAGE_ID CONVERSATION_ID [CURRENT_COUNT]
  messages-list)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    conversation_id="${2:?CONVERSATION_ID required}"
    current_count="${3:-}"
    qs=""
    if [[ -n "$current_count" ]]; then
      qs="&current_count=${current_count}"
    fi
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/conversations/${conversation_id}/messages?${qs}"
    ;;

  # Send a message (inbox, comment reply, or private reply)
  # Usage: echo '{"action":"reply_inbox","message":"Hello!"}' | pancake.sh messages-send PAGE_ID CONVERSATION_ID
  messages-send)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    conversation_id="${2:?CONVERSATION_ID required}"
    body="$(cat)"
    pancake_request_page POST "/api/public_api/v1/pages/${page_id}/conversations/${conversation_id}/messages" "$body"
    ;;

  # ==========================================================================
  # CUSTOMERS
  # ==========================================================================

  # List page customers
  # Usage: pancake.sh customers-list PAGE_ID SINCE UNTIL PAGE_NUMBER [PAGE_SIZE] [ORDER_BY]
  customers-list)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    since="${2:?SINCE timestamp required}"
    until="${3:?UNTIL timestamp required}"
    page_number="${4:?PAGE_NUMBER required}"
    page_size="${5:-20}"
    order_by="${6:-inserted_at}"
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/page_customers?since=${since}&until=${until}&page_number=${page_number}&page_size=${page_size}&order_by=${order_by}"
    ;;

  # Update customer information
  # Usage: echo '{"changes":{"name":"New Name","gender":"male"}}' | pancake.sh customers-update PAGE_ID CUSTOMER_ID
  customers-update)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    customer_id="${2:?CUSTOMER_ID required}"
    body="$(cat)"
    pancake_request_page PUT "/api/public_api/v1/pages/${page_id}/page_customers/${customer_id}" "$body"
    ;;

  # Add a note to a customer
  # Usage: echo '{"message":"Customer prefers morning calls"}' | pancake.sh customers-add-note PAGE_ID CUSTOMER_ID
  customers-add-note)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    customer_id="${2:?CUSTOMER_ID required}"
    body="$(cat)"
    pancake_request_page POST "/api/public_api/v1/pages/${page_id}/page_customers/${customer_id}/notes" "$body"
    ;;

  # Update a customer note
  # Usage: echo '{"note_id":"abc123","message":"Updated note"}' | pancake.sh customers-update-note PAGE_ID CUSTOMER_ID
  customers-update-note)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    customer_id="${2:?CUSTOMER_ID required}"
    body="$(cat)"
    pancake_request_page PUT "/api/public_api/v1/pages/${page_id}/page_customers/${customer_id}/notes" "$body"
    ;;

  # Delete a customer note
  # Usage: echo '{"note_id":"abc123"}' | pancake.sh customers-delete-note PAGE_ID CUSTOMER_ID
  customers-delete-note)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    customer_id="${2:?CUSTOMER_ID required}"
    body="$(cat)"
    pancake_request_page DELETE "/api/public_api/v1/pages/${page_id}/page_customers/${customer_id}/notes" "$body"
    ;;

  # ==========================================================================
  # STATISTICS
  # ==========================================================================

  # Get ads campaign statistics
  # Usage: pancake.sh stats-campaigns PAGE_ID SINCE UNTIL
  stats-campaigns)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    since="${2:?SINCE timestamp required}"
    until="${3:?UNTIL timestamp required}"
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/statistics/pages_campaigns?since=${since}&until=${until}"
    ;;

  # Get ads statistics
  # Usage: pancake.sh stats-ads PAGE_ID SINCE UNTIL TYPE
  # TYPE: by_id or by_time
  stats-ads)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    since="${2:?SINCE timestamp required}"
    until="${3:?UNTIL timestamp required}"
    type="${4:?TYPE required (by_id or by_time)}"
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/statistics/ads?since=${since}&until=${until}&type=${type}"
    ;;

  # Get customer engagement statistics
  # Usage: pancake.sh stats-engagement PAGE_ID DATE_RANGE [BY_HOUR] [USER_IDS]
  # DATE_RANGE format: "27/07/2021 00:00:00 - 26/08/2021 23:59:59"
  stats-engagement)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    date_range="${2:?DATE_RANGE required}"
    by_hour="${3:-false}"
    user_ids="${4:-}"
    encoded_date_range=$(url_encode "$date_range")
    qs="date_range=${encoded_date_range}&by_hour=${by_hour}"
    if [[ -n "$user_ids" ]]; then
      qs="${qs}&user_ids=${user_ids}"
    fi
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/statistics/customer_engagements?${qs}"
    ;;

  # Get page statistics
  # Usage: pancake.sh stats-page PAGE_ID SINCE UNTIL
  stats-page)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    since="${2:?SINCE timestamp required}"
    until="${3:?UNTIL timestamp required}"
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/statistics/pages?since=${since}&until=${until}"
    ;;

  # Get tag statistics
  # Usage: pancake.sh stats-tags PAGE_ID SINCE UNTIL
  stats-tags)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    since="${2:?SINCE timestamp required}"
    until="${3:?UNTIL timestamp required}"
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/statistics/tags?since=${since}&until=${until}"
    ;;

  # Get user/staff statistics
  # Usage: pancake.sh stats-users PAGE_ID DATE_RANGE
  stats-users)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    date_range="${2:?DATE_RANGE required}"
    encoded_date_range=$(url_encode "$date_range")
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/statistics/users?date_range=${encoded_date_range}"
    ;;

  # Get new customer statistics
  # Usage: pancake.sh stats-new-customers PAGE_ID DATE_RANGE [GROUP_BY]
  # GROUP_BY: day, hour, page_id
  stats-new-customers)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    date_range="${2:?DATE_RANGE required}"
    group_by="${3:-day}"
    encoded_date_range=$(url_encode "$date_range")
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/page_customers?date_range=${encoded_date_range}&group_by=${group_by}"
    ;;

  # ==========================================================================
  # TAGS
  # ==========================================================================

  # List all tags for a page
  # Usage: pancake.sh tags-list PAGE_ID
  tags-list)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/tags"
    ;;

  # ==========================================================================
  # POSTS
  # ==========================================================================

  # List posts for a page
  # Usage: pancake.sh posts-list PAGE_ID SINCE UNTIL PAGE_NUMBER PAGE_SIZE [TYPE]
  # TYPE: video, photo, text, livestream
  posts-list)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    since="${2:?SINCE timestamp required}"
    until="${3:?UNTIL timestamp required}"
    page_number="${4:?PAGE_NUMBER required}"
    page_size="${5:?PAGE_SIZE required}"
    type="${6:-}"
    qs="since=${since}&until=${until}&page_number=${page_number}&page_size=${page_size}"
    if [[ -n "$type" ]]; then
      qs="${qs}&type=${type}"
    fi
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/posts?${qs}"
    ;;

  # ==========================================================================
  # USERS (Staff)
  # ==========================================================================

  # List users/staff for a page
  # Usage: pancake.sh users-list PAGE_ID
  users-list)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/users"
    ;;

  # Update round robin users
  # Usage: echo '{"inbox":["user1"],"comment":["user2"]}' | pancake.sh users-round-robin PAGE_ID
  users-round-robin)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    body="$(cat)"
    pancake_request_page POST "/api/public_api/v1/pages/${page_id}/round_robin_users" "$body"
    ;;

  # ==========================================================================
  # CALL LOGS
  # ==========================================================================

  # Get SIP call logs
  # Usage: pancake.sh call-logs PAGE_ID SIP_ID PAGE_NUMBER PAGE_SIZE [SINCE] [UNTIL]
  call-logs)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    sip_id="${2:?SIP_ID required}"
    page_number="${3:?PAGE_NUMBER required}"
    page_size="${4:?PAGE_SIZE required}"
    since="${5:-}"
    until="${6:-}"
    qs="id=${sip_id}&page_number=${page_number}&page_size=${page_size}"
    if [[ -n "$since" ]]; then
      qs="${qs}&since=${since}"
    fi
    if [[ -n "$until" ]]; then
      qs="${qs}&until=${until}"
    fi
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/sip_call_logs?${qs}"
    ;;

  # ==========================================================================
  # EXPORT DATA
  # ==========================================================================

  # Export conversations from ads
  # Usage: pancake.sh export-ads-conversations PAGE_ID SINCE UNTIL [OFFSET]
  export-ads-conversations)
    require_env PAGE_ACCESS_TOKEN
    page_id="${1:?PAGE_ID required}"
    since="${2:?SINCE timestamp required}"
    until="${3:?UNTIL timestamp required}"
    offset="${4:-0}"
    pancake_request_page GET "/api/public_api/v1/pages/${page_id}/export_data?action=conversations_from_ads&since=${since}&until=${until}&offset=${offset}"
    ;;

  # ==========================================================================
  # UPLOAD CONTENT
  # ==========================================================================

  # Upload media content to a page
  # Usage: pancake.sh upload PAGE_ID FILE_PATH
  upload)
    confirm_write
    page_id="${1:?PAGE_ID required}"
    file_path="${2:?FILE_PATH required}"
    pancake_upload "$page_id" "$file_path"
    ;;

  # ==========================================================================
  # CHAT PLUGIN
  # ==========================================================================

  # Send message via chat plugin
  # Usage: echo '{"message":"Hello!","from_id":"web_xxx",...}' | pancake.sh chat-plugin-send PAGE_ID
  chat-plugin-send)
    page_id="${1:?PAGE_ID required}"
    body="$(cat)"
    curl -s -X POST -H "Content-Type: application/json" -d "$body" "${PANCAKE_BASE_URL}/api/v1/pke_chat_plugin/messages?page_id=${page_id}"
    ;;

  # Get chat plugin messages
  # Usage: pancake.sh chat-plugin-messages PAGE_ID CONVERSATION_ID [OFFSET]
  chat-plugin-messages)
    page_id="${1:?PAGE_ID required}"
    conversation_id="${2:?CONVERSATION_ID required}"
    offset="${3:-0}"
    curl -s -X GET "${PANCAKE_BASE_URL}/api/v1/pke_chat_plugin/messages?page_id=${page_id}&conversation_id=${conversation_id}&offset=${offset}"
    ;;

  # ==========================================================================
  # HELP
  # ==========================================================================

  help|--help|-h|"")
    cat <<'HELP'
Pancake API Script

Usage: bash scripts/pancake.sh <command> [args...]

PAGES:
  pages-list                                    List all pages
  pages-generate-token PAGE_ID                  Generate page access token

CONVERSATIONS:
  conversations-list PAGE_ID [QUERY]            List conversations
  conversations-tag PAGE_ID CONV_ID             Add/remove tag (stdin: JSON)
  conversations-assign PAGE_ID CONV_ID          Assign staff (stdin: JSON)
  conversations-read PAGE_ID CONV_ID            Mark as read
  conversations-unread PAGE_ID CONV_ID          Mark as unread

MESSAGES:
  messages-list PAGE_ID CONV_ID [COUNT]         Get messages
  messages-send PAGE_ID CONV_ID                 Send message (stdin: JSON)

CUSTOMERS:
  customers-list PAGE_ID SINCE UNTIL PAGE [SIZE] [ORDER]
  customers-update PAGE_ID CUST_ID              Update customer (stdin: JSON)
  customers-add-note PAGE_ID CUST_ID            Add note (stdin: JSON)
  customers-update-note PAGE_ID CUST_ID         Update note (stdin: JSON)
  customers-delete-note PAGE_ID CUST_ID         Delete note (stdin: JSON)

STATISTICS:
  stats-campaigns PAGE_ID SINCE UNTIL           Ads campaign stats
  stats-ads PAGE_ID SINCE UNTIL TYPE            Ads stats (by_id/by_time)
  stats-engagement PAGE_ID DATE_RANGE [BY_HOUR] Engagement stats
  stats-page PAGE_ID SINCE UNTIL                Page stats
  stats-tags PAGE_ID SINCE UNTIL                Tag stats
  stats-users PAGE_ID DATE_RANGE                User stats
  stats-new-customers PAGE_ID DATE_RANGE [GROUP]

TAGS:
  tags-list PAGE_ID                             List all tags

POSTS:
  posts-list PAGE_ID SINCE UNTIL PAGE SIZE [TYPE]

USERS:
  users-list PAGE_ID                            List staff
  users-round-robin PAGE_ID                     Update round robin (stdin: JSON)

OTHER:
  call-logs PAGE_ID SIP_ID PAGE SIZE [SINCE] [UNTIL]
  export-ads-conversations PAGE_ID SINCE UNTIL [OFFSET]
  upload PAGE_ID FILE_PATH                      Upload media
  chat-plugin-send PAGE_ID                      Send chat plugin msg (stdin: JSON)
  chat-plugin-messages PAGE_ID CONV_ID [OFFSET]

Environment Variables:
  PAGE_ACCESS_TOKEN   Required for page-scoped endpoints
  USER_ACCESS_TOKEN   Required for user-scoped endpoints
  CONFIRM_WRITE=YES   Required for write operations
  PANCAKE_BASE_URL    Override base URL (default: https://pages.fm)
HELP
    ;;

  *)
    echo "Unknown command: $cmd" >&2
    echo "Run 'bash scripts/pancake.sh help' for usage information." >&2
    exit 1
    ;;
esac
