#!/usr/bin/env bash
set -euo pipefail

# ============================================================================
# Pancake API - Common Utilities
# ============================================================================
# Shared functions for Pancake API scripts.
# This file MUST be sourced by domain-specific scripts.
#
# Required environment variables:
#   PAGE_ACCESS_TOKEN: Page access token (for page-scoped endpoints)
#   or
#   USER_ACCESS_TOKEN: User access token (for user-scoped endpoints)
#
# Optional environment variables:
#   PANCAKE_BASE_URL: Override default base URL (default: https://pages.fm)
#   CONFIRM_WRITE: Set to "YES" to allow write operations
# ============================================================================

# Default base URL
PANCAKE_BASE_URL="${PANCAKE_BASE_URL:-https://pages.fm}"

# ============================================================================
# require_env - Validate required environment variable
# ============================================================================
# Usage: require_env VAR_NAME
# Exit code: 2 if variable is not set
# ============================================================================
require_env() {
  local name="$1"
  if [[ -z "${!name:-}" ]]; then
    echo "Missing env var: $name" >&2
    exit 2
  fi
}

# ============================================================================
# confirm_write - Safety guardrail for write operations
# ============================================================================
# Call this at the start of any POST/PUT/PATCH/DELETE operation
# Exit code: 3 if CONFIRM_WRITE is not set to "YES"
# ============================================================================
confirm_write() {
  if [[ "${CONFIRM_WRITE:-}" != "YES" ]]; then
    echo "Write operation blocked. Set CONFIRM_WRITE=YES to proceed." >&2
    exit 3
  fi
}

# ============================================================================
# url_encode - URL encode a string using Python
# ============================================================================
# Usage: url_encode "string to encode"
# ============================================================================
url_encode() {
  python3 -c "import urllib.parse; print(urllib.parse.quote('$1', safe=''))"
}

# ============================================================================
# pancake_request - Make authenticated HTTP request to Pancake API
# ============================================================================
# Usage: pancake_request METHOD PATH [BODY]
#   METHOD: HTTP method (GET, POST, PUT, PATCH, DELETE)
#   PATH: API path (e.g., /api/public_api/v1/pages/{page_id}/conversations)
#   BODY: Optional JSON body for write operations
#
# Automatically appends page_access_token or access_token based on endpoint
# ============================================================================
pancake_request() {
  local method="$1"
  local path="$2"
  local body="${3:-}"

  local url="${PANCAKE_BASE_URL}${path}"

  # Determine which token to use based on the path
  local token_param=""
  if [[ "$path" == *"/api/v1/pages"* ]] && [[ "$path" != *"/api/public_api/"* ]]; then
    # User API endpoints use access_token
    require_env USER_ACCESS_TOKEN
    local encoded_token
    encoded_token=$(url_encode "$USER_ACCESS_TOKEN")
    token_param="access_token=${encoded_token}"
  else
    # Page API endpoints use page_access_token
    require_env PAGE_ACCESS_TOKEN
    local encoded_token
    encoded_token=$(url_encode "$PAGE_ACCESS_TOKEN")
    token_param="page_access_token=${encoded_token}"
  fi

  # Append token to URL
  if [[ "$url" == *"?"* ]]; then
    url="${url}&${token_param}"
  else
    url="${url}?${token_param}"
  fi

  # Build curl command
  local curl_args=(-s -X "$method")

  if [[ -n "$body" ]]; then
    curl_args+=(-H "Content-Type: application/json" -d "$body")
  fi

  curl_args+=("$url")

  curl "${curl_args[@]}"
}

# ============================================================================
# pancake_request_user - Make authenticated HTTP request using User API
# ============================================================================
# Usage: pancake_request_user METHOD PATH [BODY]
# Always uses USER_ACCESS_TOKEN
# ============================================================================
pancake_request_user() {
  local method="$1"
  local path="$2"
  local body="${3:-}"

  require_env USER_ACCESS_TOKEN

  local url="${PANCAKE_BASE_URL}${path}"
  local encoded_token
  encoded_token=$(url_encode "$USER_ACCESS_TOKEN")

  # Append token to URL
  if [[ "$url" == *"?"* ]]; then
    url="${url}&access_token=${encoded_token}"
  else
    url="${url}?access_token=${encoded_token}"
  fi

  # Build curl command
  local curl_args=(-s -X "$method")

  if [[ -n "$body" ]]; then
    curl_args+=(-H "Content-Type: application/json" -d "$body")
  fi

  curl_args+=("$url")

  curl "${curl_args[@]}"
}

# ============================================================================
# pancake_request_page - Make authenticated HTTP request using Page API
# ============================================================================
# Usage: pancake_request_page METHOD PATH [BODY]
# Always uses PAGE_ACCESS_TOKEN
# ============================================================================
pancake_request_page() {
  local method="$1"
  local path="$2"
  local body="${3:-}"

  require_env PAGE_ACCESS_TOKEN

  local url="${PANCAKE_BASE_URL}${path}"
  local encoded_token
  encoded_token=$(url_encode "$PAGE_ACCESS_TOKEN")

  # Append token to URL
  if [[ "$url" == *"?"* ]]; then
    url="${url}&page_access_token=${encoded_token}"
  else
    url="${url}?page_access_token=${encoded_token}"
  fi

  # Build curl command
  local curl_args=(-s -X "$method")

  if [[ -n "$body" ]]; then
    curl_args+=(-H "Content-Type: application/json" -d "$body")
  fi

  curl_args+=("$url")

  curl "${curl_args[@]}"
}

# ============================================================================
# pancake_upload - Upload file to Pancake
# ============================================================================
# Usage: pancake_upload PAGE_ID FILE_PATH
# ============================================================================
pancake_upload() {
  local page_id="$1"
  local file_path="$2"

  require_env PAGE_ACCESS_TOKEN

  local encoded_token
  encoded_token=$(url_encode "$PAGE_ACCESS_TOKEN")
  local url="${PANCAKE_BASE_URL}/api/public_api/v1/pages/${page_id}/upload_contents?page_access_token=${encoded_token}"

  curl -s -X POST -F "file=@${file_path}" "$url"
}
