# YubiKey OTP Support Design

**Date:** 2025-01-31
**Status:** Approved

## Overview

Add YubiKey OTP validation to `verify.sh` alongside existing TOTP support. The script auto-detects which method based on input format and validates accordingly.

### User Experience

```bash
# TOTP (existing) - 6 digits
./verify.sh ryan 123456

# YubiKey OTP (new) - 44 char ModHex string
./verify.sh ryan vvibkuhdhgubebgflfrrrthetggecujncdenbhkrbien
```

Same command, same state file, same audit logging. Just a different code format.

## Configuration

### New Environment Variables / Config Keys

- `YUBIKEY_CLIENT_ID` — from Yubico API registration
- `YUBIKEY_SECRET_KEY` — from Yubico API registration

### Config File Structure

```yaml
# ~/.openclaw/config.yaml
security:
  otp:
    secret: "BASE32_SECRET"        # existing TOTP
  yubikey:
    clientId: "12345"              # new
    secretKey: "base64string="     # new
```

Environment variables take precedence over config file (matches existing TOTP pattern).

## Onboarding

### YubiKey Setup Flow

1. Go to https://upgrade.yubico.com/getapikey/
2. Enter email address
3. Touch YubiKey to generate OTP in form field
4. Submit — receive Client ID and Secret Key
5. Configure credentials in config file or environment

### Troubleshooting: "Invalid OTP" During API Key Registration

If Yubico's site rejects your OTP:

1. **Key not registered with Yubico** — Key was reprogrammed or from private deployment
2. **Fix:** Use YubiKey Manager → Applications → OTP → Configure Slot 1 → Yubico OTP → Check "Upload to Yubico"

This re-registers the key with Yubico's cloud servers.

## Implementation Details

### Code Type Detection

```bash
detect_code_type() {
  local code="$1"
  if [[ "$code" =~ ^[cbdefghijklnrtuv]{44}$ ]]; then
    echo "yubikey"
  elif [[ "$code" =~ ^[0-9]{6}$ ]]; then
    echo "totp"
  else
    echo "unknown"
  fi
}
```

ModHex alphabet: `cbdefghijklnrtuv` (16 chars designed to work on all keyboard layouts)

### YubiKey Validation Flow

1. Extract public ID (first 12 chars) for audit logging
2. Generate random nonce
3. Call Yubico API: `https://api.yubico.com/wsapi/2.0/verify`
4. Verify response signature using `YUBIKEY_SECRET_KEY`
5. Check response status (`OK` = valid)

### API Call

```bash
curl "https://api.yubico.com/wsapi/2.0/verify?id=$CLIENT_ID&otp=$OTP&nonce=$NONCE"
# Response: status=OK (or REPLAYED_OTP, BAD_OTP, etc.)
```

### What Stays The Same

- State file format (verifications, usedCodes, failureCounts)
- Audit logging (logs "YUBIKEY" vs "TOTP" as method)
- Rate limiting
- Replay protection (Yubico API handles this, also tracked locally)
- Exit codes (0=valid, 1=invalid, 2=config error)

### Dependencies

- `curl` — already installed on macOS/Linux (no new installs)

## Files to Modify

| File | Change |
|------|--------|
| `verify.sh` | Add YubiKey detection and validation |
| `SKILL.md` | Add YubiKey documentation |
| `README.md` | Mention YubiKey support |
| `tests/` | Add YubiKey test cases |

## Files Unchanged

- `totp.mjs` — stays TOTP-only (Node CLI)
- `generate-secret.sh` — stays TOTP-only
- `check-status.sh` — works for both (just checks timestamps)

## Test Cases

1. Valid YubiKey OTP → success
2. Invalid YubiKey OTP → failure
3. Replayed YubiKey OTP → failure (Yubico returns `REPLAYED_OTP`)
4. Auto-detection: 6 digits routes to TOTP, 44 chars routes to YubiKey
5. Missing `YUBIKEY_CLIENT_ID` when YubiKey OTP provided → config error
6. Network failure calling Yubico API → appropriate error message

## Implementation Phases

### Phase 1: Core YubiKey Validation
- Add `detect_code_type()` function to verify.sh
- Add `validate_yubikey()` function with Yubico API call
- Add config loading for YUBIKEY_CLIENT_ID and YUBIKEY_SECRET_KEY
- Wire up auto-detection in main flow

### Phase 2: Testing
- Add YubiKey test cases to test suite
- Test with real YubiKey

### Phase 3: Documentation
- Update SKILL.md with YubiKey setup instructions
- Update README.md to mention YubiKey support
- Add troubleshooting for common issues

## Future Considerations

- Optional allowlist of permitted YubiKey public IDs
- Support for other key types (FIDO2/WebAuthn would require different approach)
- Self-hosted validation server option
