# YubiKey Support Implementation Plan

> **For Claude:** REQUIRED SUB-SKILL: Use ed3d-plan-and-execute:subagent-driven-development to implement this plan task-by-task.

**Goal:** Add YubiKey OTP validation to verify.sh alongside existing TOTP support

**Architecture:** Auto-detect code type by format (6 digits = TOTP, 44 ModHex chars = YubiKey), route to appropriate validator. YubiKey validation calls Yubico Cloud API with HMAC-SHA1 signed requests.

**Tech Stack:** Bash, curl, openssl (for HMAC-SHA1), jq

**Scope:** 3 phases from original design (phases 1-3)

**Codebase verified:** 2025-01-31

---

## Phase 1: Core YubiKey Validation

**Done when:** verify.sh accepts both 6-digit TOTP codes AND 44-character YubiKey OTP codes, routing each to the appropriate validator. YubiKey validation calls Yubico API and returns correct exit codes.

---

### Task 1: Add detect_code_type() function

**Files:**
- Modify: `/Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh` (insert after line 75, before current input validation)

**Step 1: Add the detect_code_type function**

Insert this function after the `reset_failures()` function (around line 75) and before the `USER_ID="${1}"` line:

```bash
# Detect code type based on format
# Returns: "totp", "yubikey", or "unknown"
detect_code_type() {
  local code="$1"
  # YubiKey OTP: exactly 44 ModHex characters (cbdefghijklnrtuv)
  if [[ "$code" =~ ^[cbdefghijklnrtuv]{44}$ ]]; then
    echo "yubikey"
  # TOTP: exactly 6 digits
  elif [[ "$code" =~ ^[0-9]{6}$ ]]; then
    echo "totp"
  else
    echo "unknown"
  fi
}
```

**Step 2: Verify function is syntactically correct**

Run: `bash -n /Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh`

Expected: No output (no syntax errors)

**Step 3: Commit**

```bash
git add verify.sh
git commit -m "feat(verify): add detect_code_type function for TOTP vs YubiKey"
```

---

### Task 2: Refactor code format validation to use detection

**Files:**
- Modify: `/Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh` (lines 85-89)

**Step 1: Replace the hardcoded 6-digit validation**

Find this block (currently lines 85-89):
```bash
# Validate OTP code format (must be exactly 6 digits)
if ! [[ "$CODE" =~ ^[0-9]{6}$ ]]; then
  echo "ERROR: Code must be exactly 6 digits" >&2
  exit 2
fi
```

Replace with:
```bash
# Detect and validate code format
CODE_TYPE=$(detect_code_type "$CODE")

if [ "$CODE_TYPE" = "unknown" ]; then
  echo "ERROR: Invalid code format. Expected 6-digit TOTP or 44-character YubiKey OTP" >&2
  exit 2
fi
```

**Step 2: Verify script still works with TOTP codes**

Run: `OTP_SECRET="JBSWY3DPEHPK3PXP" bash /Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh testuser 123456`

Expected: Exit code 1 (invalid code, but not exit 2 format error)

**Step 3: Verify script rejects invalid formats**

Run: `OTP_SECRET="JBSWY3DPEHPK3PXP" bash /Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh testuser "abc"`

Expected: Exit code 2 with message "Invalid code format"

**Step 4: Commit**

```bash
git add verify.sh
git commit -m "refactor(verify): use detect_code_type for format validation"
```

---

### Task 3: Add YubiKey configuration loading

**Files:**
- Modify: `/Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh` (insert after SECRET loading, around line 119)

**Step 1: Add YubiKey credential loading after OTP_SECRET loading**

Find the end of the SECRET loading block (after line 119, before the "if [ -z "$SECRET" ]" check). Insert:

```bash
# Get YubiKey credentials from environment or config
YUBIKEY_CLIENT_ID="${YUBIKEY_CLIENT_ID:-}"
YUBIKEY_SECRET_KEY="${YUBIKEY_SECRET_KEY:-}"

if [ -z "$YUBIKEY_CLIENT_ID" ] && [ -f "$CONFIG_FILE" ]; then
  if command -v python3 &>/dev/null; then
    YUBIKEY_CLIENT_ID=$(python3 -c "
import sys
try:
    import yaml
    with open('$CONFIG_FILE', 'r') as f:
        config = yaml.safe_load(f)
    client_id = config.get('security', {}).get('yubikey', {}).get('clientId', '')
    print(client_id if client_id else '')
except Exception:
    pass
" 2>/dev/null)
  fi
fi

if [ -z "$YUBIKEY_SECRET_KEY" ] && [ -f "$CONFIG_FILE" ]; then
  if command -v python3 &>/dev/null; then
    YUBIKEY_SECRET_KEY=$(python3 -c "
import sys
try:
    import yaml
    with open('$CONFIG_FILE', 'r') as f:
        config = yaml.safe_load(f)
    secret_key = config.get('security', {}).get('yubikey', {}).get('secretKey', '')
    print(secret_key if secret_key else '')
except Exception:
    pass
" 2>/dev/null)
  fi
fi
```

**Step 2: Verify syntax**

Run: `bash -n /Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh`

Expected: No output (no syntax errors)

**Step 3: Commit**

```bash
git add verify.sh
git commit -m "feat(verify): add YubiKey credential loading from env/config"
```

---

### Task 4: Modify SECRET requirement to be code-type-dependent

**Files:**
- Modify: `/Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh` (around lines 121-124)

**Step 1: Make SECRET check conditional on code type**

Find the current SECRET requirement block:
```bash
if [ -z "$SECRET" ]; then
  echo "ERROR: OTP_SECRET not set. Configure in environment or ~/.openclaw/config.yaml" >&2
  exit 2
fi
```

Replace with:
```bash
# Validate required credentials based on code type
if [ "$CODE_TYPE" = "totp" ] && [ -z "$SECRET" ]; then
  echo "ERROR: OTP_SECRET not set. Configure in environment or ~/.openclaw/config.yaml" >&2
  exit 2
fi

if [ "$CODE_TYPE" = "yubikey" ]; then
  if [ -z "$YUBIKEY_CLIENT_ID" ]; then
    echo "ERROR: YUBIKEY_CLIENT_ID not set. Configure in environment or ~/.openclaw/config.yaml" >&2
    exit 2
  fi
  if [ -z "$YUBIKEY_SECRET_KEY" ]; then
    echo "ERROR: YUBIKEY_SECRET_KEY not set. Configure in environment or ~/.openclaw/config.yaml" >&2
    exit 2
  fi
fi
```

**Step 2: Test YubiKey code without credentials shows proper error**

Run: `bash /Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh testuser "cccccccccccccccccccccccccccccccccccccccccccc"`

Expected: Exit code 2 with message "YUBIKEY_CLIENT_ID not set"

**Step 3: Commit**

```bash
git add verify.sh
git commit -m "feat(verify): make credential requirements code-type-dependent"
```

---

### Task 5: Move secret format validation to be TOTP-specific

**Files:**
- Modify: `/Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh` (around lines 126-135)

**Step 1: Make secret format validation conditional**

Find the current secret validation block:
```bash
# Validate secret format (base32: A-Z, 2-7, optional = padding, 16-128 chars)
if [ "${#SECRET}" -lt 16 ] || [ "${#SECRET}" -gt 128 ]; then
  echo "ERROR: Secret length must be 16-128 characters" >&2
  exit 2
fi

if ! [[ "$SECRET" =~ ^[A-Z2-7]+=*$ ]]; then
  echo "ERROR: Secret must be base32 encoded (A-Z, 2-7, optional = padding)" >&2
  exit 2
fi
```

Wrap it in a code type check:
```bash
# Validate secret format for TOTP (base32: A-Z, 2-7, optional = padding, 16-128 chars)
if [ "$CODE_TYPE" = "totp" ]; then
  if [ "${#SECRET}" -lt 16 ] || [ "${#SECRET}" -gt 128 ]; then
    echo "ERROR: Secret length must be 16-128 characters" >&2
    exit 2
  fi

  if ! [[ "$SECRET" =~ ^[A-Z2-7]+=*$ ]]; then
    echo "ERROR: Secret must be base32 encoded (A-Z, 2-7, optional = padding)" >&2
    exit 2
  fi
fi
```

**Step 2: Verify TOTP codes still validate secret format**

Run: `OTP_SECRET="invalid" bash /Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh testuser 123456`

Expected: Exit code 2 with message about secret length

**Step 3: Commit**

```bash
git add verify.sh
git commit -m "refactor(verify): make secret format validation TOTP-specific"
```

---

### Task 6: Move oathtool check to be TOTP-specific

**Files:**
- Modify: `/Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh` (around lines 137-144)

**Step 1: Make oathtool check conditional**

Find the current oathtool check:
```bash
# Check if oathtool is available
if ! command -v oathtool &> /dev/null; then
  echo "ERROR: oathtool not found. Install with:" >&2
  echo "  macOS:  brew install oath-toolkit" >&2
  echo "  Fedora: sudo dnf install oathtool" >&2
  echo "  Ubuntu: sudo apt-get install oathtool" >&2
  exit 2
fi
```

Wrap it:
```bash
# Check if oathtool is available (only needed for TOTP)
if [ "$CODE_TYPE" = "totp" ]; then
  if ! command -v oathtool &> /dev/null; then
    echo "ERROR: oathtool not found. Install with:" >&2
    echo "  macOS:  brew install oath-toolkit" >&2
    echo "  Fedora: sudo dnf install oathtool" >&2
    echo "  Ubuntu: sudo apt-get install oathtool" >&2
    exit 2
  fi
fi
```

**Step 2: Verify syntax**

Run: `bash -n /Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh`

Expected: No output

**Step 3: Commit**

```bash
git add verify.sh
git commit -m "refactor(verify): make oathtool check TOTP-specific"
```

---

### Task 7: Add validate_yubikey() function

**Files:**
- Modify: `/Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh` (insert after detect_code_type function)

**Step 1: Add the YubiKey validation function**

Insert after detect_code_type() function:

```bash
# Validate YubiKey OTP against Yubico Cloud API
# Returns: 0 for valid, 1 for invalid, 2 for config/network error
validate_yubikey() {
  local otp="$1"
  local client_id="$2"
  local secret_key_b64="$3"

  # Decode the base64-encoded secret key (Yubico provides it base64-encoded)
  local secret_key
  secret_key=$(printf '%s' "$secret_key_b64" | base64 -d 2>/dev/null)
  if [ -z "$secret_key" ]; then
    echo "ERROR: Failed to decode YUBIKEY_SECRET_KEY (must be valid base64)" >&2
    return 2
  fi

  # Generate random nonce (16 chars, alphanumeric)
  local nonce
  nonce=$(head -c 32 /dev/urandom | base64 | tr -dc 'a-zA-Z0-9' | head -c 16)

  # Build parameters for signing (alphabetically sorted)
  local params="id=${client_id}&nonce=${nonce}&otp=${otp}"

  # Generate HMAC-SHA1 signature using decoded secret key
  local signature
  signature=$(printf '%s' "$params" | openssl dgst -sha1 -hmac "$secret_key" -binary | base64)

  # URL-encode the signature (+ and / and = need encoding)
  local signature_encoded
  signature_encoded=$(printf '%s' "$signature" | sed 's/+/%2B/g; s/\//%2F/g; s/=/%3D/g')

  # Call Yubico API
  local response
  response=$(curl -s --max-time 10 "https://api.yubico.com/wsapi/2.0/verify?${params}&h=${signature_encoded}" 2>/dev/null)

  if [ -z "$response" ]; then
    echo "ERROR: Failed to contact Yubico API (network error or timeout)" >&2
    return 2
  fi

  # Parse response fields (format: key=value per line)
  local resp_status resp_nonce resp_otp
  resp_status=$(echo "$response" | grep -E '^status=' | cut -d'=' -f2 | tr -d '\r')
  resp_nonce=$(echo "$response" | grep -E '^nonce=' | cut -d'=' -f2 | tr -d '\r')
  resp_otp=$(echo "$response" | grep -E '^otp=' | cut -d'=' -f2 | tr -d '\r')

  # Verify nonce matches (prevents replay attacks on API responses)
  if [ "$resp_nonce" != "$nonce" ]; then
    echo "ERROR: YubiKey API nonce mismatch (possible MITM attack)" >&2
    return 2
  fi

  # Verify OTP in response matches what we sent
  if [ "$resp_otp" != "$otp" ]; then
    echo "ERROR: YubiKey API OTP mismatch in response" >&2
    return 2
  fi

  case "$resp_status" in
    OK)
      return 0
      ;;
    REPLAYED_OTP)
      echo "❌ YubiKey OTP already used (replay attack prevented)" >&2
      return 1
      ;;
    BAD_OTP)
      echo "❌ Invalid YubiKey OTP" >&2
      return 1
      ;;
    BAD_SIGNATURE)
      echo "ERROR: YubiKey API signature mismatch (check YUBIKEY_SECRET_KEY)" >&2
      return 2
      ;;
    NO_SUCH_CLIENT)
      echo "ERROR: Invalid YUBIKEY_CLIENT_ID" >&2
      return 2
      ;;
    MISSING_PARAMETER|OPERATION_NOT_ALLOWED)
      echo "ERROR: YubiKey API request error: $resp_status" >&2
      return 2
      ;;
    BACKEND_ERROR|NOT_ENOUGH_ANSWERS)
      echo "ERROR: Yubico API backend error. Try again." >&2
      return 2
      ;;
    *)
      echo "ERROR: Unexpected YubiKey API response: $resp_status" >&2
      return 2
      ;;
  esac
}
```

**Step 2: Verify syntax**

Run: `bash -n /Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh`

Expected: No output

**Step 3: Commit**

```bash
git add verify.sh
git commit -m "feat(verify): add validate_yubikey function with Yubico API call"
```

---

### Task 8: Add YubiKey validation path in main flow

**Files:**
- Modify: `/Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh` (around lines 186-210, the TOTP validation section)

**Step 1: Wrap TOTP validation and add YubiKey branch**

Find the TOTP validation section that starts with:
```bash
# Validate the code using oathtool
EXPECTED=$(oathtool --totp -b "$SECRET" 2>/dev/null)
```

Wrap it in a code type check and add the YubiKey path. Replace the entire TOTP validation section (from "Validate the code" to the end of the clock skew handling, ending before "# Code is valid") with:

```bash
# Validate based on code type
if [ "$CODE_TYPE" = "totp" ]; then
  # Validate the TOTP code using oathtool
  EXPECTED=$(oathtool --totp -b "$SECRET" 2>/dev/null)
  if [ "$?" -ne 0 ]; then
    echo "ERROR: Failed to generate TOTP. Check secret format (base32)." >&2
    exit 2
  fi

  if [ "$CODE" != "$EXPECTED" ]; then
    # Try previous and next windows for clock skew
    NOW=$(date +%s)
    EXPECTED_PREV=$(oathtool --totp -b "$SECRET" -N "@$((NOW - 30))" 2>/dev/null || true)
    EXPECTED_NEXT=$(oathtool --totp -b "$SECRET" -N "@$((NOW + 30))" 2>/dev/null || true)

    if [ "$CODE" != "$EXPECTED_PREV" ] && [ "$CODE" != "$EXPECTED_NEXT" ]; then
      FAIL_NOW_MS=$(date +%s)000
      record_failure "$USER_ID" "$FAIL_NOW_MS"
      NEW_FAILURE_COUNT=$(jq -r --arg userId "$USER_ID" '.failureCounts[$userId].count // 0' "$STATE_FILE")
      audit_log "VERIFY" "$USER_ID" "TOTP_FAIL"
      run_failure_hook "VERIFY_FAIL" "$USER_ID" "$NEW_FAILURE_COUNT"
      echo "❌ Invalid OTP code" >&2
      exit 1
    fi
  fi

elif [ "$CODE_TYPE" = "yubikey" ]; then
  # Extract public ID for audit logging (first 12 characters)
  YUBIKEY_PUBLIC_ID="${CODE:0:12}"

  # Validate against Yubico API
  validate_yubikey "$CODE" "$YUBIKEY_CLIENT_ID" "$YUBIKEY_SECRET_KEY"
  YUBIKEY_RESULT=$?

  if [ "$YUBIKEY_RESULT" -eq 2 ]; then
    # Configuration/API error
    exit 2
  elif [ "$YUBIKEY_RESULT" -eq 1 ]; then
    # Invalid OTP
    FAIL_NOW_MS=$(date +%s)000
    record_failure "$USER_ID" "$FAIL_NOW_MS"
    NEW_FAILURE_COUNT=$(jq -r --arg userId "$USER_ID" '.failureCounts[$userId].count // 0' "$STATE_FILE")
    audit_log "VERIFY" "$USER_ID" "YUBIKEY_FAIL:$YUBIKEY_PUBLIC_ID"
    run_failure_hook "VERIFY_FAIL" "$USER_ID" "$NEW_FAILURE_COUNT"
    exit 1
  fi
  # YUBIKEY_RESULT=0 means success, continue to update state
fi
```

**Step 2: Verify syntax**

Run: `bash -n /Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh`

Expected: No output

**Step 3: Commit**

```bash
git add verify.sh
git commit -m "feat(verify): add YubiKey validation path in main flow"
```

---

### Task 9: Update success message to indicate verification method

**Files:**
- Modify: `/Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh` (the final success output, around line 259)

**Step 1: Make success message method-aware**

Find the final success line:
```bash
echo "✅ OTP verified for $USER_ID (valid for $INTERVAL_HOURS hours)"
```

Replace with:
```bash
if [ "$CODE_TYPE" = "yubikey" ]; then
  echo "✅ YubiKey verified for $USER_ID (valid for $INTERVAL_HOURS hours)"
else
  echo "✅ OTP verified for $USER_ID (valid for $INTERVAL_HOURS hours)"
fi
```

**Step 2: Update audit log success entry to include method**

Find the audit_log call before the success message:
```bash
audit_log "VERIFY" "$USER_ID" "VERIFY_SUCCESS"
```

Replace with:
```bash
if [ "$CODE_TYPE" = "yubikey" ]; then
  audit_log "VERIFY" "$USER_ID" "YUBIKEY_SUCCESS:$YUBIKEY_PUBLIC_ID"
else
  audit_log "VERIFY" "$USER_ID" "TOTP_SUCCESS"
fi
```

**Step 3: Verify syntax**

Run: `bash -n /Volumes/T9/ryan-homedir/devel/otp-challenger/verify.sh`

Expected: No output

**Step 4: Commit**

```bash
git add verify.sh
git commit -m "feat(verify): add method-specific success messages and audit entries"
```

---

### Phase 1 Verification

Run the full test suite to ensure TOTP still works:

```bash
cd /Volumes/T9/ryan-homedir/devel/otp-challenger
bats tests/verify.bats
```

Expected: All existing TOTP tests pass (40 tests)

If any tests fail, debug and fix before proceeding to Phase 2.
