/**
 * Context Monitor System
 * Automatically monitors context length and triggers optimization when needed
 */

const fs = require('fs');
const path = require('path');
const ContextOptimizer = require('../skills/context-optimizer/optimize.js');

class ContextMonitor {
  constructor(workspaceDir = '/home/blackworm/.openclaw/workspace') {
    this.workspaceDir = workspaceDir;
    this.configFile = path.join(this.workspaceDir, 'task_processing_config.json');
    this.config = this.loadConfig();
    this.optimizer = new ContextOptimizer(workspaceDir, 'task_processing_config.json');
    this.lastOptimizationTime = null;
  }

  loadConfig() {
    try {
      if (fs.existsSync(this.configFile)) {
        const configContent = fs.readFileSync(this.configFile, 'utf8');
        return JSON.parse(configContent).context_optimization;
      } else {
        // Return default configuration if file doesn't exist
        return {
          enabled: true,
          schedule: {
            enabled: true,
            interval_minutes: 60,
            max_context_length: 50,
            min_messages_for_optimization: 30
          },
          summarization: {
            max_bullet_points: 20,
            include_facts: true,
            preserve_important_info: true
          },
          storage: {
            summary_file_pattern: 'context-summary-YYYY-MM-DD.md',
            memory_file: 'MEMORY.md',
            backup_enabled: true
          },
          cleanup: {
            remove_old_context: true,
            keep_recent_messages: 10,
            auto_purge: true
          },
          output_optimization: {
            prevent_overflow: true,
            token_saving_mode: true,
            result_summarization: true
          }
        };
      }
    } catch (error) {
      console.error('Error loading configuration:', error);
      return {
        enabled: true,
        schedule: {
          enabled: true,
          interval_minutes: 60,
          max_context_length: 50,
          min_messages_for_optimization: 30
        },
        summarization: {
          max_bullet_points: 20,
          include_facts: true,
          preserve_important_info: true
        },
        storage: {
          summary_file_pattern: 'context-summary-YYYY-MM-DD.md',
          memory_file: 'MEMORY.md',
          backup_enabled: true
        },
        cleanup: {
          remove_old_context: true,
          keep_recent_messages: 10,
          auto_purge: true
        },
        output_optimization: {
          prevent_overflow: true,
          token_saving_mode: true,
          result_summarization: true
        }
      };
    }
  }

  /**
   * Checks if context needs optimization based on message count
   */
  shouldOptimize(messages) {
    if (!this.config.enabled) {
      return false;
    }

    // Check if we have enough messages to warrant optimization
    const conversationMessages = messages.filter(msg => 
      msg.role !== 'system' && 
      !msg.content.includes('HEARTBEAT_OK') &&
      !msg.content.startsWith('✅ New session started')
    );

    if (conversationMessages.length >= this.config.schedule.min_messages_for_optimization) {
      // Check if enough time has passed since last optimization
      if (this.lastOptimizationTime) {
        const timeSinceLastOpt = (Date.now() - this.lastOptimizationTime) / (1000 * 60); // minutes
        return timeSinceLastOpt >= 10; // At least 10 minutes since last optimization
      }
      return true;
    }

    return false;
  }

  /**
   * Runs optimization automatically when needed
   */
  async autoOptimizeIfNeeded(messages) {
    if (this.shouldOptimize(messages)) {
      console.log(`Context monitor: Starting automatic optimization...`);
      
      try {
        const result = this.optimizer.optimizeContext(messages);
        
        if (result.summary && result.summary.optimized) {
          this.lastOptimizationTime = Date.now();
          console.log(`Context monitor: Automatic optimization completed!`);
          console.log(`- Generated ${result.summary.bulletPoints.length} key points`);
          console.log(`- Extracted ${result.summary.factsToRemember.length} facts`);
          console.log(`- Context cleared: ${result.contextCleared}`);
          
          // Return the optimized messages if available
          return result.optimizedMessages || messages;
        } else if (result.message) {
          console.log(`Context monitor: ${result.message}`);
        }
      } catch (error) {
        console.error('Context monitor: Error during auto-optimization:', error);
      }
    }
    
    // Return original messages if no optimization was needed
    return messages;
  }

  /**
   * Checks if a task is too large and needs to be split
   */
  isLargeTask(task) {
    // Estimate token count by character count (rough estimation: 1 token ≈ 4 characters)
    const charCount = task.length;
    const estimatedTokens = Math.ceil(charCount / 4);
    
    // Most models have a limit around 32k-128k tokens, but we'll be conservative
    // to prevent "prompt too large" errors
    const tokenThreshold = 20000; // Conservative threshold
    
    return estimatedTokens > tokenThreshold;
  }

  /**
   * Splits a large task into smaller subtasks
   */
  splitLargeTask(task) {
    // Split by paragraphs or sentences if the task is too large
    const paragraphs = task.split(/\n\s*\n/);
    
    if (paragraphs.length > 1) {
      // If we have multiple paragraphs, we can split by those
      return paragraphs.filter(p => p.trim().length > 0);
    } else {
      // If it's one large paragraph, split by sentences
      const sentences = task.match(/[^\.!?]*[\.!?]+/g) || [task];
      if (sentences.length > 1) {
        return sentences.filter(s => s.trim().length > 0);
      } else {
        // If it's still one sentence, split by chunks of reasonable size
        const chunkSize = 2000; // Approximate size that should fit within token limits
        const chunks = [];
        for (let i = 0; i < task.length; i += chunkSize) {
          chunks.push(task.substring(i, i + chunkSize));
        }
        return chunks;
      }
    }
  }

  /**
   * Processes a task, automatically optimizing context and splitting if needed
   */
  async processTask(task, messages = []) {
    // First, check if context needs optimization
    const optimizedMessages = await this.autoOptimizeIfNeeded(messages);
    
    // Then check if the task itself is too large
    if (this.isLargeTask(task)) {
      console.log(`Task is too large, splitting into smaller subtasks...`);
      
      const subtasks = this.splitLargeTask(task);
      console.log(`Split into ${subtasks.length} subtasks`);
      
      const results = [];
      for (let i = 0; i < subtasks.length; i++) {
        console.log(`Processing subtask ${i + 1}/${subtasks.length}...`);
        
        // Process each subtask individually
        const subtaskResult = await this.processSubtask(subtasks[i], optimizedMessages);
        results.push({
          subtask: i + 1,
          original: subtasks[i],
          result: subtaskResult
        });
        
        // Optimize context again between subtasks if needed
        const updatedMessages = optimizedMessages.concat([
          { role: 'user', content: subtasks[i] },
          { role: 'assistant', content: subtaskResult }
        ]);
        await this.autoOptimizeIfNeeded(updatedMessages);
      }
      
      return {
        status: 'split_and_processed',
        subtasks: results,
        summary: `Processed large task by splitting into ${subtasks.length} subtasks`
      };
    } else {
      // Task is not too large, return as-is
      return {
        status: 'ready',
        task: task,
        messages: optimizedMessages
      };
    }
  }

  /**
   * Process a single subtask
   */
  async processSubtask(subtask, messages) {
    // In a real implementation, this would interface with the actual task processing system
    // For now, we'll just return a placeholder result
    return `Processed subtask: "${subtask.substring(0, 100)}${subtask.length > 100 ? '...' : ''}"`;
  }
}

module.exports = ContextMonitor;