#!/usr/bin/env node

/**
 * Large Task Processing Command Handler
 * Implements automatic task splitting and context optimization for large tasks
 */

const path = require('path');
const ContextMonitor = require('../systems/context-monitor.js');

async function handleProcessTaskCommand(task) {
  try {
    console.log('Processing potentially large task...');
    
    // Create context monitor instance
    const contextMonitor = new ContextMonitor('/home/blackworm/.openclaw/workspace');
    
    // Process the task with automatic optimization and splitting
    const result = await contextMonitor.processTask(task);
    
    if (result.status === 'split_and_processed') {
      console.log(`Large task was automatically split and processed:`);
      console.log(`- Split into ${result.subtasks.length} subtasks`);
      console.log(`- All subtasks completed successfully`);
      
      const message = `Large task automatically split and processed:\n\n`;
      const subtaskMessage = `- Split into ${result.subtasks.length} subtasks\n`;
      const statusMessage = `- All subtasks completed successfully\n`;
      const summaryMessage = result.summary;
      
      const finalMessage = message + subtaskMessage + statusMessage + summaryMessage;
      
      return {
        success: true,
        message: finalMessage,
        taskStatus: result.status,
        subtasksProcessed: result.subtasks.length
      };
    } else {
      console.log(`Task is within size limits and ready for processing.`);
      
      return {
        success: true,
        message: `Task is within size limits and ready for processing.`,
        taskStatus: result.status
      };
    }
  } catch (error) {
    console.error('Error during task processing:', error);
    return {
      success: false,
      message: `Error during task processing: ${error.message}`
    };
  }
}

// If called directly with arguments, process the task
if (require.main === module) {
  // Get the task from command line arguments
  const args = process.argv.slice(2);
  const task = args.join(' ');
  
  if (!task) {
    console.log("Usage: node commands/process-task.js \"your large task here\"");
    process.exit(1);
  }
  
  handleProcessTaskCommand(task)
    .then(result => {
      if (result.success) {
        console.log("\nTask processing completed successfully!");
        console.log(result.message);
      } else {
        console.error("\nTask processing failed:", result.message);
      }
    })
    .catch(error => {
      console.error("Task processing error:", error);
    });
}

module.exports = { handleProcessTaskCommand };