#!/usr/bin/env python3
"""
AI Opportunity Scout — Scoring & Report Generator

Usage:
  # Score opportunities from JSON input
  python3 scout.py score --input opportunities.json --output report.md

  # Score from stdin (agent pipes data)
  echo '[{"name":"...", "problem":"...", "evidence":["..."], "demand":4, "competition":3, "feasibility":5, "monetization":4}]' | python3 scout.py score --output report.md

  # Interactive mode: agent provides scores via prompts
  python3 scout.py interactive --niche "AI agents"

  # Generate empty template
  python3 scout.py template --niche "AI agents" --count 5 --output opportunities.json
"""

import argparse
import json
import sys
from datetime import datetime
from pathlib import Path


def validate_score(score: int, field: str) -> int:
    """Validate score is 1-5."""
    if not isinstance(score, int) or score < 1 or score > 5:
        print(f"⚠️  Invalid {field} score: {score}. Must be 1-5. Clamping.", file=sys.stderr)
        return max(1, min(5, int(score)))
    return score


def score_verdict(total: int) -> str:
    """Return verdict based on total score."""
    if total >= 16:
        return "🔥 BUILD NOW"
    elif total >= 12:
        return "👍 Strong"
    elif total >= 8:
        return "🤔 Monitor"
    else:
        return "❌ Skip"


def generate_report(opportunities: list, niche: str) -> str:
    """Generate markdown report from scored opportunities."""
    # Sort by total score descending
    for opp in opportunities:
        opp["demand"] = validate_score(opp.get("demand", 3), "demand")
        opp["competition"] = validate_score(opp.get("competition", 3), "competition")
        opp["feasibility"] = validate_score(opp.get("feasibility", 3), "feasibility")
        opp["monetization"] = validate_score(opp.get("monetization", 3), "monetization")
        opp["total"] = opp["demand"] + opp["competition"] + opp["feasibility"] + opp["monetization"]
        opp["verdict"] = score_verdict(opp["total"])

    opportunities.sort(key=lambda x: x["total"], reverse=True)

    date = datetime.now().strftime("%Y-%m-%d")
    lines = []
    lines.append(f"# Opportunity Scout: {niche} — {date}\n")

    # Executive summary
    top3 = opportunities[:3]
    build_now = [o for o in opportunities if o["total"] >= 16]
    strong = [o for o in opportunities if 12 <= o["total"] < 16]

    lines.append("## Executive Summary\n")
    lines.append(f"Scanned **{len(opportunities)}** opportunities in the **{niche}** niche.")
    lines.append(f"- 🔥 Build Now: **{len(build_now)}**")
    lines.append(f"- 👍 Strong: **{len(strong)}**")
    lines.append(f"- 🤔 Monitor: **{len([o for o in opportunities if 8 <= o['total'] < 12])}**")
    lines.append(f"- ❌ Skip: **{len([o for o in opportunities if o['total'] < 8])}**\n")

    # Top 3
    lines.append("## 🏆 Top 3 Opportunities\n")
    for i, opp in enumerate(top3, 1):
        lines.append(f"### {i}. {opp['name']} (Score: {opp['total']}/20) {opp['verdict']}")
        lines.append(f"- **Problem:** {opp.get('problem', 'N/A')}")

        evidence = opp.get("evidence", [])
        if evidence:
            lines.append(f"- **Evidence:** {'; '.join(evidence[:3])}")
        
        lines.append(f"- **Scores:** D:{opp['demand']} C:{opp['competition']} F:{opp['feasibility']} M:{opp['monetization']}")

        action = opp.get("action", "")
        if action:
            lines.append(f"- **Action:** {action}")
        lines.append("")

    # Full table
    lines.append("## All Findings\n")
    lines.append("| # | Opportunity | D | C | F | M | Total | Verdict |")
    lines.append("|---|------------|---|---|---|---|-------|---------|")
    for i, opp in enumerate(opportunities, 1):
        lines.append(f"| {i} | {opp['name']} | {opp['demand']} | {opp['competition']} | {opp['feasibility']} | {opp['monetization']} | {opp['total']} | {opp['verdict']} |")
    lines.append("")

    # Recommendation
    if top3:
        best = top3[0]
        lines.append("## Recommendation\n")
        lines.append(f"**Build first: {best['name']}** (Score: {best['total']}/20)")
        lines.append(f"\n{best.get('problem', '')}")
        if best.get("action"):
            lines.append(f"\n**Next steps:** {best['action']}")
    
    lines.append(f"\n---\n*Generated by [AI Opportunity Scout](https://clawdhub.com) on {date}*")

    return "\n".join(lines)


def generate_template(niche: str, count: int) -> list:
    """Generate empty opportunity template."""
    return [
        {
            "name": f"Opportunity {i+1}",
            "problem": "Describe the unmet need or pain point",
            "evidence": ["link1", "link2"],
            "demand": 3,
            "competition": 3,
            "feasibility": 3,
            "monetization": 3,
            "action": "What to build and how to monetize"
        }
        for i in range(count)
    ]


def main():
    parser = argparse.ArgumentParser(description="AI Opportunity Scout — Scoring & Report Generator")
    subparsers = parser.add_subparsers(dest="command", help="Command to run")

    # Score command
    score_parser = subparsers.add_parser("score", help="Score opportunities and generate report")
    score_parser.add_argument("--input", "-i", help="JSON file with opportunities (or stdin)")
    score_parser.add_argument("--output", "-o", help="Output markdown file (or stdout)")
    score_parser.add_argument("--niche", "-n", default="General", help="Niche name for report header")

    # Template command
    tmpl_parser = subparsers.add_parser("template", help="Generate empty opportunities template")
    tmpl_parser.add_argument("--niche", "-n", default="General", help="Niche name")
    tmpl_parser.add_argument("--count", "-c", type=int, default=5, help="Number of template entries")
    tmpl_parser.add_argument("--output", "-o", help="Output JSON file (or stdout)")

    args = parser.parse_args()

    if args.command == "score":
        # Read input
        if args.input:
            with open(args.input, "r") as f:
                opportunities = json.load(f)
        else:
            # Read from stdin
            data = sys.stdin.read()
            opportunities = json.loads(data)

        if not isinstance(opportunities, list):
            print("Error: Input must be a JSON array of opportunities.", file=sys.stderr)
            sys.exit(1)

        # Generate report
        report = generate_report(opportunities, args.niche)

        # Output
        if args.output:
            with open(args.output, "w") as f:
                f.write(report)
            print(f"✅ Report saved to {args.output}")
        else:
            print(report)

    elif args.command == "template":
        template = generate_template(args.niche, args.count)
        output = json.dumps(template, indent=2)

        if args.output:
            with open(args.output, "w") as f:
                f.write(output)
            print(f"✅ Template saved to {args.output}")
        else:
            print(output)

    else:
        parser.print_help()


if __name__ == "__main__":
    main()
