# GitHub Action: OpenSpec Auto-Archive on Merge

Add this to `.github/workflows/openspec-archive.yml` in any repo using OpenSpec.

## How It Works

1. Triggers when a PR is merged to main
2. Looks for `OpenSpec change: <name>` in the PR body
3. Runs `openspec archive --yes` on a new branch
4. Opens a PR with the archive and spec sync changes
5. Deletes the original merged branch

## Requirements

- PR body must contain `OpenSpec change: <change-name>` (exact format, must match directory name under `openspec/changes/`)
- The change directory must exist at `openspec/changes/<change-name>/` on main after merge
- Main specs must be in valid format (`## Purpose` + `## Requirements` sections)

## Workflow YAML

```yaml
name: OpenSpec Archive on Merge

on:
  pull_request:
    types: [closed]

jobs:
  archive:
    if: github.event.pull_request.merged == true
    runs-on: ubuntu-latest
    permissions:
      contents: write
      pull-requests: write
    steps:
      - uses: actions/checkout@v4
        with:
          ref: main
          fetch-depth: 0

      - uses: actions/setup-node@v4
        with:
          node-version: '22'

      - name: Install OpenSpec CLI
        run: npm install -g @fission-ai/openspec

      - name: Extract change name from PR body
        id: extract
        env:
          PR_BODY: ${{ github.event.pull_request.body }}
        run: |
          CHANGE=$(echo "$PR_BODY" | grep -oP 'OpenSpec change:\s*\K[\w-]+' | head -1)
          if [ -z "$CHANGE" ]; then
            echo "No 'OpenSpec change: <name>' found in PR body, skipping."
            echo "skip=true" >> $GITHUB_OUTPUT
          else
            echo "Found OpenSpec change: $CHANGE"
            echo "change=$CHANGE" >> $GITHUB_OUTPUT
            echo "skip=false" >> $GITHUB_OUTPUT
          fi

      - name: Verify change exists
        if: steps.extract.outputs.skip == 'false'
        id: verify
        run: |
          if [ -d "openspec/changes/${{ steps.extract.outputs.change }}" ]; then
            echo "Change directory found."
            echo "exists=true" >> $GITHUB_OUTPUT
          else
            echo "Change directory not found at openspec/changes/${{ steps.extract.outputs.change }}, skipping."
            echo "exists=false" >> $GITHUB_OUTPUT
          fi

      - name: Archive and sync specs
        if: steps.extract.outputs.skip == 'false' && steps.verify.outputs.exists == 'true'
        run: openspec archive --yes "${{ steps.extract.outputs.change }}"

      - name: Open archive PR
        if: steps.extract.outputs.skip == 'false' && steps.verify.outputs.exists == 'true'
        env:
          GH_TOKEN: ${{ secrets.GITHUB_TOKEN }}
        run: |
          CHANGE="${{ steps.extract.outputs.change }}"
          BRANCH="openspec/archive-${CHANGE}"
          git config user.name "github-actions[bot]"
          git config user.email "github-actions[bot]@users.noreply.github.com"
          git checkout -b "$BRANCH"
          git add -A
          git diff --cached --quiet && echo "Nothing to commit" && exit 0
          git commit -m "chore(openspec): archive and sync ${CHANGE}"
          git push origin "$BRANCH"
          gh pr create \
            --base main \
            --head "$BRANCH" \
            --title "chore(openspec): archive ${CHANGE}" \
            --body "Auto-generated by OpenSpec Archive workflow.

Syncs delta specs into main specs and moves \`openspec/changes/${CHANGE}\` to archive.

Source PR: #${{ github.event.pull_request.number }}"

      - name: Clean up merged branch
        if: steps.extract.outputs.skip == 'false'
        env:
          GH_TOKEN: ${{ secrets.GITHUB_TOKEN }}
        run: |
          BRANCH="${{ github.event.pull_request.head.ref }}"
          git push origin --delete "$BRANCH" 2>/dev/null || echo "Branch already deleted or protected"

```

## Troubleshooting

**Archive fails with validation error:** Main specs are likely malformed. Run `openspec validate --specs` and fix any issues. All specs need `## Purpose` and `## Requirements` sections.

**Change directory not found:** The PR's OpenSpec change name doesn't match a directory in `openspec/changes/`. Check for typos in the PR body. Always verify with `ls openspec/changes/` before writing the PR body.

**Nothing to commit:** The archive was already done (idempotent) or the change was already archived.

**Archive PR conflicts:** If multiple changes merge close together, archive PRs may conflict. Merge them sequentially.
