# For AI Agents

This reference is optimized for AI assistants using the OpenGraph.io APIs. It contains decision trees, patterns, and detailed parameter documentation.

## API Overview

| Endpoint | Method | Purpose |
|----------|--------|---------|
| `/site/{url}` | GET | Extract OpenGraph/meta tags |
| `/scrape/{url}` | GET | Fetch rendered HTML |
| `/screenshot/{url}` | GET | Capture webpage screenshot |
| `/extract/{url}` | GET | Pull specific HTML elements |
| `/query/{url}` | POST | Ask AI questions about a page |
| `/image-agent/sessions` | POST | Create image generation session |
| `/image-agent/sessions/{id}/generate` | POST | Generate an image |
| `/image-agent/sessions/{id}/iterate` | POST | Refine an existing image |
| `/image-agent/assets/{id}/file` | GET | Download generated image |

Base URLs:
- Web data: `https://opengraph.io/api/1.1/`
- Image generation: `https://opengraph.io/image-agent/`

All requests require `app_id` query parameter.

---

## Decision Trees

### "I need data from a URL"

```
Is it meta tags / link preview data?
  → GET /site/{url}

Is it raw HTML content?
  → GET /scrape/{url} (add use_proxy=true if geo-blocked)

Is it specific elements (headings, links, etc.)?
  → GET /extract/{url}?html_elements=h1,h2,p

Is it a complex question about the page?
  → POST /query/{url} (paid feature)

Is it a visual screenshot?
  → GET /screenshot/{url}
```

### "I need to generate an image"

```
Is it a technical diagram (flowchart, architecture, ERD)?
  → kind: "diagram"
  → Prefer diagramCode + diagramFormat for control
  → Use outputStyle: "standard" (premium may alter layout)

Is it an icon or logo?
  → kind: "icon"
  → Set transparent: true

Is it a social media image (OG, Twitter card)?
  → kind: "social-card"
  → Use appropriate aspectRatio

Is it a basic QR code?
  → kind: "qr-code"

Is it a premium QR marketing card with design elements?
  → kind: "illustration"
  → Describe the full design in prompt (gradients, CTAs, mockups)
  → Use outputStyle: "premium"

Is it a general illustration?
  → kind: "illustration"
  → Be descriptive in prompt
```

---

## Image Generation Parameters

### Required
```json
{
  "prompt": "Description of image OR diagram syntax"
}
```

### Optional Parameters

**Type & Dimensions:**
```json
{
  "kind": "illustration | diagram | icon | social-card | qr-code",
  "aspectRatio": "og-image | twitter-card | square | wide | portrait | icon-large | ..."
}
```

**Styling:**
```json
{
  "stylePreset": "github-dark | vercel | stripe | neon-cyber | pastel | minimal-mono | ...",
  "brandColors": ["#0033A0", "#FF8C00"],
  "stylePreferences": "modern, minimalist, corporate",
  "outputStyle": "draft | standard | premium"
}
```

**Diagrams:**
```json
{
  "diagramCode": "flowchart LR\n  A-->B-->C",
  "diagramFormat": "mermaid | d2 | vega",
  "diagramTemplate": "auth-flow | microservices | ci-cd | ..."
}
```

**Quality & Output:**
```json
{
  "quality": "low | medium | high | fast",
  "transparent": true,
  "autoCrop": true
}
```

---

## QR Code Generation

### Basic QR Code
Just the functional QR code with minimal styling.

```json
{
  "prompt": "QR code for https://example.com",
  "kind": "qr-code"
}
```

### Premium QR Marketing Card
Complete marketing asset with QR embedded in professional design.

```json
{
  "prompt": "Premium marketing card with QR code for https://myapp.com/download, cosmic purple gradient background with floating 3D spheres, glowing accents, 'SCAN TO DOWNLOAD' call-to-action text below the code",
  "kind": "illustration",
  "aspectRatio": "square",
  "outputStyle": "premium",
  "brandColors": ["#6B4CE6", "#9B6DFF", "#E0D4FF"],
  "stylePreferences": "modern, cosmic, premium marketing, 3D elements, particle effects"
}
```

### Premium QR Examples

**App Store Download Card:**
```json
{
  "prompt": "App download marketing card with QR code linking to app store, iPhone mockup showing the app interface, modern gradient background, 'Download Now' CTA",
  "kind": "illustration",
  "aspectRatio": "og-image",
  "outputStyle": "premium",
  "brandColors": ["#0066FF", "#00D4AA"]
}
```

**Restaurant Menu QR:**
```json
{
  "prompt": "Restaurant menu QR card with elegant design, marble texture background, gold accents, 'VIEW MENU' text, upscale dining aesthetic",
  "kind": "illustration",
  "aspectRatio": "square",
  "outputStyle": "premium",
  "brandColors": ["#1a1a1a", "#C9A962"]
}
```

---

## Common Patterns

### Blog Hero Image
```json
{
  "prompt": "Modern illustration of [topic] with [style description]",
  "kind": "illustration",
  "aspectRatio": "og-image",
  "brandColors": ["#primary", "#secondary"],
  "outputStyle": "premium"
}
```

### Architecture Diagram (Reliable)
```json
{
  "diagramCode": "flowchart LR\n  A[Client] --> B[API Gateway]\n  B --> C[Auth Service]\n  B --> D[Core Service]",
  "diagramFormat": "mermaid",
  "kind": "diagram",
  "aspectRatio": "og-image",
  "brandColors": ["#0033A0", "#00A3E0"],
  "outputStyle": "standard"
}
```

### App Icon
```json
{
  "prompt": "Modern fintech app icon, dollar sign symbol, clean design",
  "kind": "icon",
  "transparent": true,
  "aspectRatio": "icon-large",
  "stylePreset": "stripe"
}
```

---

## Gotchas & Edge Cases

### ❌ Don't mix diagram syntax with description in prompt
```json
// WRONG - will fail
{
  "prompt": "graph LR A-->B Create a beautiful premium diagram"
}

// CORRECT - use diagramCode for syntax
{
  "diagramCode": "graph LR\n  A-->B",
  "diagramFormat": "mermaid",
  "stylePreferences": "beautiful, premium"
}
```

### ❌ Don't use premium for structure-critical diagrams
Premium output uses AI polish which may alter layout. Use `"outputStyle": "standard"` for diagrams where structure matters.

### ✅ Save sessionId and assetId
You'll need these for iteration and downloading.

### ✅ Use diagramCode for programmatic diagrams
If you're building Mermaid/D2 syntax programmatically, use `diagramCode` + `diagramFormat` to bypass AI generation.

### ⚠️ Free tier limits
- 100/month: Site, Screenshot, Scrape, Extract
- 4/month: Image Generation
- Query: Paid only

---

## Error Handling

| Error | Cause | Solution |
|-------|-------|----------|
| "Invalid diagram syntax" | Malformed Mermaid/D2/Vega | Fix syntax, validate locally first |
| "Rate limit exceeded" | Hit plan limits | Check dashboard, upgrade or wait |
| "Invalid URL" | Malformed or unreachable URL | Verify URL is accessible |
| "API key required" | Missing app_id | Add app_id query parameter |

---

## Aspect Ratio Quick Reference

| Preset | Dimensions | Use case |
|--------|------------|----------|
| `og-image` | 1200×630 | OpenGraph/Facebook |
| `twitter-card` | 1200×600 | Twitter cards |
| `instagram-square` | 1080×1080 | Instagram feed |
| `instagram-story` | 1080×1920 | Stories/Reels |
| `square` | 1:1 | General square |
| `wide` | 16:9 | Presentations |
| `icon-large` | 512×512 | App icons |

## Style Preset Quick Reference

| Preset | Vibe |
|--------|------|
| `github-dark` | Dark mode, monospace, developer |
| `vercel` | Modern, gradient, sleek |
| `stripe` | Professional, fintech |
| `neon-cyber` | Cyberpunk, glowing |
| `pastel` | Soft, friendly |
| `minimal-mono` | Black/white, typography |
