# API Reference

Complete reference for all OpenGraph.io API endpoints and parameters.

## Base URLs

| Service | Base URL |
|---------|----------|
| Web Data APIs | `https://opengraph.io/api/1.1/` |
| Image Generation | `https://opengraph.io/image-agent/` |

**Authentication:** All requests require `app_id` as a query parameter.

```bash
curl "https://opengraph.io/api/1.1/site/{url}?app_id=YOUR_APP_ID"
```

---

## Web Data Endpoints

### GET /site/{url} — Extract OpenGraph Data

The primary endpoint for extracting metadata, OpenGraph tags, and inferred data from any URL.

**URL Format:**
```
GET /site/{encoded_url}?app_id=XXX
```

**Parameters:**

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `app_id` | string | required | Your API key |
| `full_render` | boolean | `false` | Use headless browser for JavaScript-heavy sites |
| `cache_ok` | boolean | `true` | Allow cached responses (faster) |
| `use_proxy` | boolean | `false` | Route through residential proxy |
| `use_premium` | boolean | `false` | Use premium proxy tier |
| `use_superior` | boolean | `false` | Use superior proxy tier |
| `auto_proxy` | boolean | `false` | Auto-escalate to proxy on failure |
| `accept_lang` | string | `en-US,en;q=0.9` | Accept-Language header value |
| `max_cache_age` | int | `432000000` | Max cache age in milliseconds |

**Response Structure:**

```json
{
  "hybridGraph": {
    "title": "Page Title",
    "description": "Meta description or OG description",
    "type": "website",
    "image": "https://example.com/image.jpg",
    "imageSecureUrl": "https://example.com/image.jpg",
    "imageWidth": 1200,
    "imageHeight": 630,
    "url": "https://example.com",
    "favicon": "https://example.com/favicon.ico",
    "site_name": "Example Site",
    
    // Video-specific (YouTube, Vimeo, etc.)
    "video": "https://www.youtube.com/embed/VIDEO_ID",
    "articleLikes": "18769272",
    "articleComments": "2416252", 
    "articleViews": "1738229652",
    "profileUsername": "Channel Name"
  },
  "openGraph": {
    // Raw OpenGraph tags as found on page
    // Returns {"error": "No OpenGraph Tags Found"} if none
  },
  "htmlInferred": {
    // Data inferred from HTML when OG tags missing
    // Same structure as hybridGraph
  },
  "requestInfo": {
    "redirects": 0,
    "host": "https://example.com",
    "responseCode": 200,
    "cache_ok": true,
    "full_render": false,
    "use_proxy": false,
    "requestCount": 1
  },
  "is_cache": true,
  "url": "https://example.com",
  "timestamp": "2026-02-03T12:00:00.000Z"
}
```

**Response Fields Explained:**

| Field | Description |
|-------|-------------|
| `hybridGraph` | Best available data — combines OG tags with inferred data |
| `openGraph` | Raw OpenGraph tags exactly as found on the page |
| `htmlInferred` | Data inferred from HTML elements when OG tags are missing |
| `requestInfo` | Request metadata (response code, cache status, etc.) |

**Example — Basic Request:**
```bash
curl -s "https://opengraph.io/api/1.1/site/$(echo -n 'https://github.com' | jq -sRr @uri)?app_id=${OPENGRAPH_APP_ID}"
```

**Example — JavaScript-heavy site:**
```bash
curl -s "https://opengraph.io/api/1.1/site/$(echo -n 'https://spa-app.com' | jq -sRr @uri)?app_id=${OPENGRAPH_APP_ID}&full_render=true"
```

**Example — Geo-blocked content:**
```bash
curl -s "https://opengraph.io/api/1.1/site/$(echo -n 'https://region-locked.com' | jq -sRr @uri)?app_id=${OPENGRAPH_APP_ID}&use_proxy=true"
```

---

### GET /scrape/{url} — Fetch Rendered HTML

Fetch the full HTML content of a page, optionally with JavaScript rendering.

**URL Format:**
```
GET /scrape/{encoded_url}?app_id=XXX
```

**Parameters:**

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `app_id` | string | required | Your API key |
| `full_render` | boolean | `false` | Render JavaScript before returning HTML |
| `use_proxy` | boolean | `false` | Route through residential proxy |
| `accept_lang` | string | `en-US,en;q=0.9` | Accept-Language header |
| `cache_ok` | boolean | `true` | Allow cached responses |

**Response:**
```json
{
  "html": "<!DOCTYPE html><html>...</html>",
  "requestInfo": { ... }
}
```

**Example:**
```bash
curl -s "https://opengraph.io/api/1.1/scrape/$(echo -n 'https://example.com' | jq -sRr @uri)?app_id=${OPENGRAPH_APP_ID}&full_render=true"
```

---

### GET /screenshot/{url} — Capture Screenshot

Capture a visual screenshot of any webpage.

**URL Format:**
```
GET /screenshot/{encoded_url}?app_id=XXX
```

**Parameters:**

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `app_id` | string | required | Your API key |
| `dimensions` | string | `lg` | Size preset: `sm`, `md`, `lg`, `xl` |
| `width` | int | varies | Custom width in pixels |
| `height` | int | varies | Custom height in pixels |
| `quality` | int | `80` | JPEG quality (1-100) |
| `full_page` | boolean | `false` | Capture full page vs viewport only |
| `use_proxy` | boolean | `false` | Route through proxy |

**Dimension Presets:**

| Preset | Width | Height |
|--------|-------|--------|
| `sm` | 640 | 480 |
| `md` | 800 | 600 |
| `lg` | 1200 | 630 |
| `xl` | 1920 | 1080 |

**Response:**
```json
{
  "screenshotUrl": "https://cdn.opengraph.io/screenshots/abc123.jpg",
  "requestInfo": { ... }
}
```

**Example — OG-sized screenshot:**
```bash
curl -s "https://opengraph.io/api/1.1/screenshot/$(echo -n 'https://example.com' | jq -sRr @uri)?app_id=${OPENGRAPH_APP_ID}&dimensions=lg"
```

**Example — Full page:**
```bash
curl -s "https://opengraph.io/api/1.1/screenshot/$(echo -n 'https://example.com' | jq -sRr @uri)?app_id=${OPENGRAPH_APP_ID}&full_page=true"
```

---

### GET /extract/{url} — Extract HTML Elements

Extract specific HTML elements from a page (headings, paragraphs, links, etc.).

**URL Format:**
```
GET /extract/{encoded_url}?app_id=XXX&html_elements=h1,h2,p
```

**Parameters:**

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `app_id` | string | required | Your API key |
| `html_elements` | string | required | Comma-separated HTML tags to extract |
| `full_render` | boolean | `false` | Render JavaScript first |
| `use_proxy` | boolean | `false` | Route through proxy |

**Supported Elements:**
`h1`, `h2`, `h3`, `h4`, `h5`, `h6`, `p`, `a`, `img`, `li`, `span`, `div`, `article`, `section`, `table`, `tr`, `td`, `th`

**Response:**
```json
{
  "tags": [
    {
      "tag": "h1",
      "innerText": "Main Heading",
      "position": 0,
      "attributes": {
        "class": "title"
      }
    },
    {
      "tag": "h2", 
      "innerText": "Section Heading",
      "position": 1
    },
    {
      "tag": "p",
      "innerText": "Paragraph content...",
      "position": 2
    }
  ],
  "requestInfo": { ... }
}
```

**Example — Extract headings:**
```bash
curl -s "https://opengraph.io/api/1.1/extract/$(echo -n 'https://example.com' | jq -sRr @uri)?app_id=${OPENGRAPH_APP_ID}&html_elements=h1,h2,h3"
```

**Example — Extract links:**
```bash
curl -s "https://opengraph.io/api/1.1/extract/$(echo -n 'https://example.com' | jq -sRr @uri)?app_id=${OPENGRAPH_APP_ID}&html_elements=a"
```

---

### POST /query/{url} — AI-Powered Page Analysis

Ask questions about a webpage using AI. Returns structured answers.

> ⚠️ **Paid feature** — Not available on free tier.

**URL Format:**
```
POST /query/{encoded_url}?app_id=XXX
Content-Type: application/json
```

**Request Body:**

```json
{
  "query": "What services does this company offer?",
  "responseStructure": {
    "type": "object",
    "properties": {
      "services": {
        "type": "array",
        "items": { "type": "string" }
      },
      "pricing": { "type": "string" }
    }
  }
}
```

| Field | Type | Required | Description |
|-------|------|----------|-------------|
| `query` | string | yes | Question to ask about the page |
| `responseStructure` | object | no | JSON Schema for structured response |

**Response:**
```json
{
  "answer": {
    "services": ["Web Development", "API Integration", "Consulting"],
    "pricing": "Contact for quote"
  },
  "confidence": 0.92,
  "requestInfo": { ... }
}
```

**Example:**
```bash
curl -s -X POST "https://opengraph.io/api/1.1/query/$(echo -n 'https://example.com' | jq -sRr @uri)?app_id=${OPENGRAPH_APP_ID}" \
  -H "Content-Type: application/json" \
  -d '{"query": "What is the main product sold on this page?"}'
```

---

## Image Generation Endpoints

### POST /sessions — Create Session

Create a new image generation session.

**Request:**
```json
POST /sessions?app_id=XXX
Content-Type: application/json

{
  "name": "my-project-images"
}
```

**Response:**
```json
{
  "sessionId": "sess_abc123xyz",
  "name": "my-project-images",
  "createdAt": "2026-02-03T12:00:00.000Z"
}
```

---

### POST /sessions/{sessionId}/generate — Generate Image

Generate a new image in a session.

**Request:**
```json
POST /sessions/{sessionId}/generate?app_id=XXX
Content-Type: application/json

{
  "prompt": "Modern tech company logo with abstract geometric shapes",
  "kind": "icon",
  "aspectRatio": "square",
  "quality": "high",
  "stylePreset": "vercel",
  "brandColors": ["#0070F3", "#000000"],
  "transparent": true
}
```

**Parameters:**

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `prompt` | string | required | Image description or diagram code |
| `kind` | string | `illustration` | Image type (see below) |
| `aspectRatio` | string | `square` | Output dimensions (see below) |
| `quality` | string | `medium` | `low`, `medium`, `high`, `fast` |
| `stylePreset` | string | — | Style template (see below) |
| `brandColors` | array | — | Array of hex colors |
| `stylePreferences` | string | — | Additional style guidance |
| `outputStyle` | string | `standard` | `draft`, `standard`, `premium` |
| `transparent` | boolean | `false` | Transparent background |
| `autoCrop` | boolean | `false` | Auto-crop whitespace |

**Image Kinds:**

| Kind | Use Case |
|------|----------|
| `illustration` | General images, marketing assets |
| `diagram` | Technical diagrams, flowcharts |
| `icon` | App icons, logos |
| `social-card` | OG images, Twitter cards |
| `qr-code` | Basic QR codes |

**Aspect Ratios:**

| Preset | Dimensions | Use Case |
|--------|------------|----------|
| `square` | 1024×1024 | General, Instagram |
| `og-image` | 1200×630 | OpenGraph, Facebook |
| `twitter-card` | 1200×600 | Twitter |
| `instagram-story` | 1080×1920 | Stories, Reels |
| `wide` | 1920×1080 | Presentations |
| `portrait` | 1080×1350 | Instagram portrait |
| `icon-large` | 512×512 | App icons |
| `icon-small` | 256×256 | Small icons |

**Style Presets:**

| Preset | Description |
|--------|-------------|
| `github-dark` | Dark mode, developer aesthetic |
| `vercel` | Modern, gradient, sleek |
| `stripe` | Professional, fintech |
| `neon-cyber` | Cyberpunk, glowing effects |
| `pastel` | Soft, friendly colors |
| `minimal-mono` | Black/white, typography focus |

**Diagram-Specific Parameters:**

| Parameter | Type | Description |
|-----------|------|-------------|
| `diagramCode` | string | Mermaid/D2/Vega syntax |
| `diagramFormat` | string | `mermaid`, `d2`, `vega` |
| `diagramTemplate` | string | Template name |

**Response:**
```json
{
  "sessionId": "sess_abc123",
  "assetId": "asset_xyz789",
  "status": "succeeded",
  "url": "/assets/asset_xyz789/file",
  "width": 1024,
  "height": 1024,
  "usage": {
    "totalCostUsd": 0.05
  }
}
```

---

### POST /sessions/{sessionId}/iterate — Refine Image

Modify an existing generated image.

**Request:**
```json
POST /sessions/{sessionId}/iterate?app_id=XXX
Content-Type: application/json

{
  "assetId": "asset_xyz789",
  "prompt": "Change the background to dark blue and add a subtle glow effect"
}
```

**Response:** Same as generate endpoint.

---

### GET /sessions/{sessionId} — Get Session

Retrieve session details and asset history.

**Response:**
```json
{
  "sessionId": "sess_abc123",
  "name": "my-project-images",
  "assets": [
    {
      "assetId": "asset_xyz789",
      "prompt": "...",
      "status": "succeeded",
      "createdAt": "2026-02-03T12:00:00.000Z"
    }
  ],
  "createdAt": "2026-02-03T11:55:00.000Z"
}
```

---

### GET /assets/{assetId}/file — Download Image

Download the generated image file.

**Response:** Binary image data (PNG or JPEG).

**Example:**
```bash
curl -s "https://opengraph.io/image-agent/assets/asset_xyz789/file?app_id=${OPENGRAPH_APP_ID}" -o output.png
```

---

## Error Codes

| Code | HTTP Status | Description |
|------|-------------|-------------|
| -9 | 401 | API key required |
| 101 | 401 | Invalid API key |
| 102 | 429 | Rate limit exceeded |
| 103 | 400 | Invalid URL format |
| 104 | 502 | Target site unreachable |
| 105 | 504 | Request timeout |
| 106 | 400 | Invalid parameters |

**Error Response Format:**
```json
{
  "error": {
    "code": 102,
    "message": "Rate limit exceeded. Please upgrade your plan or wait."
  }
}
```

---

## Rate Limits

Rate limits vary by plan. Check response headers:

```
X-RateLimit-Limit: 100
X-RateLimit-Remaining: 95
X-RateLimit-Reset: 1706547200
```

| Header | Description |
|--------|-------------|
| `X-RateLimit-Limit` | Requests allowed per window |
| `X-RateLimit-Remaining` | Requests remaining |
| `X-RateLimit-Reset` | Unix timestamp when limit resets |

---

## Pricing Tiers

| Feature | Free Tier | Paid Plans |
|---------|-----------|------------|
| Site/Unfurl | 100/month | Based on plan |
| Screenshot | 100/month | Based on plan |
| Scrape | 100/month | Based on plan |
| Extract | 100/month | Based on plan |
| Query (AI) | ❌ | ✅ |
| Image Generation | 4/month | Based on plan |

See [dashboard.opengraph.io](https://dashboard.opengraph.io) for current pricing.
