#!/usr/bin/env python3
"""
Skill Scanner — Scans OpenClaw skills for malicious patterns.

Usage:
    python3 scanner.py                          # Scan all installed skills
    python3 scanner.py --skill <name>           # Scan a specific skill
    python3 scanner.py --file <path>            # Scan a specific file
    python3 scanner.py --pre-install <slug>     # Pre-install scan
    python3 scanner.py --json                   # JSON output
    python3 scanner.py --report <path>          # Generate markdown report

No external dependencies — stdlib only.
"""

import re
import os
import sys
import json
import argparse
import tempfile
import shutil
import subprocess
from pathlib import Path
from datetime import datetime, timezone

# ─── Constants ───────────────────────────────────────────────────────────────

VERSION = "1.0.0"
DEFAULT_SKILLS_DIR = Path.home() / ".openclaw" / "workspace" / "skills"
SCANNER_DIR = Path(__file__).parent.resolve()
WHITELIST_FILE = SCANNER_DIR / "whitelist.json"
REPORT_TEMPLATE = SCANNER_DIR / "report-template.md"
SCANNABLE_EXTENSIONS = {".py", ".sh", ".js", ".ts", ".bash", ".zsh", ".rb", ".pl", ".php", ".md", ".yml", ".yaml", ".json", ".toml", ".cfg", ".ini", ".conf", ".env", ".txt"}

# ─── Colors ──────────────────────────────────────────────────────────────────

class C:
    """ANSI color codes."""
    RED = "\033[91m"
    GREEN = "\033[92m"
    YELLOW = "\033[93m"
    BLUE = "\033[94m"
    MAGENTA = "\033[95m"
    CYAN = "\033[96m"
    WHITE = "\033[97m"
    BOLD = "\033[1m"
    DIM = "\033[2m"
    RESET = "\033[0m"

    @classmethod
    def disable(cls):
        for attr in ("RED", "GREEN", "YELLOW", "BLUE", "MAGENTA", "CYAN", "WHITE", "BOLD", "DIM", "RESET"):
            setattr(cls, attr, "")


# ─── Detection Patterns ─────────────────────────────────────────────────────

# Each pattern: (name, severity, regex, description)
# severity: "critical" (25pts), "high" (15pts), "medium" (8pts), "low" (3pts)

PATTERNS = [
    # === CRITICAL (25 points each) ===
    (
        "base64_exec",
        "critical",
        re.compile(
            r"""(?:eval|exec|bash|sh|python|perl|ruby|system)\s*[\(\s]*['"]?[A-Za-z0-9+/=]{40,}""",
            re.IGNORECASE,
        ),
        "Base64-encoded payload near code execution function",
    ),
    (
        "base64_decode_exec",
        "critical",
        re.compile(
            r"""base64[._](?:b64)?decode\s*\(.*?\).*?(?:exec|eval|system|popen|subprocess)""",
            re.IGNORECASE | re.DOTALL,
        ),
        "Base64 decode piped into code execution",
    ),
    (
        "reverse_shell",
        "critical",
        re.compile(
            r"""(?:bash\s+-i\s+>&|\/dev\/tcp\/|nc\s+-[elp]|ncat\s+-|mkfifo\s+.*?\bsh\b|python.*?socket.*?connect|socat\s+)""",
            re.IGNORECASE,
        ),
        "Reverse shell pattern detected",
    ),
    (
        "curl_pipe_shell",
        "critical",
        re.compile(
            r"""(?:curl|wget)\s+[^\n|]*?\|\s*(?:ba)?sh""",
            re.IGNORECASE,
        ),
        "Remote script piped directly to shell (curl/wget | bash)",
    ),
    (
        "raw_ip_http",
        "high",
        re.compile(
            r"""https?://\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}""",
            re.IGNORECASE,
        ),
        "HTTP connection to raw IP address (no domain)",
    ),

    # === HIGH (15 points each) ===
    (
        "eval_exec_dynamic",
        "high",
        re.compile(
            r"""\beval\s*\(\s*(?!['"](?:true|false|null|undefined)['"]\s*\))""",
            re.IGNORECASE,
        ),
        "Dynamic eval() call — can execute arbitrary code",
    ),
    (
        "os_system_call",
        "high",
        re.compile(
            r"""os\.system\s*\(""",
        ),
        "os.system() call — runs shell commands",
    ),
    (
        "subprocess_shell",
        "high",
        re.compile(
            r"""subprocess\.(?:call|run|Popen|check_output|check_call)\s*\([^)]*shell\s*=\s*True""",
            re.DOTALL,
        ),
        "subprocess with shell=True — command injection risk",
    ),
    (
        "password_archive",
        "high",
        re.compile(
            r"""(?:--password|--passphrase|-p\s+\S+).*?(?:\.zip|\.tar|\.7z|\.rar)|\b(?:zip|tar|7z)\b.*?(?:--password|-p\s)""",
            re.IGNORECASE,
        ),
        "Password-protected archive operation",
    ),
    (
        "webhook_exfil",
        "high",
        re.compile(
            r"""(?:webhook\.site|requestbin\.com|hookbin\.com|pipedream\.net|beeceptor\.com|requestcatcher\.com|burpcollaborator)""",
            re.IGNORECASE,
        ),
        "Known data exfiltration endpoint",
    ),
    (
        "env_exfil",
        "high",
        re.compile(
            r"""(?:open|read|load).*?\.env\b.*?(?:requests?\.|urllib|http|curl|wget|socket|send|post|fetch)""",
            re.IGNORECASE | re.DOTALL,
        ),
        "Reading .env file with network send capability",
    ),
    (
        "api_key_grab",
        "high",
        re.compile(
            r"""(?:os\.environ|process\.env).*?(?:API.?KEY|SECRET|TOKEN|PASSWORD|CREDENTIAL).*?(?:requests?\.|urllib|http|curl|fetch|send|post|socket)""",
            re.IGNORECASE | re.DOTALL,
        ),
        "API key/secret read and potential exfiltration",
    ),

    # === MEDIUM (8 points each) ===
    (
        "obfuscated_url",
        "medium",
        re.compile(
            r"""(?:rentry\.co|pastebin\.com|hastebin\.com|paste\.ee|dpaste\.org|ix\.io|termbin\.com|0x0\.st|transfer\.sh)""",
            re.IGNORECASE,
        ),
        "Obfuscated URL / paste service (potential redirector)",
    ),
    (
        "fake_dependency",
        "medium",
        re.compile(
            r"""(?:openclaw-core|openclaw-sdk|openclaw-utils|openclaw-lib|claw-internal|clawhub-api)""",
            re.IGNORECASE,
        ),
        "Reference to non-existent / fake dependency",
    ),
    (
        "hidden_file_create",
        "medium",
        re.compile(
            r"""(?:open|write|create|touch|mkdir)\s*[\(]?\s*['"]\.(?!env|gitignore|gitkeep|dockerignore|eslintrc|prettierrc|npmrc|editorconfig)\w+""",
            re.IGNORECASE,
        ),
        "Hidden dotfile creation (potential persistence)",
    ),
    (
        "crypto_mining",
        "medium",
        re.compile(
            r"""(?:xmrig|stratum\+tcp|coinhive|cryptonight|monero.*?pool|minergate|nicehash|ethermine|f2pool|antpool|hashrate)""",
            re.IGNORECASE,
        ),
        "Crypto mining indicator",
    ),
    (
        "chmod_executable",
        "medium",
        re.compile(
            r"""chmod\s+[+0-7]*[xX7]""",
        ),
        "Making file executable (chmod +x)",
    ),
    (
        "cron_persistence",
        "medium",
        re.compile(
            r"""crontab\s|/etc/cron|systemctl\s+enable|\.service\s""",
            re.IGNORECASE,
        ),
        "Persistence mechanism (cron/systemd)",
    ),
    (
        "dns_exfil",
        "medium",
        re.compile(
            r"""(?:nslookup|dig)\s+.*?\$|socket\.getaddrinfo.*?\bvar\b""",
            re.IGNORECASE,
        ),
        "Potential DNS exfiltration",
    ),
    (
        "encoded_string_long",
        "medium",
        re.compile(
            r"""['"]\s*[A-Za-z0-9+/]{80,}={0,2}\s*['"]""",
        ),
        "Long base64-encoded string (potential hidden payload)",
    ),

    # === LOW (3 points each) ===
    (
        "exec_function",
        "low",
        re.compile(
            r"""\bexec\s*\(""",
        ),
        "exec() function call",
    ),
    (
        "subprocess_basic",
        "low",
        re.compile(
            r"""subprocess\.(?:call|run|Popen|check_output|check_call)\s*\(""",
        ),
        "subprocess usage (review arguments)",
    ),
    (
        "network_request",
        "low",
        re.compile(
            r"""(?:requests?\.(?:get|post|put|delete|patch)|urllib\.request|http\.client|fetch\s*\()""",
            re.IGNORECASE,
        ),
        "Network request (verify destination)",
    ),
    (
        "file_write",
        "low",
        re.compile(
            r"""open\s*\([^)]*['\"]w['\"]""",
        ),
        "File write operation",
    ),
    (
        "env_access",
        "low",
        re.compile(
            r"""os\.environ\.get|os\.getenv|process\.env\.""",
        ),
        "Environment variable access",
    ),
    (
        "hardcoded_ip",
        "low",
        re.compile(
            r"""\b(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\.(?:25[0-5]|2[0-4]\d|[01]?\d\d?)\b""",
        ),
        "Hardcoded IP address",
    ),
]

SEVERITY_SCORES = {
    "critical": 25,
    "high": 15,
    "medium": 8,
    "low": 3,
}

SEVERITY_COLORS = {
    "critical": C.RED,
    "high": C.RED,
    "medium": C.YELLOW,
    "low": C.CYAN,
}

SEVERITY_ICONS = {
    "critical": "🔴",
    "high": "🟠",
    "medium": "🟡",
    "low": "🔵",
}


# ─── Whitelist / Blacklist ───────────────────────────────────────────────────

def load_lists():
    """Load whitelist and blacklist from whitelist.json."""
    if not WHITELIST_FILE.exists():
        return {}, {}
    with open(WHITELIST_FILE) as f:
        data = json.load(f)
    whitelist = {item["slug"]: item.get("reason", "") for item in data.get("whitelisted", [])}
    blacklist = {item["slug"]: item for item in data.get("blacklisted", [])}
    return whitelist, blacklist


# ─── Scanner Core ────────────────────────────────────────────────────────────

class Finding:
    """A single finding from scanning."""
    def __init__(self, pattern_name, severity, description, file_path, line_num, line_text, match_text):
        self.pattern_name = pattern_name
        self.severity = severity
        self.description = description
        self.file_path = file_path
        self.line_num = line_num
        self.line_text = line_text.strip()
        self.match_text = match_text.strip()[:120]
        self.score = SEVERITY_SCORES.get(severity, 3)

    def to_dict(self):
        return {
            "pattern": self.pattern_name,
            "severity": self.severity,
            "description": self.description,
            "file": str(self.file_path),
            "line": self.line_num,
            "line_text": self.line_text[:200],
            "match": self.match_text,
            "score": self.score,
        }


class SkillReport:
    """Scan results for a single skill."""
    def __init__(self, skill_name, skill_path):
        self.skill_name = skill_name
        self.skill_path = skill_path
        self.findings = []
        self.files_scanned = 0
        self.is_whitelisted = False
        self.whitelist_reason = ""
        self.is_blacklisted = False
        self.blacklist_info = {}

    @property
    def risk_score(self):
        if self.is_blacklisted:
            return 100
        raw = sum(f.score for f in self.findings)
        return min(100, raw)

    @property
    def status(self):
        score = self.risk_score
        if score < 30:
            return "clean"
        elif score < 70:
            return "suspicious"
        else:
            return "dangerous"

    @property
    def status_emoji(self):
        return {"clean": "🟢", "suspicious": "🟡", "dangerous": "🔴"}[self.status]

    @property
    def status_color(self):
        return {"clean": C.GREEN, "suspicious": C.YELLOW, "dangerous": C.RED}[self.status]

    def to_dict(self):
        return {
            "skill": self.skill_name,
            "path": str(self.skill_path),
            "risk_score": self.risk_score,
            "status": self.status,
            "files_scanned": self.files_scanned,
            "is_whitelisted": self.is_whitelisted,
            "whitelist_reason": self.whitelist_reason,
            "is_blacklisted": self.is_blacklisted,
            "blacklist_info": self.blacklist_info,
            "findings": [f.to_dict() for f in self.findings],
            "finding_count": len(self.findings),
            "critical_count": len([f for f in self.findings if f.severity == "critical"]),
            "high_count": len([f for f in self.findings if f.severity == "high"]),
            "medium_count": len([f for f in self.findings if f.severity == "medium"]),
            "low_count": len([f for f in self.findings if f.severity == "low"]),
        }


def _is_self(file_path):
    """Check if this file is part of skill-scanner itself."""
    try:
        return file_path.resolve().is_relative_to(SCANNER_DIR)
    except (AttributeError, TypeError):
        # Python < 3.9 fallback
        try:
            file_path.resolve().relative_to(SCANNER_DIR)
            return True
        except ValueError:
            return False


def _is_doc_line(file_path, stripped, severity):
    """Check if a line is documentation/comment that shouldn't be flagged (unless critical)."""
    suffix = file_path.suffix.lower() if hasattr(file_path, 'suffix') else ""
    name = file_path.name if hasattr(file_path, 'name') else str(file_path)

    # In markdown/doc files, skip table rows, list items, headers, and code block fences
    if suffix in (".md", ".txt", ".rst") or name in ("README", "CHANGELOG", "LICENSE"):
        if stripped.startswith(("|", "- ", "* ", "#", "```", "> ")):
            return True
        # Skip lines that are clearly documentation examples
        if severity not in ("critical",):
            return True  # In markdown, only flag critical patterns

    # In JSON data files, skip string values (but still flag critical)
    if suffix == ".json" and severity not in ("critical", "high"):
        return True

    # In any file, skip pure comment lines for non-critical
    if severity not in ("critical", "high"):
        if stripped.startswith(("#", "//", "/*", "*", "<!--")):
            return True

    return False


def scan_file(file_path, findings_list, relative_to=None):
    """Scan a single file for suspicious patterns."""
    try:
        with open(file_path, "r", errors="ignore") as f:
            content = f.read()
            lines = content.split("\n")
    except (IOError, OSError):
        return 0

    display_path = file_path
    if relative_to:
        try:
            display_path = file_path.relative_to(relative_to)
        except ValueError:
            pass

    line_count = 0
    for line_num, line in enumerate(lines, 1):
        # Skip comments in some cases (but still scan — malware hides in comments too)
        for pattern_name, severity, regex, description in PATTERNS:
            match = regex.search(line)
            if match:
                # Avoid self-detection: skip if file is in skill-scanner
                if _is_self(file_path):
                    continue
                # Skip markdown documentation lines (tables, list items, headers, code block markers)
                stripped = line.strip()
                if _is_doc_line(file_path, stripped, severity):
                    continue
                findings_list.append(Finding(
                    pattern_name=pattern_name,
                    severity=severity,
                    description=description,
                    file_path=display_path,
                    line_num=line_num,
                    line_text=line,
                    match_text=match.group(0),
                ))
                line_count += 1

    # Also scan multi-line patterns across the whole content
    for pattern_name, severity, regex, description in PATTERNS:
        if "DOTALL" in str(regex.flags) or regex.flags & re.DOTALL:
            for match in regex.finditer(content):
                # Find which line the match starts on
                start = match.start()
                line_num = content[:start].count("\n") + 1
                line_text = lines[line_num - 1] if line_num <= len(lines) else ""

                if _is_self(file_path):
                    continue

                # Skip documentation lines for multi-line matches too
                stripped_ml = line_text.strip()
                if _is_doc_line(file_path, stripped_ml, severity):
                    continue

                # Check for duplicates
                already_found = False
                for existing in findings_list:
                    if (existing.pattern_name == pattern_name and
                        existing.file_path == display_path and
                        existing.line_num == line_num):
                        already_found = True
                        break
                if not already_found:
                    findings_list.append(Finding(
                        pattern_name=pattern_name,
                        severity=severity,
                        description=description,
                        file_path=display_path,
                        line_num=line_num,
                        line_text=line_text,
                        match_text=match.group(0)[:120],
                    ))

    return 1  # scanned 1 file


def scan_skill(skill_path, whitelist, blacklist):
    """Scan an entire skill directory."""
    skill_path = Path(skill_path)
    skill_name = skill_path.name

    report = SkillReport(skill_name, skill_path)

    # Check blacklist
    if skill_name in blacklist:
        report.is_blacklisted = True
        report.blacklist_info = blacklist[skill_name]
        return report

    # Check whitelist
    if skill_name in whitelist:
        report.is_whitelisted = True
        report.whitelist_reason = whitelist[skill_name]

    # Scan all scannable files
    for root, dirs, files in os.walk(skill_path):
        # Skip hidden directories and common non-code dirs
        dirs[:] = [d for d in dirs if not d.startswith(".") and d not in ("node_modules", "__pycache__", ".git", "venv")]
        for filename in files:
            fp = Path(root) / filename
            suffix = fp.suffix.lower()
            # Scan known extensions + extensionless files (shell scripts)
            if suffix in SCANNABLE_EXTENSIONS or suffix == "" or filename in ("Makefile", "Dockerfile", "Vagrantfile"):
                scanned = scan_file(fp, report.findings, relative_to=skill_path)
                report.files_scanned += scanned

    return report


def scan_single_file(file_path):
    """Scan a single file and return a pseudo-report."""
    file_path = Path(file_path).resolve()
    report = SkillReport(file_path.name, file_path.parent)
    scanned = scan_file(file_path, report.findings)
    report.files_scanned = scanned
    return report


# ─── Output Formatting ──────────────────────────────────────────────────────

def print_banner():
    print(f"""
{C.CYAN}{C.BOLD}╔══════════════════════════════════════════════════╗
║          🦞 Skill Scanner v{VERSION}               ║
║       OpenClaw Security Analysis Tool            ║
╚══════════════════════════════════════════════════╝{C.RESET}
""")


def print_report(report, verbose=True):
    """Print a colored terminal report for a skill."""
    sc = report.status_color
    emoji = report.status_emoji

    # Header
    label = report.status.upper()
    extra = ""
    if report.is_whitelisted:
        extra = f" {C.DIM}(whitelisted: {report.whitelist_reason}){C.RESET}"
    if report.is_blacklisted:
        extra = f" {C.RED}{C.BOLD}⚠ BLACKLISTED: {report.blacklist_info.get('reason', 'Known malicious')}{C.RESET}"

    print(f"{C.BOLD}{'─' * 55}{C.RESET}")
    print(f"{emoji} {C.BOLD}{report.skill_name}{C.RESET}  "
          f"Score: {sc}{C.BOLD}{report.risk_score}/100{C.RESET}  "
          f"Status: {sc}{label}{C.RESET}{extra}")
    print(f"   {C.DIM}Files scanned: {report.files_scanned} | Findings: {len(report.findings)}{C.RESET}")

    if report.is_blacklisted:
        reporter = report.blacklist_info.get("reporter", "unknown")
        print(f"   {C.RED}Reported by: {reporter}{C.RESET}")
        print()
        return

    if not report.findings:
        print(f"   {C.GREEN}✓ No suspicious patterns found{C.RESET}")
        print()
        return

    if verbose:
        # Group findings by severity
        for sev in ("critical", "high", "medium", "low"):
            sev_findings = [f for f in report.findings if f.severity == sev]
            if not sev_findings:
                continue
            color = SEVERITY_COLORS[sev]
            icon = SEVERITY_ICONS[sev]
            print(f"\n   {color}{C.BOLD}{icon} {sev.upper()} ({len(sev_findings)}):{C.RESET}")
            for f in sev_findings:
                print(f"   {color}  → {f.description}{C.RESET}")
                print(f"     {C.DIM}{f.file_path}:{f.line_num}{C.RESET}")
                # Truncate long lines
                display_line = f.line_text[:100]
                if len(f.line_text) > 100:
                    display_line += "..."
                print(f"     {C.DIM}  {display_line}{C.RESET}")
    else:
        # Compact output
        crit = len([f for f in report.findings if f.severity == "critical"])
        high = len([f for f in report.findings if f.severity == "high"])
        med = len([f for f in report.findings if f.severity == "medium"])
        low = len([f for f in report.findings if f.severity == "low"])
        parts = []
        if crit:
            parts.append(f"{C.RED}{crit} critical{C.RESET}")
        if high:
            parts.append(f"{C.RED}{high} high{C.RESET}")
        if med:
            parts.append(f"{C.YELLOW}{med} medium{C.RESET}")
        if low:
            parts.append(f"{C.CYAN}{low} low{C.RESET}")
        print(f"   Findings: {', '.join(parts)}")

    print()


def print_summary(reports):
    """Print overall scan summary."""
    total = len(reports)
    clean = len([r for r in reports if r.status == "clean"])
    suspicious = len([r for r in reports if r.status == "suspicious"])
    dangerous = len([r for r in reports if r.status == "dangerous"])
    total_findings = sum(len(r.findings) for r in reports)
    total_files = sum(r.files_scanned for r in reports)

    print(f"{C.BOLD}{'═' * 55}{C.RESET}")
    print(f"{C.BOLD}📊 SCAN SUMMARY{C.RESET}")
    print(f"{C.BOLD}{'═' * 55}{C.RESET}")
    print(f"  Skills scanned:  {total}")
    print(f"  Files scanned:   {total_files}")
    print(f"  Total findings:  {total_findings}")
    print()
    print(f"  {C.GREEN}🟢 Clean:      {clean}{C.RESET}")
    print(f"  {C.YELLOW}🟡 Suspicious: {suspicious}{C.RESET}")
    print(f"  {C.RED}🔴 Dangerous:  {dangerous}{C.RESET}")
    print(f"{C.BOLD}{'═' * 55}{C.RESET}")

    if dangerous > 0:
        print(f"\n{C.RED}{C.BOLD}⚠  {dangerous} DANGEROUS SKILL(S) DETECTED — review immediately!{C.RESET}")
    elif suspicious > 0:
        print(f"\n{C.YELLOW}⚠  {suspicious} suspicious skill(s) — review recommended.{C.RESET}")
    else:
        print(f"\n{C.GREEN}✓ All skills look clean!{C.RESET}")
    print()


def generate_markdown_report(reports):
    """Generate a markdown report string."""
    now = datetime.now(timezone.utc).strftime("%Y-%m-%d %H:%M UTC")
    total_files = sum(r.files_scanned for r in reports)
    total_findings = sum(len(r.findings) for r in reports)
    clean = len([r for r in reports if r.status == "clean"])
    suspicious = len([r for r in reports if r.status == "suspicious"])
    dangerous = len([r for r in reports if r.status == "dangerous"])

    lines = [
        f"# 🔍 Skill Scanner Report",
        f"",
        f"**Scan Date:** {now}",
        f"**Scanner Version:** {VERSION}",
        f"",
        f"---",
        f"",
        f"## Summary",
        f"",
        f"| Metric | Value |",
        f"|---|---|",
        f"| Skills Scanned | {len(reports)} |",
        f"| Files Scanned | {total_files} |",
        f"| Total Findings | {total_findings} |",
        f"| 🟢 Clean | {clean} |",
        f"| 🟡 Suspicious | {suspicious} |",
        f"| 🔴 Dangerous | {dangerous} |",
        f"",
        f"---",
        f"",
        f"## Results",
        f"",
    ]

    for report in sorted(reports, key=lambda r: -r.risk_score):
        emoji = report.status_emoji
        lines.append(f"### {emoji} {report.skill_name} — Score: {report.risk_score}/100 ({report.status.upper()})")
        lines.append(f"")

        if report.is_blacklisted:
            lines.append(f"**⚠ BLACKLISTED:** {report.blacklist_info.get('reason', 'Known malicious')}")
            lines.append(f"")
            continue

        if report.is_whitelisted:
            lines.append(f"*Whitelisted: {report.whitelist_reason}*")
            lines.append(f"")

        lines.append(f"Files scanned: {report.files_scanned}")
        lines.append(f"")

        if not report.findings:
            lines.append(f"✅ No suspicious patterns found.")
            lines.append(f"")
            continue

        lines.append(f"| Severity | Pattern | File | Line | Description |")
        lines.append(f"|---|---|---|---|---|")
        for f in sorted(report.findings, key=lambda x: ("critical", "high", "medium", "low").index(x.severity)):
            lines.append(f"| {f.severity.upper()} | `{f.pattern_name}` | `{f.file_path}` | {f.line_num} | {f.description} |")
        lines.append(f"")

    lines.append(f"---")
    lines.append(f"*Generated by skill-scanner v{VERSION}*")
    return "\n".join(lines)


def output_json(reports):
    """Output results as JSON."""
    data = {
        "scanner_version": VERSION,
        "scan_date": datetime.now(timezone.utc).isoformat(),
        "summary": {
            "total_skills": len(reports),
            "total_files": sum(r.files_scanned for r in reports),
            "total_findings": sum(len(r.findings) for r in reports),
            "clean": len([r for r in reports if r.status == "clean"]),
            "suspicious": len([r for r in reports if r.status == "suspicious"]),
            "dangerous": len([r for r in reports if r.status == "dangerous"]),
        },
        "skills": [r.to_dict() for r in reports],
    }
    print(json.dumps(data, indent=2))


# ─── Pre-install Scan ────────────────────────────────────────────────────────

def pre_install_scan(slug, json_output=False):
    """Download a skill to temp dir, scan it, report, cleanup."""
    print(f"{C.CYAN}📦 Pre-install scan for: {slug}{C.RESET}")
    print(f"{C.DIM}   Downloading to temporary directory...{C.RESET}")

    whitelist, blacklist = load_lists()

    # Check blacklist immediately
    if slug in blacklist:
        info = blacklist[slug]
        print(f"\n{C.RED}{C.BOLD}🚫 BLOCKED — '{slug}' is BLACKLISTED{C.RESET}")
        print(f"{C.RED}   Reason: {info.get('reason', 'Known malicious')}{C.RESET}")
        print(f"{C.RED}   Reporter: {info.get('reporter', 'unknown')}{C.RESET}")
        print(f"\n{C.RED}   This skill will NOT be installed.{C.RESET}\n")
        return None

    # Create temp directory and attempt install
    tmpdir = tempfile.mkdtemp(prefix="skill-scan-")
    skill_path = Path(tmpdir) / slug

    try:
        # Try to use openclaw to install to temp
        result = subprocess.run(
            ["openclaw", "hub", "install", slug, "--dir", tmpdir],
            capture_output=True, text=True, timeout=60
        )

        if not skill_path.exists():
            # Maybe installed with different name or flat
            contents = list(Path(tmpdir).iterdir())
            if contents:
                skill_path = contents[0] if contents[0].is_dir() else Path(tmpdir)
            else:
                print(f"{C.RED}✗ Failed to download skill '{slug}'{C.RESET}")
                print(f"{C.DIM}   {result.stderr or result.stdout}{C.RESET}")
                return None

        report = scan_skill(skill_path, whitelist, blacklist)

        if json_output:
            output_json([report])
        else:
            print_banner()
            print(f"{C.CYAN}Pre-install scan results:{C.RESET}\n")
            print_report(report)
            print_summary([report])

        return report

    except FileNotFoundError:
        print(f"{C.RED}✗ 'openclaw' command not found. Cannot pre-install scan.{C.RESET}")
        print(f"{C.DIM}   Install OpenClaw CLI or manually place the skill in the temp dir.{C.RESET}")
        return None
    except subprocess.TimeoutExpired:
        print(f"{C.RED}✗ Download timed out for '{slug}'{C.RESET}")
        return None
    finally:
        # Cleanup
        shutil.rmtree(tmpdir, ignore_errors=True)


# ─── Main ────────────────────────────────────────────────────────────────────

def main():
    parser = argparse.ArgumentParser(
        description="🦞 Skill Scanner — Scan OpenClaw skills for malicious patterns",
        formatter_class=argparse.RawDescriptionHelpFormatter,
    )
    parser.add_argument("--skill", "-s", help="Scan a specific skill by name")
    parser.add_argument("--file", "-f", help="Scan a specific file")
    parser.add_argument("--pre-install", "-p", dest="pre_install", help="Pre-install scan a ClawHub slug")
    parser.add_argument("--json", "-j", action="store_true", help="Output results as JSON")
    parser.add_argument("--report", "-r", help="Save markdown report to file")
    parser.add_argument("--dir", "-d", default=str(DEFAULT_SKILLS_DIR), help="Skills directory to scan")
    parser.add_argument("--no-color", action="store_true", help="Disable colored output")
    parser.add_argument("--verbose", "-v", action="store_true", default=True, help="Verbose output (default)")
    parser.add_argument("--compact", "-c", action="store_true", help="Compact output")

    args = parser.parse_args()

    if args.no_color or args.json:
        C.disable()

    verbose = not args.compact

    # Pre-install scan mode
    if args.pre_install:
        report = pre_install_scan(args.pre_install, args.json)
        if report and args.report:
            md = generate_markdown_report([report])
            with open(args.report, "w") as f:
                f.write(md)
            print(f"{C.GREEN}Report saved to {args.report}{C.RESET}")
        sys.exit(0 if (report and report.risk_score < 70) else 1)

    # Single file scan mode
    if args.file:
        file_path = Path(args.file).resolve()
        if not file_path.exists():
            print(f"{C.RED}✗ File not found: {args.file}{C.RESET}")
            sys.exit(1)
        report = scan_single_file(file_path)
        if args.json:
            output_json([report])
        else:
            print_banner()
            print_report(report, verbose)
            print_summary([report])
        sys.exit(0 if report.risk_score < 70 else 1)

    # Load whitelist/blacklist
    whitelist, blacklist = load_lists()
    skills_dir = Path(args.dir)

    if not skills_dir.exists():
        print(f"{C.RED}✗ Skills directory not found: {skills_dir}{C.RESET}")
        sys.exit(1)

    # Specific skill scan
    if args.skill:
        skill_path = skills_dir / args.skill
        if not skill_path.exists():
            print(f"{C.RED}✗ Skill not found: {args.skill}{C.RESET}")
            sys.exit(1)
        reports = [scan_skill(skill_path, whitelist, blacklist)]
    else:
        # Scan all skills
        reports = []
        for item in sorted(skills_dir.iterdir()):
            if item.is_dir() and not item.name.startswith("."):
                reports.append(scan_skill(item, whitelist, blacklist))

    if not reports:
        print(f"{C.YELLOW}No skills found to scan.{C.RESET}")
        sys.exit(0)

    # Output
    if args.json:
        output_json(reports)
    else:
        print_banner()
        for report in sorted(reports, key=lambda r: -r.risk_score):
            print_report(report, verbose)
        print_summary(reports)

    # Save markdown report if requested
    if args.report:
        md = generate_markdown_report(reports)
        with open(args.report, "w") as f:
            f.write(md)
        print(f"{C.GREEN}📄 Report saved to {args.report}{C.RESET}")

    # Exit code based on worst result
    worst = max(r.risk_score for r in reports) if reports else 0
    sys.exit(0 if worst < 70 else 1)


if __name__ == "__main__":
    main()
