#!/usr/bin/env bash
#
# install-hook.sh — Safe skill installation with pre-scan
#
# Usage:
#   bash install-hook.sh <clawhub-slug> [--force]
#
# Installs a ClawHub skill to a temp directory, scans it for malicious
# patterns, then either installs, warns, or blocks based on the risk score.
#

set -euo pipefail

# ─── Config ──────────────────────────────────────────────────────────────────

SCANNER_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
SCANNER="${SCANNER_DIR}/scanner.py"
WHITELIST="${SCANNER_DIR}/whitelist.json"
SKILLS_DIR="${HOME}/.openclaw/workspace/skills"

RED='\033[91m'
GREEN='\033[92m'
YELLOW='\033[93m'
CYAN='\033[96m'
BOLD='\033[1m'
DIM='\033[2m'
RESET='\033[0m'

# ─── Argument Parsing ────────────────────────────────────────────────────────

SLUG=""
FORCE=false

for arg in "$@"; do
    case "$arg" in
        --force|-f)
            FORCE=true
            ;;
        --help|-h)
            echo "Usage: bash install-hook.sh <clawhub-slug> [--force]"
            echo ""
            echo "Safely installs a ClawHub skill with pre-installation security scan."
            echo ""
            echo "Options:"
            echo "  --force, -f    Force install even if dangerous (score > 70)"
            echo "  --help, -h     Show this help"
            echo ""
            echo "Risk thresholds:"
            echo "  Score < 30:   Auto-install (clean)"
            echo "  Score 30-70:  Show warnings, ask for confirmation"
            echo "  Score > 70:   Block installation (use --force to override)"
            exit 0
            ;;
        *)
            if [ -z "$SLUG" ]; then
                SLUG="$arg"
            else
                echo -e "${RED}Error: Unexpected argument: $arg${RESET}"
                exit 1
            fi
            ;;
    esac
done

if [ -z "$SLUG" ]; then
    echo -e "${RED}Error: No skill slug provided${RESET}"
    echo "Usage: bash install-hook.sh <clawhub-slug> [--force]"
    exit 1
fi

# ─── Check Prerequisites ────────────────────────────────────────────────────

if [ ! -f "$SCANNER" ]; then
    echo -e "${RED}Error: scanner.py not found at ${SCANNER}${RESET}"
    exit 1
fi

if ! command -v python3 &>/dev/null; then
    echo -e "${RED}Error: python3 not found${RESET}"
    exit 1
fi

# ─── Check Blacklist First ──────────────────────────────────────────────────

if [ -f "$WHITELIST" ]; then
    BLACKLISTED=$(python3 -c "
import json, sys
with open('${WHITELIST}') as f:
    data = json.load(f)
for item in data.get('blacklisted', []):
    if item['slug'] == '${SLUG}':
        print(item.get('reason', 'Known malicious'))
        sys.exit(0)
" 2>/dev/null || true)

    if [ -n "$BLACKLISTED" ]; then
        echo ""
        echo -e "${RED}${BOLD}╔══════════════════════════════════════════════════╗${RESET}"
        echo -e "${RED}${BOLD}║       🚫 INSTALLATION BLOCKED — BLACKLISTED     ║${RESET}"
        echo -e "${RED}${BOLD}╚══════════════════════════════════════════════════╝${RESET}"
        echo ""
        echo -e "${RED}  Skill:  ${SLUG}${RESET}"
        echo -e "${RED}  Reason: ${BLACKLISTED}${RESET}"
        echo ""

        if [ "$FORCE" = true ]; then
            echo -e "${YELLOW}${BOLD}⚠  --force specified, but blacklisted skills cannot be force-installed.${RESET}"
            echo -e "${YELLOW}  Remove it from whitelist.json blacklist first if you believe this is wrong.${RESET}"
        fi
        echo ""
        exit 1
    fi
fi

# ─── Create Temp Directory ──────────────────────────────────────────────────

TMPDIR=$(mktemp -d "/tmp/skill-scan-XXXXXX")
trap "rm -rf '$TMPDIR'" EXIT

echo ""
echo -e "${CYAN}${BOLD}🔍 Skill Scanner — Safe Install${RESET}"
echo -e "${DIM}────────────────────────────────────────${RESET}"
echo -e "${CYAN}  Slug: ${SLUG}${RESET}"
echo ""

# ─── Download Skill ─────────────────────────────────────────────────────────

echo -e "${DIM}📦 Downloading skill to temp directory...${RESET}"

SKILL_TMPDIR="${TMPDIR}/${SLUG}"
mkdir -p "$SKILL_TMPDIR"

# Try openclaw hub install
if command -v openclaw &>/dev/null; then
    if ! openclaw hub install "$SLUG" --dir "$TMPDIR" 2>/dev/null; then
        echo -e "${RED}✗ Failed to download skill '${SLUG}' via openclaw hub${RESET}"
        exit 1
    fi
else
    echo -e "${RED}✗ 'openclaw' CLI not found. Cannot download from ClawHub.${RESET}"
    echo -e "${DIM}  Install OpenClaw or manually place skill files in ${SKILLS_DIR}/${SLUG}/${RESET}"
    exit 1
fi

# Find the actual skill directory (might be nested)
if [ ! -d "$SKILL_TMPDIR" ]; then
    # Try to find it
    FOUND=$(find "$TMPDIR" -maxdepth 2 -name "SKILL.md" -type f 2>/dev/null | head -1)
    if [ -n "$FOUND" ]; then
        SKILL_TMPDIR=$(dirname "$FOUND")
    else
        # Just use whatever was downloaded
        DIRS=$(find "$TMPDIR" -maxdepth 1 -type d ! -path "$TMPDIR" | head -1)
        if [ -n "$DIRS" ]; then
            SKILL_TMPDIR="$DIRS"
        fi
    fi
fi

echo -e "${GREEN}✓ Downloaded${RESET}"
echo ""

# ─── Run Scanner ────────────────────────────────────────────────────────────

echo -e "${DIM}🔬 Scanning for malicious patterns...${RESET}"
echo ""

# Get JSON results for programmatic use
JSON_RESULT=$(python3 "$SCANNER" --file "$SKILL_TMPDIR" --json 2>/dev/null || true)

# Also run with directory scan for better coverage
SCAN_OUTPUT=$(python3 "$SCANNER" --dir "$TMPDIR" --skill "$(basename "$SKILL_TMPDIR")" 2>&1 || true)

# Get risk score
RISK_SCORE=$(echo "$JSON_RESULT" | python3 -c "
import json, sys
try:
    data = json.load(sys.stdin)
    skills = data.get('skills', [])
    if skills:
        print(skills[0].get('risk_score', 0))
    else:
        print(0)
except:
    print(0)
" 2>/dev/null || echo "0")

# If JSON failed, try scanning the directory directly
if [ "$RISK_SCORE" = "0" ] || [ -z "$RISK_SCORE" ]; then
    # Run full directory scan
    JSON_RESULT=$(python3 "$SCANNER" --dir "$TMPDIR" --json 2>/dev/null || true)
    RISK_SCORE=$(echo "$JSON_RESULT" | python3 -c "
import json, sys
try:
    data = json.load(sys.stdin)
    skills = data.get('skills', [])
    max_score = 0
    for s in skills:
        score = s.get('risk_score', 0)
        if score > max_score:
            max_score = score
    print(max_score)
except:
    print(0)
" 2>/dev/null || echo "0")
fi

# Print the human-readable scan output
echo "$SCAN_OUTPUT"

# ─── Decision Logic ─────────────────────────────────────────────────────────

RISK_SCORE=${RISK_SCORE:-0}

echo -e "${DIM}────────────────────────────────────────${RESET}"
echo -e "  Risk Score: ${BOLD}${RISK_SCORE}/100${RESET}"
echo ""

if [ "$RISK_SCORE" -lt 30 ]; then
    # Clean — auto install
    echo -e "${GREEN}${BOLD}✓ CLEAN — Installing skill...${RESET}"

    DEST="${SKILLS_DIR}/${SLUG}"
    if [ -d "$DEST" ]; then
        echo -e "${YELLOW}  Skill already exists at ${DEST}. Replacing...${RESET}"
        rm -rf "$DEST"
    fi

    cp -r "$SKILL_TMPDIR" "$DEST"
    echo -e "${GREEN}${BOLD}✓ Installed '${SLUG}' to ${DEST}${RESET}"
    echo ""
    exit 0

elif [ "$RISK_SCORE" -lt 70 ]; then
    # Suspicious — ask for confirmation
    echo -e "${YELLOW}${BOLD}⚠  SUSPICIOUS — This skill has some concerning patterns.${RESET}"
    echo ""

    if [ "$FORCE" = true ]; then
        echo -e "${YELLOW}  --force specified. Installing despite warnings...${RESET}"
    else
        echo -e "${YELLOW}  Review the findings above. Do you want to install anyway?${RESET}"
        echo ""
        read -r -p "  Install '${SLUG}'? [y/N]: " CONFIRM
        if [[ ! "$CONFIRM" =~ ^[Yy]$ ]]; then
            echo -e "${RED}  Installation cancelled.${RESET}"
            echo ""
            exit 1
        fi
    fi

    DEST="${SKILLS_DIR}/${SLUG}"
    if [ -d "$DEST" ]; then
        echo -e "${YELLOW}  Skill already exists at ${DEST}. Replacing...${RESET}"
        rm -rf "$DEST"
    fi

    cp -r "$SKILL_TMPDIR" "$DEST"
    echo -e "${GREEN}${BOLD}✓ Installed '${SLUG}' to ${DEST}${RESET}"
    echo -e "${YELLOW}  ⚠  Monitor this skill's behavior.${RESET}"
    echo ""
    exit 0

else
    # Dangerous — block
    echo -e "${RED}${BOLD}╔══════════════════════════════════════════════════╗${RESET}"
    echo -e "${RED}${BOLD}║       🚫 INSTALLATION BLOCKED — DANGEROUS       ║${RESET}"
    echo -e "${RED}${BOLD}╚══════════════════════════════════════════════════╝${RESET}"
    echo ""
    echo -e "${RED}  Risk Score: ${RISK_SCORE}/100${RESET}"
    echo -e "${RED}  This skill contains patterns commonly found in malware.${RESET}"
    echo ""

    if [ "$FORCE" = true ]; then
        echo -e "${RED}${BOLD}  ⚠  --force specified. Installing DANGEROUS skill...${RESET}"
        echo -e "${RED}  YOU HAVE BEEN WARNED.${RESET}"
        echo ""

        DEST="${SKILLS_DIR}/${SLUG}"
        if [ -d "$DEST" ]; then
            rm -rf "$DEST"
        fi

        cp -r "$SKILL_TMPDIR" "$DEST"
        echo -e "${YELLOW}  Installed '${SLUG}' to ${DEST}${RESET}"
        echo -e "${RED}${BOLD}  ⚠  THIS SKILL MAY BE MALICIOUS. USE AT YOUR OWN RISK.${RESET}"
        echo ""
        exit 0
    else
        echo -e "${RED}  Use --force to override: bash install-hook.sh ${SLUG} --force${RESET}"
        echo ""
        exit 1
    fi
fi
