#!/usr/bin/env bash
set -uo pipefail
# Note: -e omitted intentionally — grep returns 1 on no match, which is expected behavior here

# =============================================================================
# scan-skills.sh — Comprehensive skill file scanner for OpenClaw
# Detects prompt injection, data exfiltration, suspicious patterns, and more.
# =============================================================================

# --- Colors ---
RED='\033[0;31m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
GREEN='\033[0;32m'
BOLD='\033[1m'
DIM='\033[2m'
RESET='\033[0m'

# --- Counters ---
CRITICAL_COUNT=0
WARNING_COUNT=0
INFO_COUNT=0
FILES_SCANNED=0

# --- Options ---
SCAN_PATH=""
JSON_OUTPUT=false

usage() {
    cat <<EOF
${BOLD}scan-skills.sh${RESET} — Scan OpenClaw skill files for malicious patterns

${BOLD}USAGE:${RESET}
    ./scan-skills.sh [OPTIONS]

${BOLD}OPTIONS:${RESET}
    --path <dir>    Scan only the specified directory
    --json          Output results as JSON (for automation)
    --help          Show this help message

${BOLD}EXAMPLES:${RESET}
    ./scan-skills.sh                           # Scan all skill directories
    ./scan-skills.sh --path ./my-skills/       # Scan specific directory
    ./scan-skills.sh --json                    # JSON output
EOF
    exit 0
}

# --- Parse args ---
while [[ $# -gt 0 ]]; do
    case "$1" in
        --path) SCAN_PATH="$2"; shift 2 ;;
        --json) JSON_OUTPUT=true; shift ;;
        --help) usage ;;
        *) echo "Unknown option: $1"; usage ;;
    esac
done

# --- Collect skill directories ---
SKILL_DIRS=()
if [[ -n "$SCAN_PATH" ]]; then
    SKILL_DIRS+=("$SCAN_PATH")
else
    # Workspace skills
    if [[ -d "./skills" ]]; then
        SKILL_DIRS+=("./skills")
    fi
    # User home skills
    if [[ -d "$HOME/.openclaw/skills" ]]; then
        SKILL_DIRS+=("$HOME/.openclaw/skills")
    fi
    # Check common workspace locations
    for ws in "$HOME/openclaw/skills"; do
        if [[ -d "$ws" ]] && [[ ! " ${SKILL_DIRS[*]:-} " =~ " $ws " ]]; then
            SKILL_DIRS+=("$ws")
        fi
    done
fi

if [[ ${#SKILL_DIRS[@]} -eq 0 ]]; then
    echo -e "${YELLOW}No skill directories found to scan.${RESET}"
    exit 0
fi

# --- JSON buffer ---
JSON_FINDINGS="[]"

# --- Report finding ---
report() {
    local severity="$1"
    local file="$2"
    local line_num="$3"
    local category="$4"
    local detail="$5"
    local matched="${6:-}"

    case "$severity" in
        CRITICAL)
            ((CRITICAL_COUNT++)) || true
            if [[ "$JSON_OUTPUT" == "false" ]]; then
                echo -e "  ${RED}${BOLD}[CRITICAL]${RESET} ${category}"
                echo -e "    ${DIM}File:${RESET} ${file}:${line_num}"
                echo -e "    ${DIM}Detail:${RESET} ${detail}"
                [[ -n "$matched" ]] && echo -e "    ${DIM}Match:${RESET} ${RED}${matched}${RESET}"
                echo ""
            fi
            ;;
        WARNING)
            ((WARNING_COUNT++)) || true
            if [[ "$JSON_OUTPUT" == "false" ]]; then
                echo -e "  ${YELLOW}${BOLD}[WARNING]${RESET}  ${category}"
                echo -e "    ${DIM}File:${RESET} ${file}:${line_num}"
                echo -e "    ${DIM}Detail:${RESET} ${detail}"
                [[ -n "$matched" ]] && echo -e "    ${DIM}Match:${RESET} ${YELLOW}${matched}${RESET}"
                echo ""
            fi
            ;;
        INFO)
            ((INFO_COUNT++)) || true
            if [[ "$JSON_OUTPUT" == "false" ]]; then
                echo -e "  ${BLUE}${BOLD}[INFO]${RESET}     ${category}"
                echo -e "    ${DIM}File:${RESET} ${file}:${line_num}"
                echo -e "    ${DIM}Detail:${RESET} ${detail}"
                [[ -n "$matched" ]] && echo -e "    ${DIM}Match:${RESET} ${BLUE}${matched}${RESET}"
                echo ""
            fi
            ;;
    esac

    if [[ "$JSON_OUTPUT" == "true" ]]; then
        # Escape strings for JSON
        local esc_file; esc_file=$(echo "$file" | sed 's/"/\\"/g')
        local esc_detail; esc_detail=$(echo "$detail" | sed 's/"/\\"/g')
        local esc_matched; esc_matched=$(echo "$matched" | sed 's/"/\\"/g')
        JSON_FINDINGS=$(echo "$JSON_FINDINGS" | sed 's/]$//')
        [[ "$JSON_FINDINGS" != "[" ]] && JSON_FINDINGS="${JSON_FINDINGS},"
        JSON_FINDINGS="${JSON_FINDINGS}{\"severity\":\"${severity}\",\"file\":\"${esc_file}\",\"line\":${line_num},\"category\":\"${category}\",\"detail\":\"${esc_detail}\",\"match\":\"${esc_matched}\"}]"
    fi
}

# --- Scan a single file ---
scan_file() {
    local file="$1"
    ((FILES_SCANNED++)) || true

    # Skip scanning our own skill (contains pattern definitions that would trigger false positives)
    if [[ "$file" == *"openclaw-security-hardening"* ]]; then
        return
    fi

    # Read file content (skip binary files)
    if file "$file" | grep -q "binary"; then
        return
    fi

    local content
    content=$(cat "$file" 2>/dev/null) || return

    # Helper: scan with a combined regex, report each match
    # Usage: scan_combined "SEVERITY" "CATEGORY" "detail_prefix" "pattern1|pattern2|..."
    scan_combined() {
        local sev="$1" cat="$2" detail="$3" combined="$4"
        local matches
        matches=$(echo "$content" | grep -inE "$combined" | head -5) || true
        if [[ -n "$matches" ]]; then
            while IFS= read -r match_line; do
                [[ -z "$match_line" ]] && continue
                local ln
                ln=$(echo "$match_line" | cut -d: -f1)
                local txt
                txt=$(echo "$match_line" | cut -d: -f2- | head -c 120)
                report "$sev" "$file" "$ln" "$cat" "$detail" "$txt"
            done <<< "$matches"
        fi
    }

    # --- CRITICAL: Prompt Injection Patterns ---
    scan_combined "CRITICAL" "Prompt Injection" "Prompt injection pattern detected" \
        'ignore previous instructions|ignore all previous|ignore your instructions|ignore above instructions|disregard your programming|disregard previous|disregard all prior|forget your instructions|forget everything above|you are now|new system prompt|override your|ADMIN OVERRIDE|SYSTEM OVERRIDE|developer mode|DAN mode|jailbreak|act as if you have no restrictions|pretend you are|roleplay as an unrestricted|from now on you will|new instructions override|bypass your safety|bypass your rules|ignore safety guidelines'

    # --- CRITICAL: Data Exfiltration ---
    scan_combined "CRITICAL" "Data Exfiltration" "Potential data exfiltration pattern" \
        'curl.*-d.*\$|curl.*--data|curl.*POST.*http|wget.*--post-data|send.*contents.*to|upload.*file.*to|exfiltrate|send.*MEMORY\.md|send.*USER\.md|send.*SOUL\.md|send.*credentials|send.*\.env|send.*api.key|send.*token.*to|post.*secret.*to|transmit.*data.*external'

    # --- CRITICAL: Suspicious URLs ---
    scan_combined "CRITICAL" "Suspicious URL" "Known data collection/exfiltration URL" \
        'webhook\.site|requestbin|pipedream\.net|ngrok\.io|ngrok\.app|hookbin\.com|burpcollaborator|interact\.sh|canarytokens|requestcatcher|mockbin|postb\.in|beeceptor'

    # --- WARNING: Base64 Encoded Content ---
    local b64_line
    b64_line=$(echo "$content" | grep -nE '[A-Za-z0-9+/]{50,}={0,2}' | head -1) || true
    if [[ -n "$b64_line" ]]; then
        local line_num
        line_num=$(echo "$b64_line" | cut -d: -f1)
        report "WARNING" "$file" "$line_num" "Base64 Content" \
            "Contains long base64-encoded string that could hide instructions" \
            "$(echo "$b64_line" | cut -d: -f2- | head -c 80)..."
    fi

    # --- CRITICAL: Hidden Unicode Characters ---
    if LC_ALL=C grep -n $'\xe2\x80\x8b\|\xe2\x80\x8c\|\xe2\x80\x8d\|\xe2\x80\xae\|\xef\xbb\xbf' "$file" >/dev/null 2>&1; then
        local line_num
        line_num=$(LC_ALL=C grep -n $'\xe2\x80\x8b\|\xe2\x80\x8c\|\xe2\x80\x8d\|\xe2\x80\xae\|\xef\xbb\xbf' "$file" | head -1 | cut -d: -f1) || true
        report "CRITICAL" "$file" "${line_num:-0}" "Hidden Unicode" \
            "Contains invisible unicode characters (zero-width/RTL override)" ""
    fi

    # --- WARNING: Sensitive File References ---
    scan_combined "WARNING" "Sensitive File Reference" "References sensitive data" \
        'read.*\.env|cat.*\.env|credentials|api[_-]?key|secret[_-]?key|access[_-]?token|private[_-]?key|password[s]?|\.ssh/id_|\.aws/credentials|op item get|1password'

    # --- WARNING: System File Modification ---
    scan_combined "WARNING" "System File Modification" "Instructions to modify system files" \
        'modify.*AGENTS\.md|edit.*AGENTS\.md|write.*AGENTS\.md|overwrite.*AGENTS\.md|modify.*SOUL\.md|edit.*SOUL\.md|write.*SOUL\.md|change.*system.*prompt|update.*AGENTS\.md|append.*AGENTS\.md'

    # --- WARNING: Obfuscated Commands ---
    local hex_match
    hex_match=$(echo "$content" | grep -nE '(\\x[0-9a-fA-F]{2}){4,}' | head -1) || true
    if [[ -n "$hex_match" ]]; then
        local line_num
        line_num=$(echo "$hex_match" | cut -d: -f1)
        report "WARNING" "$file" "$line_num" "Obfuscated Command" \
            "Contains hex-encoded content that may hide commands" \
            "$(echo "$hex_match" | cut -d: -f2- | head -c 80)"
    fi

    local uni_match
    uni_match=$(echo "$content" | grep -nE '(\\u[0-9a-fA-F]{4}){4,}' | head -1) || true
    if [[ -n "$uni_match" ]]; then
        local line_num
        line_num=$(echo "$uni_match" | cut -d: -f1)
        report "WARNING" "$file" "$line_num" "Obfuscated Command" \
            "Contains unicode-escaped content that may hide commands" \
            "$(echo "$uni_match" | cut -d: -f2- | head -c 80)"
    fi

    # --- CRITICAL: Social Engineering ---
    scan_combined "CRITICAL" "Social Engineering" "Social engineering pattern detected" \
        "don't tell the user|do not tell the user|don't mention|do not mention|don't reveal|do not reveal|secretly|without mentioning|without telling|without informing|hide this from|keep this hidden|do not log|don't log|suppress.*output|silently|covertly|surreptitiously"

    # --- WARNING: Paste site URLs ---
    scan_combined "WARNING" "Paste Site URL" "Contains paste/sharing site URL" \
        'pastebin\.com|hastebin|ghostbin|dpaste|paste\.ee|justpaste\.it|controlc\.com|rentry\.co'

    # --- INFO: Excessive code blocks ---
    local code_block_count
    code_block_count=$(echo "$content" | grep -c '```') || true
    if [[ $code_block_count -gt 20 ]]; then
        report "INFO" "$file" "0" "Excessive Code Blocks" \
            "Contains $code_block_count code block markers — review for hidden instructions" ""
    fi
}

# --- Main ---
if [[ "$JSON_OUTPUT" == "false" ]]; then
    echo ""
    echo -e "${BOLD}╔══════════════════════════════════════════════════════════╗${RESET}"
    echo -e "${BOLD}║        OpenClaw Security — Skill File Scanner           ║${RESET}"
    echo -e "${BOLD}╚══════════════════════════════════════════════════════════╝${RESET}"
    echo ""
    echo -e "${DIM}Scanning directories:${RESET}"
    for dir in "${SKILL_DIRS[@]}"; do
        echo -e "  ${DIM}→${RESET} $dir"
    done
    echo ""
fi

# Find and scan all text files in skill directories
for dir in "${SKILL_DIRS[@]}"; do
    if [[ ! -d "$dir" ]]; then
        continue
    fi

    while IFS= read -r -d '' file; do
        # Scan markdown, text, yaml, json, shell, javascript, python files
        case "$file" in
            *.md|*.txt|*.yaml|*.yml|*.json|*.sh|*.bash|*.js|*.mjs|*.py|*.ts|*.toml)
                scan_file "$file"
                ;;
        esac
    done < <(find "$dir" -not -path "*/node_modules/*" -not -path "*/.git/*" -not -path "*/.venv/*" -not -path "*/venv/*" -not -path "*/__pycache__/*" -type f -print0 2>/dev/null)
done

# --- Summary ---
if [[ "$JSON_OUTPUT" == "true" ]]; then
    echo "{\"files_scanned\":${FILES_SCANNED},\"critical\":${CRITICAL_COUNT},\"warnings\":${WARNING_COUNT},\"info\":${INFO_COUNT},\"findings\":${JSON_FINDINGS}}"
else
    echo -e "${BOLD}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${RESET}"
    echo -e "${BOLD}Scan Complete${RESET}"
    echo -e "  Files scanned: ${FILES_SCANNED}"
    echo -e "  ${RED}${BOLD}Critical:${RESET} ${CRITICAL_COUNT}"
    echo -e "  ${YELLOW}${BOLD}Warnings:${RESET} ${WARNING_COUNT}"
    echo -e "  ${BLUE}${BOLD}Info:${RESET}     ${INFO_COUNT}"
    echo ""

    if [[ $CRITICAL_COUNT -gt 0 ]]; then
        echo -e "  ${RED}${BOLD}⚠ CRITICAL issues found! Review immediately.${RESET}"
    elif [[ $WARNING_COUNT -gt 0 ]]; then
        echo -e "  ${YELLOW}Warnings found — manual review recommended.${RESET}"
    else
        echo -e "  ${GREEN}✓ No issues detected. Skills look clean.${RESET}"
    fi
    echo ""
fi

# Exit with appropriate code
if [[ $CRITICAL_COUNT -gt 0 ]]; then
    exit 2
elif [[ $WARNING_COUNT -gt 0 ]]; then
    exit 1
else
    exit 0
fi
