#!/usr/bin/env bash
set -euo pipefail

# =============================================================================
# integrity-check.sh — Skill file integrity monitor for OpenClaw
# Creates hash baselines and detects unauthorized modifications.
# =============================================================================

# --- Colors ---
RED='\033[0;31m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
GREEN='\033[0;32m'
BOLD='\033[1m'
DIM='\033[2m'
RESET='\033[0m'

# --- Config ---
SECURITY_DIR="$HOME/.openclaw/security"
HASH_FILE="$SECURITY_DIR/skill-hashes.json"

# --- Options ---
MODE="check"  # check, init, update
SCAN_PATH=""

usage() {
    cat <<EOF
${BOLD}integrity-check.sh${RESET} — Monitor skill file integrity

${BOLD}USAGE:${RESET}
    ./integrity-check.sh [OPTIONS]

${BOLD}OPTIONS:${RESET}
    --init          Create initial hash baseline
    --update        Update baseline to current state
    --path <dir>    Check only the specified directory
    --help          Show this help message

${BOLD}EXAMPLES:${RESET}
    ./integrity-check.sh --init            # First run: create baseline
    ./integrity-check.sh                   # Check for changes
    ./integrity-check.sh --update          # Accept current state
    ./integrity-check.sh --path ./skills/  # Check specific directory
EOF
    exit 0
}

# --- Parse args ---
while [[ $# -gt 0 ]]; do
    case "$1" in
        --init) MODE="init"; shift ;;
        --update) MODE="update"; shift ;;
        --path) SCAN_PATH="$2"; shift 2 ;;
        --help) usage ;;
        *) echo "Unknown option: $1"; usage ;;
    esac
done

# --- Collect skill directories ---
SKILL_DIRS=()
if [[ -n "$SCAN_PATH" ]]; then
    SKILL_DIRS+=("$SCAN_PATH")
else
    for dir in "./skills" "$HOME/.openclaw/skills" "$HOME/openclaw/skills"; do
        if [[ -d "$dir" ]]; then
            # Resolve to absolute path to avoid duplicates
            local_abs=$(cd "$dir" && pwd)
            skip=false
            for existing in "${SKILL_DIRS[@]:-}"; do
                if [[ -n "$existing" ]]; then
                    existing_abs=$(cd "$existing" && pwd)
                    if [[ "$local_abs" == "$existing_abs" ]]; then
                        skip=true
                        break
                    fi
                fi
            done
            if [[ "$skip" == "false" ]]; then
                SKILL_DIRS+=("$dir")
            fi
        fi
    done
fi

# --- Compute hashes for all skill files ---
compute_hashes() {
    local result="{"
    local first=true

    for dir in "${SKILL_DIRS[@]}"; do
        if [[ ! -d "$dir" ]]; then
            continue
        fi

        while IFS= read -r -d '' file; do
            case "$file" in
                *.md|*.txt|*.yaml|*.yml|*.json|*.sh|*.bash|*.js|*.mjs|*.py|*.ts|*.toml)
                    local hash
                    hash=$(shasum -a 256 "$file" | cut -d' ' -f1)
                    local rel_path
                    rel_path=$(realpath "$file" 2>/dev/null || echo "$file")

                    if [[ "$first" == "true" ]]; then
                        first=false
                    else
                        result="${result},"
                    fi
                    # Escape the path for JSON
                    local escaped_path
                    escaped_path=$(echo "$rel_path" | sed 's/"/\\"/g')
                    result="${result}\"${escaped_path}\":\"${hash}\""
                    ;;
            esac
        done < <(find "$dir" -not -path "*/node_modules/*" -not -path "*/.git/*" -not -path "*/.venv/*" -not -path "*/venv/*" -not -path "*/__pycache__/*" -type f -print0 2>/dev/null)
    done

    result="${result}}"
    echo "$result"
}

# --- Initialize baseline ---
init_baseline() {
    echo -e "${BOLD}Initializing integrity baseline...${RESET}"
    echo ""

    mkdir -p "$SECURITY_DIR"

    if [[ -f "$HASH_FILE" ]]; then
        echo -e "${YELLOW}Warning: Baseline already exists at ${HASH_FILE}${RESET}"
        echo -e "${YELLOW}Use --update to overwrite, or delete it first.${RESET}"
        exit 1
    fi

    local hashes
    hashes=$(compute_hashes)

    echo "$hashes" > "$HASH_FILE"

    # Count entries
    local count
    count=$(echo "$hashes" | grep -o '":' | wc -l | tr -d ' ')

    echo -e "${GREEN}✓ Baseline created with ${count} file hashes${RESET}"
    echo -e "${DIM}Stored at: ${HASH_FILE}${RESET}"
    echo ""
    echo -e "${DIM}Run ./integrity-check.sh periodically to detect changes.${RESET}"
}

# --- Update baseline ---
update_baseline() {
    echo -e "${BOLD}Updating integrity baseline...${RESET}"
    echo ""

    mkdir -p "$SECURITY_DIR"

    local hashes
    hashes=$(compute_hashes)

    echo "$hashes" > "$HASH_FILE"

    local count
    count=$(echo "$hashes" | grep -o '":' | wc -l | tr -d ' ')

    echo -e "${GREEN}✓ Baseline updated with ${count} file hashes${RESET}"
    echo -e "${DIM}Stored at: ${HASH_FILE}${RESET}"
}

# --- Check integrity ---
check_integrity() {
    if [[ ! -f "$HASH_FILE" ]]; then
        echo -e "${YELLOW}No baseline found. Run with --init first.${RESET}"
        echo -e "${DIM}  ./integrity-check.sh --init${RESET}"
        exit 1
    fi

    echo ""
    echo -e "${BOLD}╔══════════════════════════════════════════════════════════╗${RESET}"
    echo -e "${BOLD}║       OpenClaw Security — Integrity Check               ║${RESET}"
    echo -e "${BOLD}╚══════════════════════════════════════════════════════════╝${RESET}"
    echo ""

    local modified=0
    local new_files=0
    local removed=0
    local unchanged=0

    # Load stored hashes into an associative approach using temp files (bash 3 compat)
    local stored_file
    stored_file=$(mktemp)
    local current_file
    current_file=$(mktemp)
    trap "rm -f '$stored_file' '$current_file'" EXIT

    # Parse stored hashes: extract "path":"hash" pairs
    # Use python if available for reliable JSON parsing, else sed
    if command -v python3 &>/dev/null; then
        python3 -c "
import json, sys
with open('$HASH_FILE') as f:
    data = json.load(f)
for path, h in sorted(data.items()):
    print(f'{path}\t{h}')
" > "$stored_file"
    else
        # Fallback: basic parsing (works for simple flat JSON)
        cat "$HASH_FILE" | tr ',' '\n' | sed 's/^{//;s/}$//;s/^"//;s/"$//' | \
            sed 's/":"/'$'\t''/' > "$stored_file"
    fi

    # Compute current hashes
    for dir in "${SKILL_DIRS[@]}"; do
        if [[ ! -d "$dir" ]]; then
            continue
        fi
        while IFS= read -r -d '' file; do
            case "$file" in
                *.md|*.txt|*.yaml|*.yml|*.json|*.sh|*.bash|*.js|*.mjs|*.py|*.ts|*.toml)
                    local hash
                    hash=$(shasum -a 256 "$file" | cut -d' ' -f1)
                    local rel_path
                    rel_path=$(realpath "$file" 2>/dev/null || echo "$file")
                    echo -e "${rel_path}\t${hash}" >> "$current_file"
                    ;;
            esac
        done < <(find "$dir" -not -path "*/node_modules/*" -not -path "*/.git/*" -not -path "*/.venv/*" -not -path "*/venv/*" -not -path "*/__pycache__/*" -type f -print0 2>/dev/null)
    done

    # Check for modified and unchanged files
    while IFS=$'\t' read -r path stored_hash; do
        [[ -z "$path" ]] && continue
        local current_hash
        current_hash=$(grep "^${path}"$'\t' "$current_file" 2>/dev/null | cut -f2 || true)

        if [[ -z "$current_hash" ]]; then
            echo -e "  ${RED}❌ REMOVED${RESET}  $path"
            ((removed++)) || true
        elif [[ "$current_hash" != "$stored_hash" ]]; then
            echo -e "  ${YELLOW}⚠ MODIFIED${RESET} $path"
            ((modified++)) || true
        else
            ((unchanged++)) || true
        fi
    done < "$stored_file"

    # Check for new files
    while IFS=$'\t' read -r path current_hash; do
        [[ -z "$path" ]] && continue
        if ! grep -q "^${path}"$'\t' "$stored_file" 2>/dev/null; then
            echo -e "  ${BLUE}🆕 NEW${RESET}      $path"
            ((new_files++)) || true
        fi
    done < "$current_file"

    # Summary
    echo ""
    echo -e "${BOLD}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${RESET}"
    echo -e "  ${GREEN}✓ Unchanged:${RESET} ${unchanged}"
    echo -e "  ${YELLOW}⚠ Modified:${RESET}  ${modified}"
    echo -e "  ${BLUE}🆕 New:${RESET}       ${new_files}"
    echo -e "  ${RED}❌ Removed:${RESET}  ${removed}"
    echo ""

    if [[ $modified -gt 0 || $removed -gt 0 ]]; then
        echo -e "  ${RED}${BOLD}⚠ Integrity violations detected!${RESET}"
        echo -e "  ${DIM}Review changes. Run --update to accept current state.${RESET}"
        echo ""
        exit 1
    elif [[ $new_files -gt 0 ]]; then
        echo -e "  ${YELLOW}New files found. Run --update to add them to baseline.${RESET}"
        echo ""
        exit 0
    else
        echo -e "  ${GREEN}✓ All files match baseline. No tampering detected.${RESET}"
        echo ""
        exit 0
    fi
}

# --- Main ---
case "$MODE" in
    init) init_baseline ;;
    update) update_baseline ;;
    check) check_integrity ;;
esac
