#!/usr/bin/env bash
set -uo pipefail

# =============================================================================
# install-guard.sh — Pre-install security gate for OpenClaw skills
# Run before installing any new skill to check for malicious content.
# Returns exit code 0 (safe) or 1 (suspicious) for automation.
# =============================================================================

# --- Colors ---
RED='\033[0;31m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
GREEN='\033[0;32m'
BOLD='\033[1m'
DIM='\033[2m'
RESET='\033[0m'

# --- Options ---
STRICT=false
SKILL_PATH=""
ISSUES=0

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

usage() {
    cat <<EOF
${BOLD}install-guard.sh${RESET} — Pre-install security check for OpenClaw skills

${BOLD}USAGE:${RESET}
    ./install-guard.sh [OPTIONS] <skill-directory>

${BOLD}OPTIONS:${RESET}
    --strict    Fail on warnings too (not just critical)
    --help      Show this help message

${BOLD}EXAMPLES:${RESET}
    ./install-guard.sh /path/to/new-skill/
    ./install-guard.sh --strict /path/to/new-skill/

${BOLD}EXIT CODES:${RESET}
    0 — Skill appears safe
    1 — Suspicious patterns detected
EOF
    exit 0
}

# --- Parse args ---
while [[ $# -gt 0 ]]; do
    case "$1" in
        --strict) STRICT=true; shift ;;
        --help) usage ;;
        -*)
            echo "Unknown option: $1"
            usage
            ;;
        *)
            SKILL_PATH="$1"
            shift
            ;;
    esac
done

if [[ -z "$SKILL_PATH" ]]; then
    echo -e "${RED}Error: No skill directory specified.${RESET}"
    echo ""
    usage
fi

if [[ ! -d "$SKILL_PATH" ]]; then
    echo -e "${RED}Error: '$SKILL_PATH' is not a directory.${RESET}"
    exit 1
fi

echo ""
echo -e "${BOLD}╔══════════════════════════════════════════════════════════╗${RESET}"
echo -e "${BOLD}║     OpenClaw Security — Install Guard                   ║${RESET}"
echo -e "${BOLD}╚══════════════════════════════════════════════════════════╝${RESET}"
echo ""
echo -e "${DIM}Checking: ${SKILL_PATH}${RESET}"
echo ""

fail() {
    echo -e "  ${RED}${BOLD}[FAIL]${RESET} $1"
    ((ISSUES++)) || true
}

warn() {
    echo -e "  ${YELLOW}${BOLD}[WARN]${RESET} $1"
    if [[ "$STRICT" == "true" ]]; then
        ((ISSUES++)) || true
    fi
}

pass() {
    echo -e "  ${GREEN}[PASS]${RESET} $1"
}

info() {
    echo -e "  ${BLUE}[INFO]${RESET} $1"
}

# ==========================================================================
echo -e "${BOLD}[1/4] Content Scan${RESET}"
# ==========================================================================

# Run the main scan-skills.sh against this directory
if [[ -x "$SCRIPT_DIR/scan-skills.sh" ]]; then
    scan_output=$("$SCRIPT_DIR/scan-skills.sh" --json --path "$SKILL_PATH" 2>/dev/null || true)

    if [[ -n "$scan_output" ]]; then
        critical=$(echo "$scan_output" | grep -o '"critical":[0-9]*' | cut -d: -f2 || echo "0")
        warnings=$(echo "$scan_output" | grep -o '"warnings":[0-9]*' | cut -d: -f2 || echo "0")

        if [[ "${critical:-0}" -gt 0 ]]; then
            fail "Content scan found ${critical} CRITICAL issue(s)"
            # Show details
            echo -e "${DIM}"
            "$SCRIPT_DIR/scan-skills.sh" --path "$SKILL_PATH" 2>/dev/null || true
            echo -e "${RESET}"
            ((ISSUES += critical)) || true
        elif [[ "${warnings:-0}" -gt 0 ]]; then
            warn "Content scan found ${warnings} warning(s)"
            if [[ "$STRICT" == "true" ]]; then
                echo -e "${DIM}"
                "$SCRIPT_DIR/scan-skills.sh" --path "$SKILL_PATH" 2>/dev/null || true
                echo -e "${RESET}"
            fi
        else
            pass "Content scan — no issues found"
        fi
    else
        pass "Content scan — no issues found"
    fi
else
    warn "scan-skills.sh not found — skipping content scan"
fi

echo ""

# ==========================================================================
echo -e "${BOLD}[2/4] Script Safety Check${RESET}"
# ==========================================================================

SCRIPTS_DIR="$SKILL_PATH/scripts"
if [[ -d "$SCRIPTS_DIR" ]]; then
    script_count=$(find "$SCRIPTS_DIR" -type f \( -name "*.sh" -o -name "*.bash" -o -name "*.py" -o -name "*.js" -o -name "*.mjs" \) | wc -l | tr -d ' ')
    info "Found ${script_count} script(s) in scripts/"

    while IFS= read -r -d '' script; do
        [[ ! -f "$script" ]] && continue
        script_name=$(basename "$script")
        content=$(cat "$script" 2>/dev/null) || continue

        # Helper: check pattern and report
        check_pattern() {
            local pat="$1" sev="$2" msg="$3"
            local match
            match=$(echo "$content" | grep -nE "$pat" | head -1 || true)
            if [[ -n "$match" ]]; then
                "$sev" "$script_name: $msg"
                echo -e "    ${DIM}$(echo "$match" | head -c 120)${RESET}"
                return 0
            fi
            return 1
        }

        # --- Dangerous shell patterns ---
        check_pattern 'rm\s+(-[a-zA-Z]*)?r[a-zA-Z]*f?\s+(/|\$HOME|~|\.\.)' fail "Dangerous rm -rf with broad path" || true
        check_pattern 'curl.*\|\s*(ba)?sh' fail "curl piped to shell execution" || true
        check_pattern 'wget.*\|\s*(ba)?sh' fail "wget piped to shell execution" || true
        check_pattern 'eval\s+[\$"'"'"']' warn "Uses eval with variable expansion" || true
        check_pattern 'exec\s+[0-9]*[<>]' warn "Uses exec with redirection" || true
        check_pattern '(curl|wget|fetch).*&&.*(chmod|bash|sh|python|node)' fail "Downloads and executes in sequence" || true
        check_pattern 'base64.*-[dD].*\|.*(ba)?sh' fail "Base64 decode piped to shell" || true
        check_pattern '(python|node).*-[ce].*http' fail "Runtime executing code from network" || true
        check_pattern 'export\s+(PATH|LD_PRELOAD|DYLD_)' warn "Modifies critical environment variables" || true
        check_pattern '(/etc/passwd|/etc/shadow|/etc/sudoers)' fail "Accesses system auth files" || true

    done < <(find "$SCRIPTS_DIR" -type f \( -name "*.sh" -o -name "*.bash" -o -name "*.py" -o -name "*.js" -o -name "*.mjs" \) -print0 2>/dev/null)

    if [[ $ISSUES -eq 0 ]]; then
        pass "Scripts — no dangerous patterns found"
    fi
else
    pass "No scripts/ directory"
fi

echo ""

# ==========================================================================
echo -e "${BOLD}[3/4] Dependency Check${RESET}"
# ==========================================================================

PKG_JSON="$SKILL_PATH/package.json"
if [[ -f "$PKG_JSON" ]]; then
    info "Found package.json — checking dependencies"

    # Known suspicious packages
    SUSPICIOUS_PKGS=(
        "event-stream"      # Historical supply chain attack
        "flatmap-stream"    # Historical supply chain attack
        "ua-parser-js"      # Compromised versions
        "coa"               # Compromised versions
        "rc"                # Compromised versions
        "colors"            # Sabotaged by author
        "faker"             # Sabotaged by author
        "node-ipc"          # Protestware
        "peacenotwar"       # Protestware
    )

    deps=$(cat "$PKG_JSON" | grep -oE '"[^"]+"\s*:\s*"[^"]*"' | cut -d'"' -f2 || true)

    for pkg in "${SUSPICIOUS_PKGS[@]}"; do
        if echo "$deps" | grep -q "^${pkg}$"; then
            fail "Suspicious npm package: $pkg (known compromised/malicious)"
        fi
    done

    # Check for git:// URLs in dependencies (can be hijacked)
    if grep -qE 'git://' "$PKG_JSON"; then
        warn "package.json uses insecure git:// protocol URLs"
    fi

    # Check for github URLs without commit pins
    if grep -qE 'github\.com/[^#]*"' "$PKG_JSON" | grep -v '#' &>/dev/null; then
        warn "package.json has GitHub dependencies without commit hash pins"
    fi

    # Check for postinstall scripts
    if grep -qE '"(preinstall|postinstall|prepare)"' "$PKG_JSON"; then
        warn "package.json has install lifecycle scripts — review carefully"
        install_script=$(grep -oE '"(preinstall|postinstall|prepare)"\s*:\s*"[^"]*"' "$PKG_JSON" || true)
        if [[ -n "$install_script" ]]; then
            echo -e "    ${DIM}$install_script${RESET}"
        fi
    fi

    if [[ $ISSUES -eq 0 ]]; then
        pass "Dependencies — no known issues"
    fi
else
    pass "No package.json"
fi

# Check for requirements.txt (Python)
REQ_TXT="$SKILL_PATH/requirements.txt"
if [[ -f "$REQ_TXT" ]]; then
    info "Found requirements.txt — checking Python dependencies"

    # Check for --index-url pointing to non-PyPI
    if grep -qiE '(-i|--index-url|--extra-index-url)\s+' "$REQ_TXT"; then
        warn "requirements.txt uses custom package index — potential dependency confusion"
    fi

    pass "Python dependencies — basic check passed"
fi

echo ""

# ==========================================================================
echo -e "${BOLD}[4/4] Structure Check${RESET}"
# ==========================================================================

# Check for SKILL.md
if [[ -f "$SKILL_PATH/SKILL.md" ]]; then
    pass "SKILL.md exists"

    # Check for proper frontmatter
    if head -1 "$SKILL_PATH/SKILL.md" | grep -q '^---'; then
        pass "SKILL.md has frontmatter"
    else
        warn "SKILL.md missing frontmatter header"
    fi
else
    warn "No SKILL.md found — non-standard skill structure"
fi

# Check for hidden files
hidden_files=$(find "$SKILL_PATH" -name ".*" -not -name ".gitignore" -not -name ".git" -not -name ".gitkeep" -not -name ".npmignore" -type f 2>/dev/null | head -5)
if [[ -n "$hidden_files" ]]; then
    warn "Hidden files found in skill:"
    while IFS= read -r hf; do
        echo -e "    ${DIM}$hf${RESET}"
    done <<< "$hidden_files"
fi

# Check for symlinks (could point outside skill directory)
symlinks=$(find "$SKILL_PATH" -type l 2>/dev/null | head -5)
if [[ -n "$symlinks" ]]; then
    warn "Symlinks found (could reference files outside skill):"
    while IFS= read -r sl; do
        local target
        target=$(readlink "$sl" || echo "unresolvable")
        echo -e "    ${DIM}$sl -> $target${RESET}"
    done <<< "$symlinks"
fi

echo ""

# ==========================================================================
# Summary
# ==========================================================================
echo -e "${BOLD}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${RESET}"

if [[ $ISSUES -gt 0 ]]; then
    echo -e "${RED}${BOLD}⚠ SUSPICIOUS — ${ISSUES} issue(s) found${RESET}"
    echo -e "${DIM}Review the findings above before installing this skill.${RESET}"
    echo ""
    exit 1
else
    echo -e "${GREEN}${BOLD}✓ SAFE — No issues detected${RESET}"
    echo -e "${DIM}Skill appears safe to install.${RESET}"
    echo ""
    exit 0
fi
