#!/usr/bin/env bash
set -euo pipefail

# =============================================================================
# harden-workspace.sh — Workspace security hardener for OpenClaw
# Checks permissions, .gitignore, gateway config, and sensitive data exposure.
# =============================================================================

# --- Colors ---
RED='\033[0;31m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
GREEN='\033[0;32m'
BOLD='\033[1m'
DIM='\033[2m'
RESET='\033[0m'

# --- Options ---
FIX_MODE=false
ISSUES=0
FIXED=0

usage() {
    cat <<EOF
${BOLD}harden-workspace.sh${RESET} — Harden your OpenClaw workspace

${BOLD}USAGE:${RESET}
    ./harden-workspace.sh [OPTIONS]

${BOLD}OPTIONS:${RESET}
    --fix     Auto-remediate safe issues
    --help    Show this help message

${BOLD}EXAMPLES:${RESET}
    ./harden-workspace.sh          # Check only
    ./harden-workspace.sh --fix    # Check and fix
EOF
    exit 0
}

while [[ $# -gt 0 ]]; do
    case "$1" in
        --fix) FIX_MODE=true; shift ;;
        --help) usage ;;
        *) echo "Unknown option: $1"; usage ;;
    esac
done

# --- Find workspace root ---
# Look for AGENTS.md as workspace indicator
WORKSPACE=""
for candidate in "." "$HOME/openclaw"; do
    if [[ -f "$candidate/AGENTS.md" ]]; then
        WORKSPACE=$(cd "$candidate" && pwd)
        break
    fi
done

if [[ -z "$WORKSPACE" ]]; then
    echo -e "${YELLOW}Could not find OpenClaw workspace (no AGENTS.md found).${RESET}"
    echo -e "${DIM}Run from your workspace directory or ensure AGENTS.md exists.${RESET}"
    exit 1
fi

echo ""
echo -e "${BOLD}╔══════════════════════════════════════════════════════════╗${RESET}"
echo -e "${BOLD}║      OpenClaw Security — Workspace Hardening            ║${RESET}"
echo -e "${BOLD}╚══════════════════════════════════════════════════════════╝${RESET}"
echo ""
echo -e "${DIM}Workspace: ${WORKSPACE}${RESET}"
echo -e "${DIM}Fix mode:  ${FIX_MODE}${RESET}"
echo ""

# --- Helper ---
check_pass() {
    echo -e "  ${GREEN}✓${RESET} $1"
}

check_fail() {
    echo -e "  ${RED}✗${RESET} $1"
    ((ISSUES++)) || true
}

check_warn() {
    echo -e "  ${YELLOW}!${RESET} $1"
    ((ISSUES++)) || true
}

check_fixed() {
    echo -e "  ${GREEN}✓ FIXED${RESET} $1"
    ((FIXED++)) || true
}

# ==========================================================================
echo -e "${BOLD}[1/6] File Permissions${RESET}"
# ==========================================================================

SENSITIVE_FILES=(
    "MEMORY.md"
    "USER.md"
    "SOUL.md"
    ".credentials"
    ".env"
    "TOOLS.md"
)

for f in "${SENSITIVE_FILES[@]}"; do
    filepath="$WORKSPACE/$f"
    if [[ -f "$filepath" ]]; then
        # Check if file is world-readable (macOS: check 'other' read bit)
        perms=$(stat -f '%Lp' "$filepath" 2>/dev/null || stat -c '%a' "$filepath" 2>/dev/null)
        other_read=$((perms % 10))
        if [[ $other_read -ge 4 ]]; then
            if [[ "$FIX_MODE" == "true" ]]; then
                chmod 600 "$filepath"
                check_fixed "$f — set to 600 (owner read/write only)"
            else
                check_fail "$f is world-readable (perms: $perms) — should be 600"
            fi
        else
            check_pass "$f permissions OK ($perms)"
        fi
    fi
done

echo ""

# ==========================================================================
echo -e "${BOLD}[2/6] .gitignore Coverage${RESET}"
# ==========================================================================

REQUIRED_PATTERNS=(
    ".credentials"
    ".env"
    "*.key"
    "*.pem"
    ".op-secret-key"
    "*.secret"
    "node_modules/"
)

GITIGNORE="$WORKSPACE/.gitignore"

if [[ -f "$GITIGNORE" ]]; then
    missing_patterns=()
    for pattern in "${REQUIRED_PATTERNS[@]}"; do
        if grep -qF "$pattern" "$GITIGNORE" 2>/dev/null; then
            check_pass ".gitignore includes '$pattern'"
        else
            missing_patterns+=("$pattern")
            if [[ "$FIX_MODE" == "true" ]]; then
                echo "$pattern" >> "$GITIGNORE"
                check_fixed "Added '$pattern' to .gitignore"
            else
                check_fail ".gitignore missing '$pattern'"
            fi
        fi
    done
else
    check_warn "No .gitignore found"
    if [[ "$FIX_MODE" == "true" ]]; then
        {
            echo "# Security-sensitive files"
            for pattern in "${REQUIRED_PATTERNS[@]}"; do
                echo "$pattern"
            done
        } > "$GITIGNORE"
        check_fixed "Created .gitignore with security patterns"
    fi
fi

echo ""

# ==========================================================================
echo -e "${BOLD}[3/6] Sensitive Content in Version Control${RESET}"
# ==========================================================================

# Check if git is initialized
if [[ -d "$WORKSPACE/.git" ]]; then
    # Check if sensitive files are tracked by git
    for f in MEMORY.md USER.md SOUL.md .env .credentials; do
        filepath="$WORKSPACE/$f"
        if [[ -f "$filepath" ]]; then
            if git -C "$WORKSPACE" ls-files --error-unmatch "$f" &>/dev/null; then
                check_warn "$f is tracked by git — consider removing from VCS"
                if [[ "$FIX_MODE" == "true" ]]; then
                    echo -e "    ${DIM}(Manual action needed: git rm --cached $f)${RESET}"
                fi
            else
                check_pass "$f is not tracked by git"
            fi
        fi
    done
else
    check_pass "No git repository (version control exposure N/A)"
fi

echo ""

# ==========================================================================
echo -e "${BOLD}[4/6] Gateway Configuration${RESET}"
# ==========================================================================

# Look for gateway config
GATEWAY_CONFIGS=(
    "$HOME/.openclaw/config.yaml"
    "$HOME/.openclaw/config.json"
    "$HOME/.config/openclaw/config.yaml"
    "$HOME/.config/openclaw/config.json"
    "$WORKSPACE/gateway.yaml"
    "$WORKSPACE/gateway.json"
)

gateway_found=false
for cfg in "${GATEWAY_CONFIGS[@]}"; do
    if [[ -f "$cfg" ]]; then
        gateway_found=true
        echo -e "  ${DIM}Found: $cfg${RESET}"

        # Check for auth settings
        if grep -qiE '(auth|token|apiKey|api_key)' "$cfg" 2>/dev/null; then
            check_pass "Gateway config appears to have auth settings"
        else
            check_warn "Gateway config may not have auth enabled — verify manually"
        fi

        # Check for open DM policy
        if grep -qiE 'dm.*policy.*open' "$cfg" 2>/dev/null; then
            check_fail "Gateway DM policy is set to 'open' — should be 'pairing'"
        elif grep -qiE 'dm.*policy.*pairing' "$cfg" 2>/dev/null; then
            check_pass "Gateway DM policy is 'pairing'"
        else
            check_warn "Could not determine DM policy — verify manually"
        fi

        # Check permissions on config file
        perms=$(stat -f '%Lp' "$cfg" 2>/dev/null || stat -c '%a' "$cfg" 2>/dev/null)
        other_read=$((perms % 10))
        if [[ $other_read -ge 4 ]]; then
            if [[ "$FIX_MODE" == "true" ]]; then
                chmod 600 "$cfg"
                check_fixed "Gateway config permissions set to 600"
            else
                check_fail "Gateway config is world-readable ($perms)"
            fi
        else
            check_pass "Gateway config permissions OK ($perms)"
        fi
        break
    fi
done

if [[ "$gateway_found" == "false" ]]; then
    check_warn "No gateway config found — skipping gateway checks"
fi

echo ""

# ==========================================================================
echo -e "${BOLD}[5/6] Credential Exposure${RESET}"
# ==========================================================================

# Check for plaintext credential files in obvious locations
CRED_LOCATIONS=(
    "$HOME/Documents/passwords.txt"
    "$HOME/Desktop/passwords.txt"
    "$WORKSPACE/passwords.txt"
    "$WORKSPACE/.env"
)

for loc in "${CRED_LOCATIONS[@]}"; do
    if [[ -f "$loc" ]]; then
        check_warn "Plaintext credentials file found: $loc"
        # Check permissions
        perms=$(stat -f '%Lp' "$loc" 2>/dev/null || stat -c '%a' "$loc" 2>/dev/null)
        other_read=$((perms % 10))
        if [[ $other_read -ge 4 ]]; then
            if [[ "$FIX_MODE" == "true" ]]; then
                chmod 600 "$loc"
                check_fixed "Set $loc to 600"
            else
                check_fail "$loc is world-readable ($perms)"
            fi
        fi
    fi
done

echo ""

# ==========================================================================
echo -e "${BOLD}[6/6] Workspace Hygiene${RESET}"
# ==========================================================================

# Check for executable files that shouldn't be
if [[ -d "$WORKSPACE/skills" ]]; then
    suspicious_exec=$(find "$WORKSPACE/skills" -name "*.md" -perm +111 2>/dev/null | head -5)
    if [[ -n "$suspicious_exec" ]]; then
        while IFS= read -r f; do
            if [[ "$FIX_MODE" == "true" ]]; then
                chmod -x "$f"
                check_fixed "Removed execute bit from $f"
            else
                check_warn "Markdown file has execute permission: $f"
            fi
        done <<< "$suspicious_exec"
    else
        check_pass "No markdown files with execute permissions"
    fi
fi

# Check for memory directory
if [[ -d "$WORKSPACE/memory" ]]; then
    check_pass "Memory directory exists"
    # Check permissions
    perms=$(stat -f '%Lp' "$WORKSPACE/memory" 2>/dev/null || stat -c '%a' "$WORKSPACE/memory" 2>/dev/null)
    other_read=$((perms % 10))
    if [[ $other_read -ge 4 ]]; then
        if [[ "$FIX_MODE" == "true" ]]; then
            chmod 700 "$WORKSPACE/memory"
            check_fixed "Memory directory set to 700"
        else
            check_warn "Memory directory is world-readable"
        fi
    fi
fi

# Check for security incidents log directory
if [[ ! -d "$WORKSPACE/memory" ]]; then
    check_warn "No memory/ directory for security incident logging"
    if [[ "$FIX_MODE" == "true" ]]; then
        mkdir -p "$WORKSPACE/memory"
        check_fixed "Created memory/ directory"
    fi
fi

echo ""

# ==========================================================================
# Summary
# ==========================================================================
echo -e "${BOLD}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${RESET}"
echo -e "${BOLD}Hardening Complete${RESET}"
echo -e "  ${RED}Issues found:${RESET} ${ISSUES}"
if [[ "$FIX_MODE" == "true" ]]; then
    echo -e "  ${GREEN}Issues fixed:${RESET} ${FIXED}"
fi
echo ""

if [[ $ISSUES -eq 0 ]] || [[ "$FIX_MODE" == "true" && $FIXED -ge $ISSUES ]]; then
    echo -e "  ${GREEN}${BOLD}✓ Workspace is hardened.${RESET}"
else
    echo -e "  ${YELLOW}Run with --fix to auto-remediate safe issues.${RESET}"
fi
echo ""

exit $((ISSUES > 0 ? 1 : 0))
