#!/usr/bin/env bash
set -uo pipefail

# =============================================================================
# audit-outbound.sh — Outbound data flow auditor for OpenClaw
# Detects URLs, network commands, and data transmission patterns in skills.
# =============================================================================

# --- Colors ---
RED='\033[0;31m'
YELLOW='\033[0;33m'
BLUE='\033[0;34m'
GREEN='\033[0;32m'
BOLD='\033[1m'
DIM='\033[2m'
RESET='\033[0m'

# --- Counters ---
SUSPICIOUS_COUNT=0
WHITELISTED_COUNT=0
FILES_SCANNED=0

# --- Options ---
SCAN_PATH=""
SHOW_WHITELIST=false
ADD_WHITELIST=""

# --- Whitelist of safe domains ---
WHITELIST_FILE="$HOME/.openclaw/security/domain-whitelist.txt"

# Default safe domains
DEFAULT_WHITELIST=(
    "github.com"
    "githubusercontent.com"
    "raw.githubusercontent.com"
    "docs.openclaw.ai"
    "openclaw.ai"
    "clawhub.com"
    "npmjs.com"
    "registry.npmjs.org"
    "pypi.org"
    "crates.io"
    "brew.sh"
    "formulae.brew.sh"
    "api.github.com"
    "stackoverflow.com"
    "developer.mozilla.org"
    "wikipedia.org"
    "anthropic.com"
    "docs.anthropic.com"
    "openai.com"
    "platform.openai.com"
    "google.com"
    "googleapis.com"
    "apple.com"
    "developer.apple.com"
    "localhost"
    "127.0.0.1"
    "0.0.0.0"
)

usage() {
    cat <<EOF
${BOLD}audit-outbound.sh${RESET} — Audit outbound data flow patterns in skills

${BOLD}USAGE:${RESET}
    ./audit-outbound.sh [OPTIONS]

${BOLD}OPTIONS:${RESET}
    --path <dir>           Scan only the specified directory
    --show-whitelist       Show whitelisted domains
    --whitelist <domain>   Add a domain to the whitelist
    --help                 Show this help message

${BOLD}EXAMPLES:${RESET}
    ./audit-outbound.sh                          # Audit all skills
    ./audit-outbound.sh --path ./my-skills/      # Audit specific directory
    ./audit-outbound.sh --show-whitelist          # Show safe domains
    ./audit-outbound.sh --whitelist myapi.com     # Add to whitelist
EOF
    exit 0
}

# --- Parse args ---
while [[ $# -gt 0 ]]; do
    case "$1" in
        --path) SCAN_PATH="$2"; shift 2 ;;
        --show-whitelist) SHOW_WHITELIST=true; shift ;;
        --whitelist) ADD_WHITELIST="$2"; shift 2 ;;
        --help) usage ;;
        *) echo "Unknown option: $1"; usage ;;
    esac
done

# --- Load whitelist ---
load_whitelist() {
    local domains=("${DEFAULT_WHITELIST[@]}")
    if [[ -f "$WHITELIST_FILE" ]]; then
        while IFS= read -r domain; do
            [[ -z "$domain" || "$domain" == \#* ]] && continue
            domains+=("$domain")
        done < "$WHITELIST_FILE"
    fi
    echo "${domains[@]}"
}

WHITELIST_DOMAINS=($(load_whitelist))

# --- Show whitelist ---
if [[ "$SHOW_WHITELIST" == "true" ]]; then
    echo -e "${BOLD}Whitelisted Domains${RESET}"
    echo ""
    echo -e "${DIM}Default:${RESET}"
    for d in "${DEFAULT_WHITELIST[@]}"; do
        echo -e "  ${GREEN}✓${RESET} $d"
    done
    if [[ -f "$WHITELIST_FILE" ]]; then
        echo ""
        echo -e "${DIM}Custom (${WHITELIST_FILE}):${RESET}"
        while IFS= read -r domain; do
            [[ -z "$domain" || "$domain" == \#* ]] && continue
            echo -e "  ${GREEN}✓${RESET} $domain"
        done < "$WHITELIST_FILE"
    fi
    exit 0
fi

# --- Add to whitelist ---
if [[ -n "$ADD_WHITELIST" ]]; then
    mkdir -p "$(dirname "$WHITELIST_FILE")"
    echo "$ADD_WHITELIST" >> "$WHITELIST_FILE"
    echo -e "${GREEN}✓ Added '${ADD_WHITELIST}' to whitelist${RESET}"
    exit 0
fi

# --- Check if domain is whitelisted ---
is_whitelisted() {
    local url="$1"
    for domain in "${WHITELIST_DOMAINS[@]}"; do
        if echo "$url" | grep -qi "$domain"; then
            return 0
        fi
    done
    return 1
}

# --- Collect skill directories ---
SKILL_DIRS=()
if [[ -n "$SCAN_PATH" ]]; then
    SKILL_DIRS+=("$SCAN_PATH")
else
    for dir in "./skills" "$HOME/.openclaw/skills" "$HOME/openclaw/skills"; do
        if [[ -d "$dir" ]]; then
            local_abs=$(cd "$dir" && pwd)
            skip=false
            for existing in "${SKILL_DIRS[@]:-}"; do
                if [[ -n "$existing" ]]; then
                    existing_abs=$(cd "$existing" && pwd)
                    [[ "$local_abs" == "$existing_abs" ]] && skip=true && break
                fi
            done
            [[ "$skip" == "false" ]] && SKILL_DIRS+=("$dir")
        fi
    done
fi

if [[ ${#SKILL_DIRS[@]} -eq 0 ]]; then
    echo -e "${YELLOW}No skill directories found to audit.${RESET}"
    exit 0
fi

# --- Report ---
report_outbound() {
    local severity="$1"
    local file="$2"
    local line_num="$3"
    local category="$4"
    local detail="$5"
    local matched="${6:-}"

    ((SUSPICIOUS_COUNT++)) || true

    case "$severity" in
        HIGH)
            echo -e "  ${RED}${BOLD}[HIGH]${RESET}    ${category}"
            ;;
        MEDIUM)
            echo -e "  ${YELLOW}${BOLD}[MEDIUM]${RESET}  ${category}"
            ;;
        LOW)
            echo -e "  ${BLUE}${BOLD}[LOW]${RESET}     ${category}"
            ;;
    esac
    echo -e "    ${DIM}File:${RESET} ${file}:${line_num}"
    echo -e "    ${DIM}Detail:${RESET} ${detail}"
    [[ -n "$matched" ]] && echo -e "    ${DIM}Match:${RESET} ${matched:0:120}"
    echo ""
}

# --- Scan file for outbound patterns ---
scan_outbound() {
    local file="$1"
    ((FILES_SCANNED++)) || true

    if file "$file" | grep -q "binary"; then
        return
    fi

    local content
    content=$(cat "$file" 2>/dev/null) || return

    # Skip the security hardening skill itself
    if [[ "$file" == *"openclaw-security-hardening"* ]]; then
        return
    fi

    # --- URLs ---
    while IFS= read -r match; do
        [[ -z "$match" ]] && continue
        local line_num
        line_num=$(echo "$match" | cut -d: -f1)
        local line_content
        line_content=$(echo "$match" | cut -d: -f2-)

        # Extract URLs from the line
        local urls
        urls=$(echo "$line_content" | grep -oE 'https?://[^ "'"'"')<>]+' || true)

        while IFS= read -r url; do
            [[ -z "$url" ]] && continue
            if is_whitelisted "$url"; then
                ((WHITELISTED_COUNT++)) || true
            else
                report_outbound "HIGH" "$file" "$line_num" "External URL" \
                    "Non-whitelisted URL found in skill file" "$url"
            fi
        done <<< "$urls"
    done < <(echo "$content" | grep -nE 'https?://' || true)

    # --- Network commands ---
    local -a net_patterns=(
        'curl '
        'curl\t'
        'wget '
        'wget\t'
        'web_fetch'
        'browser.*navigate'
        'fetch\('
        'axios\.'
        'requests\.(get|post|put)'
        'http\.request'
        'urllib'
    )

    for pattern in "${net_patterns[@]}"; do
        local match
        match=$(echo "$content" | grep -inE "$pattern" | head -3)
        if [[ -n "$match" ]]; then
            local line_num
            line_num=$(echo "$match" | head -1 | cut -d: -f1)
            local matched_line
            matched_line=$(echo "$match" | head -1 | cut -d: -f2- | head -c 120)
            report_outbound "MEDIUM" "$file" "$line_num" "Network Command" \
                "Contains network/fetch command: matches '$pattern'" "$matched_line"
        fi
    done

    # --- Email/message/webhook sending ---
    local -a send_patterns=(
        'send.*email'
        'send.*message.*to'
        'send.*webhook'
        'post.*webhook'
        'notify.*external'
        'smtp'
        'sendgrid'
        'mailgun'
        'twilio.*send'
        'slack.*webhook'
        'discord.*webhook'
    )

    for pattern in "${send_patterns[@]}"; do
        local match
        match=$(echo "$content" | grep -inE "$pattern" | head -1)
        if [[ -n "$match" ]]; then
            local line_num
            line_num=$(echo "$match" | cut -d: -f1)
            local matched_line
            matched_line=$(echo "$match" | cut -d: -f2- | head -c 120)
            report_outbound "HIGH" "$file" "$line_num" "Data Sending" \
                "Contains data sending pattern" "$matched_line"
        fi
    done

    # --- Raw IP addresses ---
    local ip_match
    ip_match=$(echo "$content" | grep -nE '\b([0-9]{1,3}\.){3}[0-9]{1,3}\b' | head -3)
    if [[ -n "$ip_match" ]]; then
        while IFS= read -r line; do
            [[ -z "$line" ]] && continue
            local line_num
            line_num=$(echo "$line" | cut -d: -f1)
            local ip
            ip=$(echo "$line" | grep -oE '\b([0-9]{1,3}\.){3}[0-9]{1,3}\b' | head -1)

            # Skip common safe IPs
            case "$ip" in
                127.0.0.1|0.0.0.0|192.168.*|10.*|172.1[6-9].*|172.2[0-9].*|172.3[0-1].*)
                    continue ;;
            esac

            report_outbound "MEDIUM" "$file" "$line_num" "IP Address" \
                "Contains non-private IP address" "$ip"
        done <<< "$ip_match"
    fi
}

# --- Main ---
echo ""
echo -e "${BOLD}╔══════════════════════════════════════════════════════════╗${RESET}"
echo -e "${BOLD}║     OpenClaw Security — Outbound Data Flow Audit        ║${RESET}"
echo -e "${BOLD}╚══════════════════════════════════════════════════════════╝${RESET}"
echo ""
echo -e "${DIM}Scanning directories:${RESET}"
for dir in "${SKILL_DIRS[@]}"; do
    echo -e "  ${DIM}→${RESET} $dir"
done
echo ""

for dir in "${SKILL_DIRS[@]}"; do
    [[ ! -d "$dir" ]] && continue
    while IFS= read -r -d '' file; do
        case "$file" in
            *.md|*.txt|*.yaml|*.yml|*.json|*.sh|*.bash|*.js|*.mjs|*.py|*.ts|*.toml)
                scan_outbound "$file"
                ;;
        esac
    done < <(find "$dir" -not -path "*/node_modules/*" -not -path "*/.git/*" -not -path "*/.venv/*" -not -path "*/venv/*" -not -path "*/__pycache__/*" -type f -print0 2>/dev/null)
done

# --- Summary ---
echo -e "${BOLD}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${RESET}"
echo -e "${BOLD}Audit Complete${RESET}"
echo -e "  Files scanned:   ${FILES_SCANNED}"
echo -e "  ${RED}${BOLD}Suspicious:${RESET}      ${SUSPICIOUS_COUNT}"
echo -e "  ${GREEN}Whitelisted:${RESET}     ${WHITELISTED_COUNT}"
echo ""

if [[ $SUSPICIOUS_COUNT -gt 0 ]]; then
    echo -e "  ${YELLOW}Review suspicious outbound patterns above.${RESET}"
    echo -e "  ${DIM}Add safe domains with: --whitelist <domain>${RESET}"
    exit 1
else
    echo -e "  ${GREEN}✓ No suspicious outbound patterns found.${RESET}"
    exit 0
fi
