---
name: openclaw-security-hardening
description: Protect OpenClaw installations from prompt injection, data exfiltration, malicious skills, and workspace tampering
version: 1.0.0
author: openclaw-community
tags: [security, hardening, audit, protection]
---

# OpenClaw Security Hardening

A comprehensive security toolkit for protecting OpenClaw installations from attacks via malicious skill files, prompt injection, data exfiltration, and workspace tampering.

## Threat Model

This skill protects against:

| Threat | Description | Tool |
|--------|-------------|------|
| **Prompt Injection** | Malicious skills containing instructions to override system prompts, ignore safety rules, or manipulate agent behavior | `scan-skills.sh` |
| **Data Exfiltration** | Skills that instruct the agent to send sensitive data (credentials, memory, config) to external endpoints | `audit-outbound.sh` |
| **Skill Tampering** | Unauthorized modification of installed skills after initial review | `integrity-check.sh` |
| **Workspace Exposure** | Sensitive files with wrong permissions, missing .gitignore rules, insecure gateway config | `harden-workspace.sh` |
| **Supply Chain** | Installing a new skill that contains hidden malicious patterns | `install-guard.sh` |

## Quick Start

```bash
# Run a full security scan of all installed skills
./scripts/scan-skills.sh

# Audit outbound data flow patterns
./scripts/audit-outbound.sh

# Initialize integrity baseline
./scripts/integrity-check.sh --init

# Harden your workspace
./scripts/harden-workspace.sh --fix

# Check a new skill before installing
./scripts/install-guard.sh /path/to/new-skill/
```

## Tools

### 1. `scan-skills.sh` — Skill File Scanner

Scans all installed skill files for malicious patterns including prompt injection, data exfiltration attempts, suspicious URLs, hidden unicode, obfuscated commands, and social engineering.

**Usage:**
```bash
# Scan all skill directories
./scripts/scan-skills.sh

# Scan a specific directory only
./scripts/scan-skills.sh --path /path/to/skills/

# Output as JSON for automation
./scripts/scan-skills.sh --json

# Show help
./scripts/scan-skills.sh --help
```

**What it detects:**
- Prompt injection patterns (override instructions, new system prompts, admin overrides)
- Data exfiltration (curl/wget to external URLs, sending file contents)
- Suspicious URLs (webhooks, pastebin, requestbin, ngrok, etc.)
- Base64-encoded content that could hide instructions
- Hidden unicode characters (zero-width spaces, RTL override, homoglyphs)
- References to sensitive files (.env, credentials, API keys, tokens)
- Instructions to modify system files (AGENTS.md, SOUL.md)
- Obfuscated commands (hex encoded, unicode escaped)
- Social engineering ("don't tell the user", "secretly", "without mentioning")

**Severity levels:**
- 🔴 **CRITICAL** — Likely malicious, immediate action needed
- 🟡 **WARNING** — Suspicious, review manually
- 🔵 **INFO** — Noteworthy but probably benign

---

### 2. `integrity-check.sh` — Skill Integrity Monitor

Creates SHA256 hash baselines of all skill files and detects unauthorized modifications.

**Usage:**
```bash
# Initialize baseline (first run)
./scripts/integrity-check.sh --init

# Check for changes (run periodically)
./scripts/integrity-check.sh

# Update baseline after reviewing changes
./scripts/integrity-check.sh --update

# Check specific directory
./scripts/integrity-check.sh --path /path/to/skills/

# Show help
./scripts/integrity-check.sh --help
```

**Reports:**
- ✅ Unchanged files
- ⚠️ Modified files (hash mismatch)
- 🆕 New files (not in baseline)
- ❌ Removed files (in baseline but missing)

**Automation:** Add to your heartbeat or cron to run daily:
```bash
# In HEARTBEAT.md or cron
0 8 * * * /path/to/scripts/integrity-check.sh 2>&1 | grep -E '(MODIFIED|NEW|REMOVED)'
```

---

### 3. `audit-outbound.sh` — Outbound Data Flow Auditor

Scans skill files for patterns that could cause data to leave your machine.

**Usage:**
```bash
# Audit all skills
./scripts/audit-outbound.sh

# Audit specific directory
./scripts/audit-outbound.sh --path /path/to/skills/

# Show whitelisted domains
./scripts/audit-outbound.sh --show-whitelist

# Add domain to whitelist
./scripts/audit-outbound.sh --whitelist example.com

# Show help
./scripts/audit-outbound.sh --help
```

**Detects:**
- HTTP/HTTPS URLs embedded in skill instructions
- References to curl, wget, fetch, web_fetch, browser navigate
- Email/message/webhook sending instructions
- Raw IP addresses in instructions
- Non-whitelisted external domains

---

### 4. `harden-workspace.sh` — Workspace Hardener

Checks and fixes common security misconfigurations in your OpenClaw workspace.

**Usage:**
```bash
# Check only (report issues)
./scripts/harden-workspace.sh

# Auto-fix safe issues
./scripts/harden-workspace.sh --fix

# Show help
./scripts/harden-workspace.sh --help
```

**Checks:**
- File permissions on sensitive files (MEMORY.md, USER.md, SOUL.md, credentials)
- .gitignore coverage for sensitive patterns
- Gateway auth configuration
- DM policy settings
- Sensitive content in version-controlled files

---

### 5. `install-guard.sh` — Pre-Install Security Gate

Run before installing any new skill to check for malicious content.

**Usage:**
```bash
# Check a skill before installing
./scripts/install-guard.sh /path/to/new-skill/

# Strict mode (fail on warnings too)
./scripts/install-guard.sh --strict /path/to/new-skill/

# Show help
./scripts/install-guard.sh --help
```

**Checks:**
- All patterns from scan-skills.sh
- Dangerous shell patterns in scripts (rm -rf, curl|bash, eval, etc.)
- Suspicious npm dependencies (if package.json exists)
- Exit code 0 = safe, 1 = suspicious (for CI/automation)

---

## Security Rules Template

Copy `assets/security-rules-template.md` into your `AGENTS.md` to add runtime security rules for your agent. These rules instruct the agent to refuse prompt injection attempts and protect sensitive data.

```bash
cat assets/security-rules-template.md >> /path/to/AGENTS.md
```

## Recommended Setup

1. **Initial setup:**
   ```bash
   ./scripts/scan-skills.sh              # Scan existing skills
   ./scripts/audit-outbound.sh           # Audit outbound patterns
   ./scripts/integrity-check.sh --init   # Create baseline
   ./scripts/harden-workspace.sh --fix   # Fix workspace issues
   ```

2. **Add security rules to AGENTS.md** from the template

3. **Before installing new skills:**
   ```bash
   ./scripts/install-guard.sh /path/to/new-skill/
   ```

4. **Periodic checks** (add to heartbeat or cron):
   ```bash
   ./scripts/integrity-check.sh          # Detect tampering
   ./scripts/scan-skills.sh              # Re-scan for new patterns
   ```
