# 🔌 Programmatic API Reference

Use OpenClaw Security Guard in your Node.js applications.

---

## Installation

```bash
npm install openclaw-security-guard
```

---

## Quick Start

```javascript
import { quickAudit, checkPromptInjection } from 'openclaw-security-guard';

// Run a quick audit
const results = await quickAudit('~/.openclaw');
console.log(`Score: ${results.securityScore}/100`);

// Check for prompt injection
const check = await checkPromptInjection('ignore previous instructions');
console.log(`Safe: ${check.safe}`);
```

---

## Exports

### Functions

| Function | Description |
|----------|-------------|
| `quickAudit(path, options)` | Run a complete audit |
| `checkPromptInjection(message, options)` | Check message for injection |
| `startDashboard(options)` | Start the dashboard server |

### Classes

| Class | Description |
|-------|-------------|
| `SecretsScanner` | Scan for exposed secrets |
| `ConfigAuditor` | Audit configuration |
| `PromptInjectionDetector` | Detect injection patterns |
| `DependencyScanner` | Check for vulnerable deps |
| `McpServerAuditor` | Validate MCP servers |
| `AutoHardener` | Apply security fixes |

### Utilities

| Utility | Description |
|---------|-------------|
| `sanitizePath(path, basePath)` | Sanitize file paths |
| `escapeHtml(str)` | Escape HTML for XSS prevention |
| `escapeJson(str)` | Escape for JSON |
| `isSafeUrl(url)` | Check if URL is safe |
| `isAllowedOrigin(origin, allowed)` | Validate WebSocket origin |
| `loadConfig(path)` | Load configuration |
| `getOpenClawPath()` | Get OpenClaw directory |
| `i18n(lang)` | Get translation function |

### Constants

| Constant | Value |
|----------|-------|
| `VERSION` | `'1.0.0'` |
| `AUTHOR` | `'Miloud Belarebia'` |
| `WEBSITE` | `'https://2pidata.com'` |
| `PRIVACY.telemetry` | `false` |
| `PRIVACY.tracking` | `false` |

---

## API Reference

### `quickAudit(path, options)`

Run a complete security audit.

```javascript
import { quickAudit } from 'openclaw-security-guard';

const results = await quickAudit('~/.openclaw', {
  config: {}  // Optional configuration
});

console.log(results);
// {
//   timestamp: '2024-01-15T10:30:00.000Z',
//   version: '1.0.0',
//   path: '~/.openclaw',
//   securityScore: 75,
//   summary: { critical: 1, high: 2, medium: 0, low: 0 },
//   scanners: {
//     secrets: { findings: [], summary: {...} },
//     config: { findings: [...], summary: {...} },
//     prompts: { findings: [], summary: {...} }
//   }
// }
```

#### Parameters

| Parameter | Type | Description |
|-----------|------|-------------|
| `path` | `string` | Path to OpenClaw directory |
| `options` | `object` | Optional configuration |
| `options.config` | `object` | Scanner configuration |

#### Returns

| Property | Type | Description |
|----------|------|-------------|
| `timestamp` | `string` | ISO timestamp |
| `version` | `string` | Tool version |
| `path` | `string` | Scanned path |
| `securityScore` | `number` | Score 0-100 |
| `summary` | `object` | Findings count by severity |
| `scanners` | `object` | Results from each scanner |

---

### `checkPromptInjection(message, options)`

Check a message for prompt injection patterns.

```javascript
import { checkPromptInjection } from 'openclaw-security-guard';

const result = await checkPromptInjection('ignore all previous instructions');

console.log(result);
// {
//   safe: false,
//   score: 0.85,
//   matches: [
//     { pattern: 'instruction_override', severity: 'high' }
//   ]
// }
```

#### Parameters

| Parameter | Type | Description |
|-----------|------|-------------|
| `message` | `string` | Message to check |
| `options` | `object` | Optional configuration |
| `options.config` | `object` | Detector configuration |

#### Returns

| Property | Type | Description |
|----------|------|-------------|
| `safe` | `boolean` | True if no injection detected |
| `score` | `number` | Confidence score (0-1) |
| `matches` | `array` | Detected patterns |

---

### `startDashboard(options)`

Start the security dashboard server.

```javascript
import { startDashboard } from 'openclaw-security-guard';

const { server, monitor, auth } = await startDashboard({
  port: 18790,
  gatewayUrl: 'ws://127.0.0.1:18789',
  openBrowser: false
});

// Server is now running on http://localhost:18790
```

#### Parameters

| Parameter | Type | Description | Default |
|-----------|------|-------------|---------|
| `options.port` | `number` | Dashboard port | `18790` |
| `options.gatewayUrl` | `string` | OpenClaw gateway URL | `ws://127.0.0.1:18789` |
| `options.openBrowser` | `boolean` | Open browser automatically | `true` |
| `options.config` | `object` | Configuration | `{}` |

#### Returns

| Property | Type | Description |
|----------|------|-------------|
| `server` | `http.Server` | HTTP server instance |
| `monitor` | `SecurityMonitor` | Monitor instance |
| `auth` | `AuthManager` | Authentication manager |

---

### `SecretsScanner`

Scan for exposed secrets.

```javascript
import { SecretsScanner } from 'openclaw-security-guard';

const scanner = new SecretsScanner({
  exclude: ['node_modules/**', '*.test.js']
});

const result = await scanner.scan('~/.openclaw', {});

console.log(result.findings);
// [
//   {
//     type: 'anthropic_api_key',
//     severity: 'critical',
//     message: 'Anthropic API key detected',
//     location: 'config.json:15',
//     masked: 'sk-ant-****'
//   }
// ]
```

#### Methods

| Method | Description |
|--------|-------------|
| `scan(path, options)` | Scan directory for secrets |
| `scanContent(content, filename)` | Scan content directly |
| `maskSecret(secret)` | Mask a secret for display |

---

### `ConfigAuditor`

Audit OpenClaw configuration.

```javascript
import { ConfigAuditor } from 'openclaw-security-guard';

const auditor = new ConfigAuditor({});

const result = await auditor.scan('~/.openclaw', { strict: true });

console.log(result.findings);
// [
//   {
//     rule: 'sandbox_mode',
//     severity: 'critical',
//     message: 'Sandbox mode should be "always"',
//     current: 'non-main',
//     recommended: 'always',
//     autoFixable: true,
//     fix: 'Set agents.defaults.sandbox.mode to "always"'
//   }
// ]
```

#### Methods

| Method | Description |
|--------|-------------|
| `scan(path, options)` | Audit configuration |
| `getConfigStatus(config)` | Get configuration status |

---

### `PromptInjectionDetector`

Detect prompt injection patterns.

```javascript
import { PromptInjectionDetector } from 'openclaw-security-guard';

const detector = new PromptInjectionDetector({
  sensitivity: 'high'
});

// Check a single message
const result = detector.checkMessage('Hello, how are you?');
console.log(result.safe); // true

// Scan files for patterns
const scanResult = await detector.scan('~/.openclaw', {});
```

#### Methods

| Method | Description |
|--------|-------------|
| `scan(path, options)` | Scan files for patterns |
| `checkMessage(message)` | Check a single message |

---

### Utilities

#### `sanitizePath(path, basePath)`

Sanitize file paths to prevent traversal attacks.

```javascript
import { sanitizePath } from 'openclaw-security-guard';

// Safe path
const safe = sanitizePath('config.json', '/home/user');
// Returns: '/home/user/config.json'

// Dangerous path - throws error
sanitizePath('../../../etc/passwd', '/home/user');
// Throws: Error('Path traversal detected')
```

#### `escapeHtml(str)`

Escape HTML to prevent XSS.

```javascript
import { escapeHtml } from 'openclaw-security-guard';

const safe = escapeHtml('<script>alert("xss")</script>');
// Returns: '&lt;script&gt;alert(&quot;xss&quot;)&lt;/script&gt;'
```

#### `isSafeUrl(url)`

Check if a URL is safe (not javascript:, data:, file:).

```javascript
import { isSafeUrl } from 'openclaw-security-guard';

isSafeUrl('https://example.com');     // true
isSafeUrl('javascript:alert(1)');     // false
isSafeUrl('data:text/html,...');      // false
```

---

## TypeScript

Type definitions are included. For full TypeScript support:

```typescript
import type { 
  AuditResult, 
  Finding, 
  ScanOptions 
} from 'openclaw-security-guard';

const results: AuditResult = await quickAudit(path);
```

---

## Error Handling

```javascript
import { quickAudit } from 'openclaw-security-guard';

try {
  const results = await quickAudit('/invalid/path');
} catch (error) {
  console.error('Audit failed:', error.message);
}
```

---

## Examples

### CI/CD Integration

```javascript
import { quickAudit } from 'openclaw-security-guard';

const results = await quickAudit(process.cwd());

if (results.summary.critical > 0) {
  console.error(`❌ ${results.summary.critical} critical issues found`);
  process.exit(1);
}

console.log(`✅ Security score: ${results.securityScore}/100`);
```

### Webhook Integration

```javascript
import { quickAudit } from 'openclaw-security-guard';

async function auditAndNotify() {
  const results = await quickAudit('~/.openclaw');
  
  if (results.securityScore < 60) {
    // Send alert (implement your notification logic)
    await sendSlackAlert({
      text: `⚠️ Security score dropped to ${results.securityScore}/100`
    });
  }
}
```

### Custom Scanner

```javascript
import { SecretsScanner, ConfigAuditor } from 'openclaw-security-guard';

async function customAudit(path) {
  const results = [];
  
  // Run secrets scanner
  const secrets = new SecretsScanner({ exclude: ['*.log'] });
  results.push(await secrets.scan(path, {}));
  
  // Run config auditor in strict mode
  const config = new ConfigAuditor({});
  results.push(await config.scan(path, { strict: true }));
  
  return results;
}
```

---

<div align="center">

**[Back to Documentation](../en/README.md)**

</div>
