---
name: openclaw-profanity
description: Content moderation plugin for OpenClaw/Moltbot AI agents. Use when building chatbots that need profanity filtering, moderating user messages in Discord/Slack/Telegram bots, or adding content moderation to OpenClaw agents.
---

# OpenClaw Profanity Plugin

Profanity detection plugin for OpenClaw and Moltbot AI agents. Adds automated content moderation to your chatbot with leetspeak, Unicode, and multi-language support.

## Installation

```bash
npm install openclaw-profanity
```

## Setup with OpenClaw

```javascript
import { OpenClaw } from 'openclaw';
import { profanityPlugin } from 'openclaw-profanity';

const bot = new OpenClaw({
  plugins: [
    profanityPlugin({
      action: 'warn',              // warn | censor | block | log
      detectLeetspeak: true,
      normalizeUnicode: true,
      languages: ['english'],
      customWords: [],
      ignoreWords: []
    })
  ]
});
```

## Configuration Options

| Option | Type | Default | Description |
|--------|------|---------|-------------|
| `action` | string | `'warn'` | Action on profanity: `warn`, `censor`, `block`, `log` |
| `detectLeetspeak` | boolean | `true` | Catch `f4ck`, `sh1t` patterns |
| `normalizeUnicode` | boolean | `true` | Catch Cyrillic lookalikes |
| `languages` | array | `['english']` | Languages to check |
| `customWords` | array | `[]` | Additional words to flag |
| `ignoreWords` | array | `[]` | Words to whitelist |
| `onViolation` | function | - | Custom handler for violations |

## Actions

### `warn` - Respond with warning
```javascript
profanityPlugin({ action: 'warn' })
// Bot responds: "Please keep the chat clean."
```

### `censor` - Replace and continue
```javascript
profanityPlugin({ action: 'censor', replaceWith: '***' })
// "What the ***" is processed normally
```

### `block` - Ignore message entirely
```javascript
profanityPlugin({ action: 'block' })
// Message is not processed
```

### `log` - Log and continue
```javascript
profanityPlugin({ action: 'log' })
// Logs violation, processes normally
```

## Custom Violation Handler

```javascript
profanityPlugin({
  action: 'custom',
  onViolation: async (message, result, context) => {
    // Track repeat offenders
    await trackViolation(message.userId, result.profaneWords);

    // Custom response
    if (getViolationCount(message.userId) > 3) {
      await banUser(message.userId);
      return { blocked: true };
    }

    return { blocked: false, warning: "First warning..." };
  }
})
```

## Platform Examples

### Discord Bot
```javascript
const bot = new OpenClaw({
  platform: 'discord',
  plugins: [
    profanityPlugin({
      action: 'censor',
      detectLeetspeak: true,
      languages: ['english', 'spanish']
    })
  ]
});
```

### Telegram Bot
```javascript
const bot = new OpenClaw({
  platform: 'telegram',
  plugins: [
    profanityPlugin({
      action: 'warn',
      onViolation: (msg, result) => {
        return {
          reply: `Watch your language, ${msg.username}!`,
          deleteOriginal: true
        };
      }
    })
  ]
});
```

### Slack Bot
```javascript
const bot = new OpenClaw({
  platform: 'slack',
  plugins: [
    profanityPlugin({
      action: 'log',
      onViolation: (msg, result) => {
        notifyModerators(msg.channel, msg.user, result);
      }
    })
  ]
});
```

## Detection Capabilities

The plugin catches:

- **Direct profanity**: Standard bad words
- **Leetspeak**: `f4ck`, `sh1t`, `@$$`, `b1tch`
- **Unicode tricks**: Cyrillic `а` instead of `a`, etc.
- **Spaced letters**: `f u c k`, `s.h.i.t`
- **Mixed obfuscation**: `fü©k`, `$h!t`

## Resources

- npm: https://www.npmjs.com/package/openclaw-profanity
- GitHub: https://github.com/GLINCKER/glin-profanity/tree/release/packages/openclaw
- Core library docs: https://www.typeweaver.com/docs/glin-profanity
