#!/usr/bin/env python3
"""
Set up an Obsidian task board with Kanban and Dataview support.

Usage:
    python3 setup.py <vault-path> [--folder Tasks] [--columns Backlog,Todo,In Progress,Review,Done]
"""

import argparse
import os
import sys


def create_board(folder_path: str, columns: list[str]) -> str:
    """Generate Kanban board markdown."""
    lines = [
        "---\n",
        "\n",
        "kanban-plugin: basic\n",
        "\n",
        "---\n",
        "\n",
    ]
    for col in columns:
        lines.append(f"## {col}\n\n\n")

    lines.append('\n%% kanban:settings\n')
    lines.append('```\n')
    lines.append('{"kanban-plugin":"basic","lane-width":272,"show-checkboxes":true}\n')
    lines.append('```\n')
    lines.append('%%\n')

    return "".join(lines)


def create_dashboard(tasks_folder: str) -> str:
    """Generate Dataview dashboard markdown."""
    return f"""# Task Dashboard

## 🔴 Urgent (P1)
```dataview
TABLE status, category, due
FROM "{tasks_folder}"
WHERE priority = "P1" AND status != "done"
SORT due ASC
```

## 🟡 Pipeline (P2)
```dataview
TABLE status, category
FROM "{tasks_folder}"
WHERE priority = "P2" AND status != "done"
SORT status ASC
```

## 🟢 Long Game (P3)
```dataview
TABLE status, category, parked_until as "Parked Until"
FROM "{tasks_folder}"
WHERE priority = "P3" AND status != "done"
SORT status ASC
```

## ⏰ Overdue
```dataview
TABLE priority, category, due
FROM "{tasks_folder}"
WHERE due AND due < date(today) AND status != "done"
SORT due ASC
```

## ✅ Recently Completed
```dataview
TABLE category
FROM "{tasks_folder}"
WHERE status = "done"
SORT file.mtime DESC
LIMIT 10
```
"""


def main():
    parser = argparse.ArgumentParser(description="Set up Obsidian task board")
    parser.add_argument("vault_path", help="Path to Obsidian vault root")
    parser.add_argument("--folder", default="Tasks", help="Subfolder name (default: Tasks)")
    parser.add_argument(
        "--columns",
        default="Backlog,Todo,In Progress,Review,Done",
        help="Comma-separated Kanban columns",
    )
    args = parser.parse_args()

    vault = os.path.expanduser(args.vault_path)
    if not os.path.isdir(vault):
        print(f"Error: vault path '{vault}' does not exist", file=sys.stderr)
        sys.exit(1)

    folder_path = os.path.join(vault, args.folder)
    os.makedirs(folder_path, exist_ok=True)

    columns = [c.strip() for c in args.columns.split(",")]

    # Relative path from vault root for Dataview queries
    tasks_folder = args.folder

    # Create Board.md
    board_path = os.path.join(folder_path, "Board.md")
    if os.path.exists(board_path):
        print(f"Board already exists: {board_path}")
    else:
        with open(board_path, "w") as f:
            f.write(create_board(folder_path, columns))
        print(f"Created: {board_path}")

    # Create Dashboard.md
    dashboard_path = os.path.join(folder_path, "Dashboard.md")
    if os.path.exists(dashboard_path):
        print(f"Dashboard already exists: {dashboard_path}")
    else:
        with open(dashboard_path, "w") as f:
            f.write(create_dashboard(tasks_folder))
        print(f"Created: {dashboard_path}")

    print(f"\nTask board ready at: {folder_path}")
    print("\nRequired Obsidian plugins:")
    print("  - Kanban (for Board.md)")
    print("  - Dataview (for Dashboard.md)")
    print("\nInstall via: Settings > Community Plugins > Browse")


if __name__ == "__main__":
    main()
