---
name: OpenClaw Minecraft
version: 0.1.0
specialty: minecraft-control
category: tools
description: Control Minecraft bots through a Mineflayer controller API using JSON actions and cron-driven autonomy.
---

# Mineflayer Controller Skill

Use this skill to create and control Minecraft bots via a controller API. The controller executes only JSON actions. Do not send API keys or other secrets to the controller. The agent performs LLM reasoning locally and only sends actions.

## Register First (Open Registration)
If open registration is enabled, the agent can self-issue a token:
```bash
curl -sS -X POST "https://56eb-125-246-120-211.ngrok-free.app/v1/auth/open/register" \
  -H "Content-Type: application/json" \
  -d '{"agentName":"andy"}'
```

Recommended: store the response in `memory/mc-auth.json` and reuse the `accessToken` for future calls.

## Requirements
- Base URL: `https://56eb-125-246-120-211.ngrok-free.app/v1`
- `MC_CONTROLLER_TOKEN` (agent-specific JWT)

To obtain a JWT, an operator should call `POST /v1/auth/register` with the master issuer secret and share the returned `accessToken` with the agent. Refresh tokens can be rotated via `POST /v1/auth/refresh`.

If you don't want to share the master secret, the controller can expose `POST /v1/auth/proxy/register` with allowlist + rate-limit. In that mode, the agent requests an access token using a `proxyKey`.

If open registration is enabled (`POST /v1/auth/open/register`), the agent can request an access token without a proxy key. This is less secure and should only be used when you accept open access.

## Safety Rules
- Never send LLM API keys to the controller.
- Only control bots owned by your token.
- Prefer `queue: queue` unless you must interrupt a running task.

## Workflow
1. Ensure the bot exists; create it if missing.
2. Connect the bot to the Minecraft server.
3. Read state for current context.
4. Send an action request.
5. Poll task status or subscribe to events.

## Autonomous Loop
For autonomous behavior, repeat:
1. `GET /bots/{botId}/state`
2. If `metadata.currentTaskId` is set, wait briefly and loop.
3. Decide a safe action bundle (5 to 10 actions) and send it via `POST /bots/{botId}/act-batch`.
4. Use `mode: until` or `mode: loop` with `timeoutMs` per action.

## Cron Integration
This project uses Gateway cron (not heartbeat) for autonomous loops.

After installing this skill, ensure:
- Overwrite the workspace root `CRON_PROMPT.md` with `skills/openclaw-minecraft/CRON_PROMPT.md` (always, regardless of existing file).
- A cron job runs every 30 seconds and instructs the agent to follow `CRON_PROMPT.md`.

## API Patterns

### Create bot
```bash
curl -sS -X POST "https://56eb-125-246-120-211.ngrok-free.app/v1/bots" \
  -H "Authorization: Bearer $MC_CONTROLLER_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"botId":"<botId>","username":"<botId>"}'
```

### Connect bot
```bash
curl -sS -X POST "https://56eb-125-246-120-211.ngrok-free.app/v1/bots/<botId>/connect" \
  -H "Authorization: Bearer $MC_CONTROLLER_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{"host":"127.0.0.1","port":25565,"version":"1.21.9"}'
```

### Read state
```bash
curl -sS -X GET "https://56eb-125-246-120-211.ngrok-free.app/v1/bots/<botId>/state" \
  -H "Authorization: Bearer $MC_CONTROLLER_TOKEN"
```

### Send batch (loop)
```bash
curl -sS -X POST "https://56eb-125-246-120-211.ngrok-free.app/v1/bots/<botId>/act-batch" \
  -H "Authorization: Bearer $MC_CONTROLLER_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "actions":[
      {
        "action":"chat",
        "params":{"message":"hello"},
        "mode":"loop",
        "intervalMs":2000,
        "maxIterations":3
      }
    ]
  }'
```

### Send batch (until)
```bash
curl -sS -X POST "https://56eb-125-246-120-211.ngrok-free.app/v1/bots/<botId>/act-batch" \
  -H "Authorization: Bearer $MC_CONTROLLER_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "actions":[
      {
        "action":"move_to",
        "params":{"x":10,"y":64,"z":-12},
        "mode":"until",
        "stopCondition":{"type":"reach_position","radius":1.5},
        "timeoutMs":60000
      }
    ]
  }'
```

## Action Guidance
- Convert natural-language goals to a **batch** of JSON actions.
- If the goal requires multiple steps, include them in order in one batch.
- Each batch must include 5 to 10 actions.
- Use `mode: until` for navigation or repeated tasks.
- Use `mode: loop` for periodic actions (e.g., scanning, chat).
- Use only supported actions: `chat`, `move_to`, `move_relative`, `move`, `dig`, `place`, `equip`, `use_item`, `attack`, `follow`, `jump`.

## Known Limitations
- JSON-only payloads for now. Media/attachments are not supported yet.
- Actions are best-effort and may fail if the bot is not connected or lacks items.
