# Troubleshooting Guide

Solutions to common issues with the MCP Integration plugin.

## 🔍 Diagnostic Commands

### Quick Status Check

```bash
# Check OpenClaw status
openclaw status

# Check MCP plugin logs
openclaw logs | grep MCP

# Check plugin directory
ls -la ~/.openclaw/extensions/mcp-integration/

# Test configuration syntax
cat ~/.openclaw/openclaw.json | jq '.plugins.entries["mcp-integration"]'
```

## 🚨 Common Issues

### Issue 1: Plugin Not Loading

**Symptoms:**
```
openclaw status
# mcp-integration: not found
```

**Diagnostic:**
```bash
# Check if plugin exists
ls -la ~/.openclaw/extensions/mcp-integration/

# Check plugin metadata
cat ~/.openclaw/extensions/mcp-integration/config/openclaw.plugin.json

# Check for errors
openclaw logs | grep -i "mcp" | grep -i "error"
```

**Solutions:**

**A. Plugin not installed:**
```bash
# Install plugin
cd ~/.openclaw/extensions/
git clone <repository-url> mcp-integration
cd mcp-integration
npm install

# Restart
openclaw gateway restart
```

**B. Configuration error:**
```bash
# Validate JSON
cat ~/.openclaw/openclaw.json | jq .

# Fix syntax errors, then:
openclaw gateway restart
```

**C. Dependencies missing:**
```bash
cd ~/.openclaw/extensions/mcp-integration/
npm install
openclaw gateway restart
```

### Issue 2: Server Connection Failed

**Symptoms:**
```
[MCP] Failed to connect to kr-legal: fetch failed
```

**Diagnostic:**
```bash
# Test server is running
curl http://localhost:3000/mcp

# Check if port is open
netstat -tuln | grep 3000

# Test server with JSON-RPC
curl -X POST http://localhost:3000/mcp \
  -H "Content-Type: application/json" \
  -d '{"jsonrpc":"2.0","id":1,"method":"tools/list","params":{}}'
```

**Solutions:**

**A. Server not running:**
```bash
# Start your MCP server
node server.js
# or
npm start
```

**B. Wrong URL in config:**
```json
{
  "servers": {
    "kr-legal": {
      "enabled": true,
      "url": "http://localhost:3000/mcp"  // Check port and path
    }
  }
}
```

**C. Firewall blocking:**
```bash
# Allow local connections
sudo ufw allow from 127.0.0.1

# Or allow specific port
sudo ufw allow 3000/tcp
```

**D. Server not responding to MCP protocol:**
```bash
# Test if server responds correctly
curl -v -X POST http://localhost:3000/mcp \
  -H "Content-Type: application/json" \
  -d '{
    "jsonrpc": "2.0",
    "id": 1,
    "method": "tools/list",
    "params": {}
  }'

# Should return JSON-RPC response with tools
```

### Issue 3: Tool Not Found

**Symptoms:**
```
Error: Tool not found: kr-legal:search_statute
```

**Diagnostic:**
```bash
# List available tools in OpenClaw
# In chat: "List MCP tools"

# Check server is connected
openclaw logs | grep "Connected to kr-legal"

# Check what tools were registered
openclaw logs | grep "tools available"
```

**Solutions:**

**A. Server disconnected:**
```bash
# Restart gateway to reconnect
openclaw gateway restart

# Check logs
openclaw logs | grep "Connecting to kr-legal"
openclaw logs | grep "Connected to kr-legal"
```

**B. Tool name mismatch:**
```json
// Correct format: "server:tool"
{
  "server": "kr-legal",
  "tool": "search_statute"  // Not "searchStatute" or "search-statute"
}
```

**C. Server disabled:**
```json
{
  "servers": {
    "kr-legal": {
      "enabled": true  // Must be true
    }
  }
}
```

### Issue 4: Invalid Arguments

**Symptoms:**
```
Error: Invalid arguments for search_statute. Required: query
```

**Diagnostic:**
```bash
# List tools to see schema
# In chat: "List MCP tools"

# Check what parameters are required
```

**Solution:**

Check the tool's `inputSchema` and provide all required parameters:

```json
// If schema shows:
{
  "inputSchema": {
    "type": "object",
    "properties": {
      "query": { "type": "string" },
      "limit": { "type": "number" }
    },
    "required": ["query"]  // ← "query" is required
  }
}

// Then call with:
{
  "action": "call",
  "server": "kr-legal",
  "tool": "search_statute",
  "args": {
    "query": "민법"  // ← Must provide required param
  }
}
```

### Issue 5: HTTP Transport Errors

**Symptoms:**
```
[Transport] POST failed: HTTP 500
[Transport] SSE stream error
```

**Diagnostic:**
```bash
# Enable debug mode
# Edit http-transport.js:
# new StreamableHTTPClientTransport(url, { debug: true })

# Check server logs for errors

# Test with curl
curl -v -X POST http://localhost:3000/mcp \
  -H "Content-Type: application/json" \
  -d '{"jsonrpc":"2.0","id":1,"method":"tools/list"}'
```

**Solutions:**

**A. Server error (500):**
- Check MCP server logs
- Verify server implementation
- Ensure server handles JSON-RPC correctly

**B. Invalid response format:**
```javascript
// Server must return valid JSON-RPC:
{
  "jsonrpc": "2.0",
  "id": 1,  // Must match request ID
  "result": {  // Or "error"
    "tools": [...]
  }
}
```

**C. SSE not supported:**
- GET SSE stream is optional
- Plugin will fall back to POST-only mode
- Error is logged but doesn't affect functionality

### Issue 6: Session ID Issues

**Symptoms:**
```
[Transport] Session ID not persisted
[Transport] Creating new session on each request
```

**Solution:**

This is usually not a problem. The transport automatically manages sessions:
- First request gets a session ID from `mcp-session-id` header
- Subsequent requests use that session ID
- If server doesn't support sessions, each request is independent

### Issue 7: Plugin Crashes OpenClaw

**Symptoms:**
```
openclaw status
# gateway: crashed
```

**Diagnostic:**
```bash
# Check crash logs
openclaw logs | grep -A 20 "crash"

# Check JavaScript errors
openclaw logs | grep "Error:"
openclaw logs | grep "TypeError:"
```

**Solutions:**

**A. Syntax error in plugin code:**
```bash
# Check plugin files
node --check ~/.openclaw/extensions/mcp-integration/index.js
node --check ~/.openclaw/extensions/mcp-integration/http-transport.js
```

**B. Missing dependency:**
```bash
cd ~/.openclaw/extensions/mcp-integration/
npm install
```

**C. Incompatible OpenClaw version:**
```bash
# Check required version
cat ~/.openclaw/extensions/mcp-integration/config/openclaw.plugin.json | jq '.openclaw.minVersion'

# Check installed version
openclaw --version

# Upgrade if needed
openclaw update
```

### Issue 8: Memory Leaks

**Symptoms:**
```
# OpenClaw using excessive memory over time
top -p $(pgrep -f openclaw)
```

**Solutions:**

**A. Connection not closing:**
```javascript
// In MCPManager.disconnect():
for (const [name, { client }] of this.clients.entries()) {
  await client.close();  // ← Must close clients
}
```

**B. SSE streams not cleaned up:**
```javascript
// In http-transport.js close():
if (this._sseAbortController) {
  this._sseAbortController.abort();  // ← Must abort SSE
}
```

**C. Event listeners not removed:**
```javascript
// Remove event listeners before destroying
transport.onmessage = null;
transport.onerror = null;
transport.onclose = null;
```

## 🧪 Debug Mode

### Enable Full Debug Logging

**1. Edit http-transport.js:**
```javascript
const transport = new StreamableHTTPClientTransport(url, { 
  debug: true  // ← Add this
});
```

**2. Restart OpenClaw:**
```bash
openclaw gateway restart
```

**3. View debug logs:**
```bash
openclaw logs | grep -i transport
```

**Debug output:**
```
[Transport] Started
[Transport] Opening GET SSE stream
[Transport] GET SSE stream opened
[Transport] SSE message: {"jsonrpc":"2.0",...}
[Transport] POST request: {"jsonrpc":"2.0","id":1,"method":"tools/list"}
[Transport] POST response: {"jsonrpc":"2.0","id":1,"result":{...}}
```

## 📊 Collecting Diagnostic Info

### Create Diagnostic Report

```bash
#!/bin/bash
# save as: diagnose-mcp.sh

echo "=== System Info ==="
uname -a
node --version
openclaw --version

echo -e "\n=== Plugin Status ==="
ls -la ~/.openclaw/extensions/mcp-integration/

echo -e "\n=== Configuration ==="
cat ~/.openclaw/openclaw.json | jq '.plugins.entries["mcp-integration"]'

echo -e "\n=== Recent Logs ==="
openclaw logs | grep -i mcp | tail -50

echo -e "\n=== Server Tests ==="
for url in http://localhost:3000/mcp http://localhost:3001/mcp; do
  echo "Testing $url..."
  curl -s -X POST "$url" \
    -H "Content-Type: application/json" \
    -d '{"jsonrpc":"2.0","id":1,"method":"tools/list"}' \
    | jq . 2>/dev/null || echo "Failed"
done
```

Run and share:
```bash
chmod +x diagnose-mcp.sh
./diagnose-mcp.sh > mcp-diagnostic.txt
```

## 📞 Getting Help

### Before Asking for Help

1. Check this troubleshooting guide
2. Review the logs: `openclaw logs | grep MCP`
3. Test server connection manually with `curl`
4. Verify configuration syntax with `jq`
5. Try with minimal configuration
6. Create diagnostic report (above)

### Support Channels

1. **GitHub Issues**: Include diagnostic report
2. **Discord**: OpenClaw Community #plugins channel
3. **Email**: Attach diagnostic report

### Useful Information to Include

- OpenClaw version: `openclaw --version`
- Node version: `node --version`
- Plugin version: Check `package.json`
- Configuration (redacted secrets)
- Error logs
- Steps to reproduce
- Diagnostic report

## 🔗 Related Documentation

- [Main README](../README.md)
- [Configuration](CONFIGURATION.md)
- [API Reference](API.md)

---

**Quick Fixes:**
- Restart gateway: `openclaw gateway restart`
- Check logs: `openclaw logs | grep MCP`
- Test server: `curl http://localhost:3000/mcp`
- Validate config: `cat ~/.openclaw/openclaw.json | jq .`
