# Configuration Guide

Complete guide to configuring the MCP Integration plugin for OpenClaw.

## 📋 Configuration File Location

The plugin is configured in OpenClaw's main configuration file:

```
~/.openclaw/openclaw.json
```

## 🔧 Basic Configuration

### Minimal Configuration

```json
{
  "plugins": {
    "entries": {
      "mcp-integration": {
        "enabled": true,
        "config": {
          "enabled": true,
          "servers": {
            "my-server": {
              "enabled": true,
              "transport": "http",
              "url": "http://localhost:3000/mcp"
            }
          }
        }
      }
    }
  }
}
```

### Full Configuration Example

```json
{
  "plugins": {
    "entries": {
      "mcp-integration": {
        "enabled": true,
        "config": {
          "enabled": true,
          "servers": {
            "kr-legal": {
              "enabled": true,
              "transport": "http",
              "url": "http://localhost:3000/mcp",
              "timeout": 30000,
              "retries": 3
            },
            "database": {
              "enabled": true,
              "transport": "http",
              "url": "http://localhost:3001/mcp"
            },
            "weather": {
              "enabled": false,
              "transport": "http",
              "url": "http://localhost:3002/mcp"
            }
          }
        }
      }
    }
  }
}
```

## 📊 Configuration Schema

### Root Configuration

```typescript
{
  enabled: boolean;              // Enable/disable entire plugin
  servers: {
    [serverName: string]: ServerConfig;
  }
}
```

### Server Configuration

```typescript
{
  enabled: boolean;              // Enable/disable this server
  transport: "http";             // Transport type
  url: string;                   // Server URL
  timeout?: number;              // Request timeout in ms (default: 30000)
  retries?: number;              // Retry attempts (default: 0)
}
```

## 🌍 Environment Variables

MCP servers often require environment variables for API keys or configuration.

### Setting Environment Variables

**Option 1: In openclaw.json**
```json
{
  "env": {
    "KR_LEGAL_API_KEY": "your-api-key-here"
  },
  "plugins": {
    "entries": {
      "mcp-integration": {
        "config": {
          "servers": {
            "kr-legal": {
              "url": "http://localhost:3000/mcp"
            }
          }
        }
      }
    }
  }
}
```

**Option 2: System environment**
```bash
# In ~/.bashrc or ~/.zshrc
export KR_LEGAL_API_KEY="your-api-key-here"

# Then reference in server
```

**Option 3: .env file**
```bash
# ~/.openclaw/.env
KR_LEGAL_API_KEY=your-api-key-here
```

```json
{
  "envFile": "/home/user/.openclaw/.env",
  "plugins": { ... }
}
```

## 🎭 Per-Agent Configuration

Enable MCP tools for specific agents only:

```json
{
  "agents": {
    "main": {
      "tools": {
        "allowlist": ["mcp"]
      }
    },
    "chat": {
      "tools": {
        "denylist": ["mcp"]
      }
    }
  },
  "plugins": {
    "entries": {
      "mcp-integration": { ... }
    }
  }
}
```

## 🔄 Multiple Server Configurations

### Example: Different Services

```json
{
  "servers": {
    "legal": {
      "enabled": true,
      "transport": "http",
      "url": "http://localhost:3000/mcp"
    },
    "database": {
      "enabled": true,
      "transport": "http",
      "url": "http://localhost:3001/mcp"
    },
    "weather": {
      "enabled": true,
      "transport": "http",
      "url": "http://localhost:3002/mcp"
    },
    "calendar": {
      "enabled": true,
      "transport": "http",
      "url": "http://localhost:3003/mcp"
    }
  }
}
```

### Example: Development vs Production

**Development:**
```json
{
  "servers": {
    "test-server": {
      "enabled": true,
      "url": "http://localhost:3000/mcp"
    }
  }
}
```

**Production:**
```json
{
  "servers": {
    "prod-server": {
      "enabled": true,
      "url": "https://mcp.example.com/mcp"
    }
  }
}
```

## 🔐 Secure Configuration

### HTTPS Connections

Always use HTTPS in production:

```json
{
  "servers": {
    "secure-server": {
      "enabled": true,
      "transport": "http",
      "url": "https://secure.example.com/mcp"
    }
  }
}
```

### API Key Management

**Best Practice:**
```json
{
  "env": {
    "MCP_API_KEY": "${MCP_API_KEY}"
  }
}
```

Then set in environment:
```bash
export MCP_API_KEY="secret-key-here"
```

**Never commit:**
```json
{
  "env": {
    "MCP_API_KEY": "sk-1234567890abcdef"  // ❌ BAD - exposed in git
  }
}
```

## 🧪 Testing Configuration

### Validate JSON

```bash
# Check syntax
cat ~/.openclaw/openclaw.json | jq .

# Validate specific plugin config
cat ~/.openclaw/openclaw.json | jq '.plugins.entries["mcp-integration"]'
```

### Test Server Connection

```bash
# Test URL is accessible
curl http://localhost:3000/mcp

# Test with JSON-RPC request
curl -X POST http://localhost:3000/mcp \
  -H "Content-Type: application/json" \
  -d '{
    "jsonrpc": "2.0",
    "id": 1,
    "method": "tools/list",
    "params": {}
  }'
```

### Verify Plugin Loaded

```bash
# Check OpenClaw logs
openclaw logs | grep MCP

# Expected output:
# [MCP] Plugin registered
# [MCP] Starting...
# [MCP] Connecting to kr-legal at http://localhost:3000/mcp
# [MCP] Connected to kr-legal: 5 tools available
# [MCP] Started
```

## 🔄 Reload Configuration

### Apply Changes

```bash
# Method 1: Full restart
openclaw gateway restart

# Method 2: Reload config (if supported)
openclaw config reload
```

### Check Status

```bash
# Verify plugin status
openclaw status

# Check specific plugin
openclaw plugins status mcp-integration
```

## 🐛 Troubleshooting Configuration

### Common Issues

**Issue 1: Plugin not loading**
```bash
# Check file exists
ls -la ~/.openclaw/extensions/mcp-integration/

# Check config syntax
cat ~/.openclaw/openclaw.json | jq '.plugins.entries["mcp-integration"]'
```

**Issue 2: Server not connecting**
```bash
# Test URL
curl http://localhost:3000/mcp

# Check logs
openclaw logs | grep "Failed to connect"
```

**Issue 3: Invalid configuration**
```bash
# Validate against schema
# The plugin will log validation errors on startup
openclaw logs | grep "config"
```

## 📋 Configuration Checklist

Before deploying:

- [ ] Valid JSON syntax
- [ ] All required fields present
- [ ] URLs are accessible
- [ ] API keys set securely (environment variables)
- [ ] HTTPS used in production
- [ ] Appropriate timeouts set
- [ ] Test connection to each server
- [ ] Gateway restarted after changes
- [ ] Logs checked for errors

## 🔗 Related Documentation

- [Main README](../README.md)
- [API Reference](API.md)
- [Troubleshooting Guide](TROUBLESHOOTING.md)

---

**Tips:**
- Start with minimal configuration
- Test each server individually
- Use environment variables for secrets
- Monitor logs after configuration changes
- Keep backups of working configurations
