# API Reference

Complete API documentation for the MCP Integration plugin.

## 🛠️ The `mcp` Tool

The plugin exposes a single tool called `mcp` that provides access to all connected MCP servers.

### Tool Schema

```typescript
{
  name: "mcp",
  description: "Call MCP (Model Context Protocol) server tools",
  parameters: {
    type: "object",
    properties: {
      action: {
        type: "string",
        enum: ["list", "call"],
        description: "Action to perform"
      },
      server: {
        type: "string",
        description: "MCP server name (required for call)"
      },
      tool: {
        type: "string",
        description: "Tool name (required for call)"
      },
      args: {
        type: "object",
        description: "Tool arguments (optional for call)"
      }
    },
    required: ["action"]
  }
}
```

## 📊 Actions

### 1. List Tools (`action: "list"`)

Get a list of all available tools from all connected MCP servers.

#### Request

```json
{
  "tool": "mcp",
  "args": {
    "action": "list"
  }
}
```

#### Response

```json
{
  "content": [
    {
      "type": "text",
      "text": "[
        {
          \"id\": \"kr-legal:search_statute\",
          \"server\": \"kr-legal\",
          \"name\": \"search_statute\",
          \"description\": \"Search Korean statutes\",
          \"inputSchema\": {
            \"type\": \"object\",
            \"properties\": {
              \"query\": {\"type\": \"string\"},
              \"limit\": {\"type\": \"number\"}
            },
            \"required\": [\"query\"]
          }
        },
        {
          \"id\": \"database:query\",
          \"server\": \"database\",
          \"name\": \"query\",
          \"description\": \"Execute database query\",
          \"inputSchema\": {...}
        }
      ]"
    }
  ]
}
```

#### Tool Info Structure

```typescript
interface ToolInfo {
  id: string;              // Format: "server:toolName"
  server: string;          // Server name
  name: string;            // Tool name
  description: string;     // Tool description
  inputSchema: JSONSchema; // JSON Schema for arguments
}
```

#### Example Usage

**OpenClaw Chat:**
```
User: What MCP tools are available?

AI: Let me check the available MCP tools.

[Calls: mcp(action=list)]

Available MCP tools:
- kr-legal:search_statute - Search Korean statutes
- kr-legal:search_case_law - Search court decisions
- database:query - Execute SQL queries
- weather:get_forecast - Get weather forecast
```

### 2. Call Tool (`action: "call"`)

Execute a specific tool on an MCP server.

#### Request

```json
{
  "tool": "mcp",
  "args": {
    "action": "call",
    "server": "kr-legal",
    "tool": "search_statute",
    "args": {
      "query": "민법",
      "limit": 5
    }
  }
}
```

#### Parameters

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `action` | string | Yes | Must be "call" |
| `server` | string | Yes | Name of MCP server (from config) |
| `tool` | string | Yes | Name of tool to call |
| `args` | object | No | Arguments to pass to tool |

#### Response

```json
{
  "content": [
    {
      "type": "text",
      "text": "{
        \"result\": {
          \"results\": [
            {
              \"title\": \"민법\",
              \"statute_id\": \"0001\",
              ...
            }
          ]
        }
      }"
    }
  ]
}
```

#### Error Response

```json
{
  "content": [
    {
      "type": "text",
      "text": "Error: Tool not found: kr-legal:invalid_tool"
    }
  ],
  "isError": true
}
```

#### Example Usage

**OpenClaw Chat:**
```
User: Search for Korean civil law

AI: I'll search Korean statutes for civil law.

[Calls: mcp(
  action=call,
  server=kr-legal,
  tool=search_statute,
  args={query: "민법", limit: 5}
)]

Found 5 statutes related to civil law:
1. 민법 (Civil Code) - Statute #0001
2. 민법 시행령 (Civil Code Enforcement Decree)
...
```

## 🔌 MCPManager Class

Internal class that manages MCP server connections. Not directly accessible from agents.

### Methods

#### `connectServer(name, config)`

Connect to an MCP server.

```typescript
async connectServer(
  name: string,
  config: ServerConfig
): Promise<Tool[]>
```

**Parameters:**
- `name`: Server name
- `config`: Server configuration object

**Returns:**
- Array of tools provided by the server

**Throws:**
- Error if connection fails

**Example:**
```javascript
const tools = await mcpManager.connectServer('kr-legal', {
  enabled: true,
  transport: 'http',
  url: 'http://localhost:3000/mcp'
});
// Returns: [{name: 'search_statute', ...}, ...]
```

#### `callTool(serverName, toolName, args)`

Call a tool on a specific server.

```typescript
async callTool(
  serverName: string,
  toolName: string,
  args: object = {}
): Promise<ToolResult>
```

**Parameters:**
- `serverName`: Name of MCP server
- `toolName`: Name of tool to call
- `args`: Tool arguments (optional)

**Returns:**
- Tool execution result

**Throws:**
- Error if tool not found
- Error if tool execution fails

**Example:**
```javascript
const result = await mcpManager.callTool(
  'kr-legal',
  'search_statute',
  { query: '민법', limit: 5 }
);
// Returns: {content: [{type: 'text', text: '...'}]}
```

#### `listTools()`

Get all available tools from all servers.

```typescript
listTools(): ToolInfo[]
```

**Returns:**
- Array of tool information objects

**Example:**
```javascript
const tools = mcpManager.listTools();
// Returns: [
//   {id: 'kr-legal:search_statute', server: 'kr-legal', name: 'search_statute', ...},
//   {id: 'database:query', server: 'database', name: 'query', ...}
// ]
```

#### `disconnect()`

Disconnect from all MCP servers.

```typescript
async disconnect(): Promise<void>
```

**Example:**
```javascript
await mcpManager.disconnect();
// All servers disconnected
```

## 🌐 StreamableHTTPClientTransport Class

Implements MCP Streamable HTTP transport specification.

### Constructor

```typescript
constructor(url: string | URL, options?: TransportOptions)
```

**Parameters:**
```typescript
interface TransportOptions {
  sessionId?: string;  // Optional session ID
  debug?: boolean;     // Enable debug logging
}
```

**Example:**
```javascript
const transport = new StreamableHTTPClientTransport(
  'http://localhost:3000/mcp',
  { debug: true }
);
```

### Methods

#### `start()`

Start the transport.

```typescript
async start(): Promise<void>
```

#### `send(message)`

Send a JSON-RPC message to the server.

```typescript
async send(message: JSONRPCRequest): Promise<JSONRPCResponse>
```

**Example:**
```javascript
const response = await transport.send({
  jsonrpc: '2.0',
  id: 1,
  method: 'tools/list',
  params: {}
});
```

#### `close()`

Close the transport.

```typescript
async close(): Promise<void>
```

### Event Handlers

```typescript
transport.onmessage = (message: JSONRPCMessage) => {
  // Handle incoming message
};

transport.onerror = (error: Error) => {
  // Handle error
};

transport.onclose = () => {
  // Handle connection close
};
```

## 📡 JSON-RPC Protocol

### Request Format

```typescript
{
  jsonrpc: "2.0",
  id: number | string,
  method: string,
  params?: object
}
```

### Response Format

```typescript
{
  jsonrpc: "2.0",
  id: number | string,
  result?: any,
  error?: {
    code: number,
    message: string,
    data?: any
  }
}
```

### Standard Methods

#### `tools/list`

List available tools.

**Request:**
```json
{
  "jsonrpc": "2.0",
  "id": 1,
  "method": "tools/list",
  "params": {}
}
```

**Response:**
```json
{
  "jsonrpc": "2.0",
  "id": 1,
  "result": {
    "tools": [
      {
        "name": "search_statute",
        "description": "Search Korean statutes",
        "inputSchema": {...}
      }
    ]
  }
}
```

#### `tools/call`

Execute a tool.

**Request:**
```json
{
  "jsonrpc": "2.0",
  "id": 2,
  "method": "tools/call",
  "params": {
    "name": "search_statute",
    "arguments": {
      "query": "민법",
      "limit": 5
    }
  }
}
```

**Response:**
```json
{
  "jsonrpc": "2.0",
  "id": 2,
  "result": {
    "content": [
      {
        "type": "text",
        "text": "..."
      }
    ]
  }
}
```

## 🔗 Related Documentation

- [Main README](../README.md)
- [Configuration Guide](CONFIGURATION.md)
- [Usage Examples](EXAMPLES.md)

---

**API Version:** 0.1.0  
**MCP SDK Version:** 0.5.0  
**Last Updated:** 2026-02-01
