---
name: huckleberry
description: Track baby sleep, feeding, diapers, and growth via Huckleberry app API. Use for logging baby activities through natural language.
homepage: https://github.com/aaronn/openclaw-huckleberry-skill
metadata:
  clawdbot:
    emoji: "👶"
    requires:
      bins: ["python3"]
      packages: ["huckleberry-api"]
    install:
      - id: pip-huckleberry
        kind: pip
        package: huckleberry-api
        label: Install huckleberry-api (pip)
---

# Huckleberry Baby Tracker

Track baby activities (sleep, feeding, diapers, growth) via the Huckleberry app's Firebase backend.

## Setup

1. Install the API:
   ```bash
   # Install from GitHub (required for bottle feeding support until next PyPI release)
   pip install git+https://github.com/Woyken/py-huckleberry-api.git
   # or with uv:
   uv pip install git+https://github.com/Woyken/py-huckleberry-api.git
   ```

2. Configure credentials (choose one):
   - Environment variables:
     ```bash
     export HUCKLEBERRY_EMAIL="your-email@example.com"
     export HUCKLEBERRY_PASSWORD="your-password"
     export HUCKLEBERRY_TIMEZONE="America/Los_Angeles"  # optional
     ```
   - Config file at `~/.config/huckleberry/credentials.json`:
     ```json
     {
       "email": "your-email@example.com",
       "password": "your-password",
       "timezone": "America/Los_Angeles"
     }
     ```

## CLI Usage

The CLI is at `~/clawd/skills/huckleberry/scripts/hb.py`

```bash
# List children
python3 ~/clawd/skills/huckleberry/scripts/hb.py children

# Sleep tracking
python3 ~/clawd/skills/huckleberry/scripts/hb.py sleep-start
python3 ~/clawd/skills/huckleberry/scripts/hb.py sleep-pause
python3 ~/clawd/skills/huckleberry/scripts/hb.py sleep-resume
python3 ~/clawd/skills/huckleberry/scripts/hb.py sleep-complete
python3 ~/clawd/skills/huckleberry/scripts/hb.py sleep-cancel

# Breastfeeding
python3 ~/clawd/skills/huckleberry/scripts/hb.py feed-start --side left
python3 ~/clawd/skills/huckleberry/scripts/hb.py feed-switch
python3 ~/clawd/skills/huckleberry/scripts/hb.py feed-pause
python3 ~/clawd/skills/huckleberry/scripts/hb.py feed-resume --side right
python3 ~/clawd/skills/huckleberry/scripts/hb.py feed-complete
python3 ~/clawd/skills/huckleberry/scripts/hb.py feed-cancel

# Bottle feeding
python3 ~/clawd/skills/huckleberry/scripts/hb.py bottle 120 --type "Formula" --units ml

# Diaper
python3 ~/clawd/skills/huckleberry/scripts/hb.py diaper both --pee-amount medium --poo-amount big --color yellow --consistency loose

# Growth
python3 ~/clawd/skills/huckleberry/scripts/hb.py growth --weight 5.2 --height 55 --head 38 --units metric
python3 ~/clawd/skills/huckleberry/scripts/hb.py growth-get

# History
python3 ~/clawd/skills/huckleberry/scripts/hb.py history --date 2026-01-27
python3 ~/clawd/skills/huckleberry/scripts/hb.py history --days 7 --type sleep --type feed
python3 ~/clawd/skills/huckleberry/scripts/hb.py history --json
```

## Complete Parameter Reference

### Sleep Commands

| Command | Parameters | Description |
|---------|------------|-------------|
| `sleep-start` | — | Start a new sleep session (timer begins) |
| `sleep-pause` | — | Pause the current sleep session |
| `sleep-resume` | — | Resume a paused sleep session |
| `sleep-complete` | `--notes` | End sleep and save to history |
| `sleep-cancel` | — | Cancel without saving to history |

### Breastfeeding Commands

| Command | Parameters | Description |
|---------|-----------|-------------|
| `feed-start` | `--side {left,right}` (default: left) | Start nursing session |
| `feed-pause` | — | Pause session, accumulate duration |
| `feed-resume` | `--side {left,right}` (optional) | Resume on specified or last side |
| `feed-switch` | — | Switch to other side (auto-resumes if paused) |
| `feed-complete` | `--notes` | End session and save to history |
| `feed-cancel` | — | Cancel without saving |

### Bottle Feeding

```
bottle <amount> [options]
```

| Parameter | Values | Required | Default |
|-----------|--------|----------|---------|
| `amount` | Any number | **Yes** | — |
| `--type` / `-t` | `"Breast Milk"`, `"Formula"`, `"Mixed"` | No | `"Formula"` |
| `--units` / `-u` | `ml`, `oz` | No | `ml` |
| `--notes` / `-n` | Any text | No | — |

### Diaper Change

```
diaper <mode> [options]
```

| Parameter | Values | Required | Default |
|-----------|--------|----------|---------|
| `mode` | `pee`, `poo`, `both`, `dry` | **Yes** | — |
| `--pee-amount` | `little`, `medium`, `big` | No | — |
| `--poo-amount` | `little`, `medium`, `big` | No | — |
| `--color` | `yellow`, `brown`, `black`, `green`, `red`, `gray` | No | — |
| `--consistency` | `solid`, `loose`, `runny`, `mucousy`, `hard`, `pebbles`, `diarrhea` | No | — |
| `--rash` | (flag) | No | false |
| `--notes` | Any text | No | — |

#### Color Guide
- **yellow** — Normal for breastfed babies
- **brown** — Normal for formula-fed or older babies
- **green** — Can be normal, or indicate fast digestion/foremilk
- **black** — Normal first few days (meconium), concerning later
- **red** — May indicate blood, consult pediatrician
- **gray** — Uncommon, may indicate liver issues

#### Consistency Guide
- **solid** — Formed stool
- **loose** — Soft but not watery
- **runny** — Watery consistency
- **mucousy** — Contains mucus
- **hard** — Firm, may indicate constipation
- **pebbles** — Small hard pieces
- **diarrhea** — Very watery, frequent

### Growth Measurements

```
growth [options]
growth-get
```

| Parameter | Values | Required | Notes |
|-----------|--------|----------|-------|
| `--weight` / `-w` | Number | At least one | kg (metric) or lbs (imperial) |
| `--height` / `-l` | Number | measurement | cm (metric) or inches (imperial) |
| `--head` | Number | required | cm (metric) or inches (imperial) |
| `--units` / `-u` | `metric`, `imperial` | No | Default: `metric` |
| `--notes` / `-n` | Any text | No | — |

### History / Calendar

```
history [options]
```

| Parameter | Values | Required | Default |
|-----------|--------|----------|---------|
| `--date` / `-d` | `YYYY-MM-DD` | No | Today |
| `--days` | Number | No | 1 |
| `--type` / `-t` | `sleep`, `feed`, `diaper`, `health` | No | All types |

Use `--type` multiple times to filter: `--type sleep --type feed`

## Agent Guidelines: When to Ask for Details

### AI Attribution on Notes
**Always** include AI attribution when logging entries:

**Creating new entries:**
- No user note: `--notes "Created via AI"`
- User provides note: `--notes "user's note | Created via AI"`

**Editing existing entries:**
- No user note: `--notes "Updated via AI"`
- User provides note: append ` | Updated via AI` to existing notes

This creates a paper trail for AI-assisted entries.

### When to Ask for Clarification
When a user request is sparse, ask for clarification before logging. Here's when:

### Diaper Changes
If user says just "diaper change" or "poop":
- **Always ask:** Was it pee, poo, or both?
- **For poo, consider asking:** Color? Consistency? Amount?
- **Skip details if:** User seems rushed or says "just log it"

Example follow-up:
> "Got it! Was it pee, poo, or both? Any details to note (color, consistency, amount)?"

### Bottle Feeding
If user says "bottle" without amount:
- **Always ask:** How much? (in ml or oz)
- **Consider asking:** Formula, breast milk, or mixed?

Example follow-up:
> "How much was the bottle? And was it formula, breast milk, or mixed?"

### Growth Measurements
If user says "log weight" without value:
- **Always ask:** What's the weight? (and clarify units if ambiguous)

### Sleep/Feeding Timers
These are typically clear commands, but clarify if ambiguous:
- "Baby's eating" → "Starting breastfeeding — which side, left or right?"
- "Feed done" → Check if breastfeeding or bottle context

## Natural Language Examples

| User says | Action |
|-----------|--------|
| "Baby fell asleep" | `sleep-start` |
| "She woke up" | `sleep-complete` |
| "Cancel that sleep" | `sleep-cancel` |
| "Feeding on the left" | `feed-start --side left` |
| "Switch sides" | `feed-switch` |
| "Done nursing" | `feed-complete` |
| "4oz bottle of formula" | `bottle 4 --type Formula --units oz` |
| "120ml breast milk bottle" | `bottle 120 --type "Breast Milk" --units ml` |
| "Diaper change, pee and poo" | → Ask about amounts/color/consistency |
| "Just a wet diaper" | `diaper pee` |
| "Dry check" | `diaper dry` |
| "Weight is 5.5kg" | `growth --weight 5.5 --units metric` |
| "What did the baby do today?" | `history --days 1` |
| "Sleep history for the week" | `history --days 7 --type sleep` |

## Multi-Child Support

If the account has multiple children, use `--child` / `-c`:
```bash
python3 hb.py --child "Baby Name" sleep-start
```

Without `--child`, commands default to the first child in the account.

## Troubleshooting

**Authentication errors:**
- Verify email/password are correct
- Check credentials file permissions
- Huckleberry doesn't support 2FA for API access

**"No children found":**
- Ensure the account has at least one child profile in the Huckleberry app

**Timer already active:**
- Complete or cancel the existing session before starting a new one

## Technical Notes

- Uses Firebase Firestore via gRPC (same as mobile app)
- Real-time sync: Changes appear immediately in the Huckleberry app
- Token auto-refresh: Sessions stay authenticated
- **Timezone handling:** Huckleberry requires an `offset` field (minutes behind UTC) for entries to display correctly. E.g., PST (UTC-8) = 480 minutes. The CLI automatically calculates this from the configured timezone. Without this field, entries appear at UTC time in the app.

---

## Credits

Built on [py-huckleberry-api](https://github.com/Woyken/py-huckleberry-api) by Woyken — a reverse-engineered Python client for Huckleberry's Firebase backend.

---

*Created with AI - 2026-01-27*
*Updated with AI - 2026-01-28*

## Notes on All Entry Types

The `--notes` / `-n` parameter is available on all entry types:
- `sleep-complete --notes "Slept through the night!"`
- `feed-complete --notes "Good latch today"`
- `bottle --notes "Logged via AI"`
- `diaper --notes "Checked by AI"`
- `growth --notes "Measured at pediatrician"`

The upstream py-huckleberry-api only supports notes on diaper entries. This skill extends that to all types by updating the Firestore document directly after creation.

## Not Supported by Upstream API

The following features exist in Huckleberry but aren't exposed in the py-huckleberry-api:
- Sleep conditions (happy/upset at start/end)
- Sleep locations (car, nursing, stroller, crib, etc.)

These would require modifying the upstream library to accept additional parameters.
