#!/bin/bash
# Start OpenClaw Docker Container
# Usage: ./start.sh [clean|full] [port]

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
VERSION="${2:-clean}"
PORT="${3:-18789}"

if [ -z "$VERSION" ] || [[ ! "$VERSION" =~ ^(clean|full)$ ]]; then
    echo "Usage: $0 [clean|full] [port]"
    echo ""
    echo "Options:"
    echo "  clean  - Start clean version (no personal data)"
    echo "  full   - Start full version (with DreamMoon config & history)"
    echo ""
    echo "Example:"
    echo "  $0 clean      # Start clean version on port 18789"
    echo "  $0 full 8080  # Start full version on port 8080"
    exit 1
fi

CONTAINER_NAME="openclaw-$VERSION"

echo "=========================================="
echo "Starting OpenClaw $VERSION version"
echo "Port: $PORT"
echo "=========================================="

# Stop existing container if running
if docker ps -a --format '{{.Names}}' | grep -q "^$CONTAINER_NAME$"; then
    echo "🛑 Stopping existing container: $CONTAINER_NAME"
    docker stop "$CONTAINER_NAME" 2>/dev/null || true
    docker rm "$CONTAINER_NAME" 2>/dev/null || true
fi

# Prepare volume mounts
if [ "$VERSION" == "full" ]; then
    # For full version, mount workspace for persistence
    mkdir -p /home/zfanmy/openclaw_data/workspace
    VOLUME_MOUNTS="-v /home/zfanmy/openclaw_data/workspace:/root/.openclaw/workspace"
else
    VOLUME_MOUNTS=""
fi

# Run container
docker run -d \
    --name "$CONTAINER_NAME" \
    --restart unless-stopped \
    -p "$PORT:18789" \
    -e NODE_ENV=production \
    $VOLUME_MOUNTS \
    "openclaw:$VERSION"

echo ""
echo "✅ Container started: $CONTAINER_NAME"
echo ""
echo "Access URLs:"
echo "  - WebUI: http://localhost:$PORT"
echo "  - LAN:   http://$(hostname -I | awk '{print $1}'):$PORT"
echo ""
echo "Logs: docker logs -f $CONTAINER_NAME"
echo "Stop: docker stop $CONTAINER_NAME"
echo "=========================================="
