/**
 * Test suite for ContextCompressor
 *
 * Run with: node --test src/compressor.test.js
 */

import { test } from 'node:test';
import assert from 'node:assert';
import { ContextCompressor, createCompressor } from './compressor.js';

// Sample test contexts
const SAMPLE_CONTEXTS = {
  simple: `Hello, how are you today?
I'm doing well, thanks!
Hello, how are you today?
I'm doing well, thanks!`,

  verbose: `The quick brown fox jumps over the lazy dog. The quick brown fox is very quick.
The lazy dog is sleeping. The dog is very lazy and tired.
This is a sentence about programming. Programming is important.
We should write clean code. Clean code is maintainable code.`,

  withCode: `Here is some context about a function:

\`\`\`javascript
function calculateTotal(items) {
  return items.reduce((sum, item) => sum + item.price, 0);
}
\`\`\`

The function calculates the total price of items.
It uses the reduce method to sum up prices.
This is a very useful utility function.`,

  withBoilerplate: `---
Copyright (c) 2024 OpenClaw
Licensed under MIT License
---

Main content here that is actually important.
This section contains valuable information.

---
Generated by OpenClaw Context Optimizer
Auto-generated content
---`,

  realWorld: `User: I prefer using TypeScript over JavaScript for large projects.
Assistant: That's a good choice. TypeScript provides type safety.
User: My name is John and I work at OpenAI.
Assistant: Nice to meet you, John!
User: I always use async/await instead of promises.
Assistant: Async/await is indeed more readable.
User: Can you help me with something?
Assistant: Sure, what do you need?
User: I prefer using TypeScript over JavaScript for large projects.
Assistant: You mentioned that earlier.`
};

test('ContextCompressor - Constructor', () => {
  const compressor = new ContextCompressor();

  assert.ok(compressor, 'Compressor instance created');
  assert.strictEqual(compressor.options.targetCompressionRatio, 0.5, 'Default compression ratio is 0.5');
  assert.strictEqual(compressor.options.similarityThreshold, 0.9, 'Default similarity threshold is 0.9');
});

test('ContextCompressor - Factory function', () => {
  const compressor = createCompressor();

  assert.ok(compressor instanceof ContextCompressor, 'Factory creates ContextCompressor instance');
});

test('ContextCompressor - Token estimation', () => {
  const compressor = new ContextCompressor();

  const shortText = 'Hello world';
  const tokens = compressor.estimateTokens(shortText);

  assert.ok(tokens > 0, 'Estimates tokens for short text');
  assert.ok(tokens >= 2 && tokens <= 5, 'Reasonable estimate for "Hello world"');

  const emptyTokens = compressor.estimateTokens('');
  assert.strictEqual(emptyTokens, 0, 'Returns 0 for empty text');
});

test('ContextCompressor - Deduplication strategy', async () => {
  const compressor = new ContextCompressor();

  const result = await compressor.compress(SAMPLE_CONTEXTS.simple, 'deduplication');

  assert.ok(result.compressed, 'Returns compressed text');
  assert.ok(result.metrics.compressionRatio > 0, 'Achieves compression');
  assert.ok(result.compressed.length < result.original.length, 'Compressed text is shorter');

  console.log('Deduplication Test:');
  console.log(`Original tokens: ${result.metrics.originalTokens}`);
  console.log(`Compressed tokens: ${result.metrics.compressedTokens}`);
  console.log(`Compression ratio: ${(result.metrics.compressionRatio * 100).toFixed(1)}%`);
  console.log(`Quality score: ${result.metrics.qualityScore}`);
});

test('ContextCompressor - Pruning strategy', async () => {
  const compressor = new ContextCompressor();

  const context = `Hello, how are you?
This is important information about the project.
Thanks for your help!
The key point is that we need to optimize context.
Okay, sure.`;

  const result = await compressor.compress(context, 'pruning');

  assert.ok(result.compressed, 'Returns compressed text');
  assert.ok(!result.compressed.includes('Hello, how are you?'), 'Removes low-value phrases');
  assert.ok(result.compressed.includes('important information'), 'Preserves important content');

  console.log('\nPruning Test:');
  console.log(`Original tokens: ${result.metrics.originalTokens}`);
  console.log(`Compressed tokens: ${result.metrics.compressedTokens}`);
  console.log(`Compression ratio: ${(result.metrics.compressionRatio * 100).toFixed(1)}%`);
});

test('ContextCompressor - Summarization strategy', async () => {
  const compressor = new ContextCompressor();

  const result = await compressor.compress(SAMPLE_CONTEXTS.verbose, 'summarization');

  assert.ok(result.compressed, 'Returns compressed text');
  assert.ok(result.metrics.compressionRatio > 0, 'Achieves compression');

  console.log('\nSummarization Test:');
  console.log(`Original tokens: ${result.metrics.originalTokens}`);
  console.log(`Compressed tokens: ${result.metrics.compressedTokens}`);
  console.log(`Compression ratio: ${(result.metrics.compressionRatio * 100).toFixed(1)}%`);
  console.log(`Quality score: ${result.metrics.qualityScore}`);
});

test('ContextCompressor - Template removal strategy', async () => {
  const compressor = new ContextCompressor();

  const result = await compressor.compress(SAMPLE_CONTEXTS.withBoilerplate, 'template');

  assert.ok(result.compressed, 'Returns compressed text');
  assert.ok(!result.compressed.includes('Copyright'), 'Removes copyright notices');
  assert.ok(!result.compressed.includes('Generated by'), 'Removes auto-generated notices');
  assert.ok(result.compressed.includes('important'), 'Preserves main content');

  console.log('\nTemplate Removal Test:');
  console.log(`Original tokens: ${result.metrics.originalTokens}`);
  console.log(`Compressed tokens: ${result.metrics.compressedTokens}`);
  console.log(`Compression ratio: ${(result.metrics.compressionRatio * 100).toFixed(1)}%`);
});

test('ContextCompressor - Hybrid strategy', async () => {
  const compressor = new ContextCompressor();

  const result = await compressor.compress(SAMPLE_CONTEXTS.realWorld, 'hybrid');

  assert.ok(result.compressed, 'Returns compressed text');
  assert.ok(result.metrics.compressionRatio > 0.3, 'Achieves significant compression (>30%)');
  assert.ok(result.metrics.qualityScore > 0.4, 'Maintains reasonable quality');

  console.log('\nHybrid Strategy Test:');
  console.log(`Original tokens: ${result.metrics.originalTokens}`);
  console.log(`Compressed tokens: ${result.metrics.compressedTokens}`);
  console.log(`Compression ratio: ${(result.metrics.compressionRatio * 100).toFixed(1)}%`);
  console.log(`Quality score: ${result.metrics.qualityScore}`);
  console.log(`Compression time: ${result.metrics.compressionTime}ms`);
});

test('ContextCompressor - Preserves code blocks', async () => {
  const compressor = new ContextCompressor();

  const result = await compressor.compress(SAMPLE_CONTEXTS.withCode, 'summarization');

  assert.ok(result.compressed.includes('```'), 'Preserves code block markers');
  assert.ok(result.compressed.includes('function calculateTotal'), 'Preserves code content');

  console.log('\nCode Preservation Test:');
  console.log(`Original tokens: ${result.metrics.originalTokens}`);
  console.log(`Compressed tokens: ${result.metrics.compressedTokens}`);
  console.log(`Code preserved: ${result.compressed.includes('```')}`);
});

test('ContextCompressor - Calculate metrics', () => {
  const compressor = new ContextCompressor();

  const original = 'The quick brown fox jumps over the lazy dog.';
  const compressed = 'Quick fox jumps.';

  const metrics = compressor.calculateMetrics(original, compressed);

  assert.ok(metrics.originalTokens > 0, 'Calculates original tokens');
  assert.ok(metrics.compressedTokens > 0, 'Calculates compressed tokens');
  assert.ok(metrics.compressionRatio > 0, 'Calculates compression ratio');
  assert.ok(metrics.tokensRemoved > 0, 'Calculates tokens removed');
  assert.strictEqual(typeof metrics.percentageReduction, 'number', 'Includes percentage reduction');
});

test('ContextCompressor - Quality score', () => {
  const compressor = new ContextCompressor();

  const original = 'John works at OpenAI. He builds AI systems. Python is his favorite language.';
  const good = 'John works at OpenAI building AI systems with Python.';
  const poor = 'He builds.';

  const goodScore = compressor.calculateQualityScore(original, good);
  const poorScore = compressor.calculateQualityScore(original, poor);

  assert.ok(goodScore > poorScore, 'Better compression has higher quality score');
  assert.ok(goodScore >= 0 && goodScore <= 1, 'Quality score is between 0 and 1');

  console.log('\nQuality Score Test:');
  console.log(`Good compression quality: ${goodScore.toFixed(3)}`);
  console.log(`Poor compression quality: ${poorScore.toFixed(3)}`);
});

test('ContextCompressor - Empty input', async () => {
  const compressor = new ContextCompressor();

  const result = await compressor.compress('', 'hybrid');

  assert.strictEqual(result.compressed, '', 'Returns empty string for empty input');
  assert.strictEqual(result.metrics.originalTokens, 0, 'Zero tokens for empty input');
  assert.strictEqual(result.metrics.compressionRatio, 0, 'Zero compression ratio for empty input');
});

test('ContextCompressor - Structured object input', async () => {
  const compressor = new ContextCompressor();

  const structuredContext = {
    user: 'John',
    preferences: ['TypeScript', 'React', 'Node.js'],
    history: ['Built 5 projects', 'Senior developer', 'Built 5 projects']
  };

  const result = await compressor.compress(structuredContext, 'hybrid');

  assert.ok(result.compressed, 'Handles structured object input');
  assert.ok(result.metrics.compressionRatio >= 0, 'Calculates metrics for structured input');

  console.log('\nStructured Input Test:');
  console.log(`Original tokens: ${result.metrics.originalTokens}`);
  console.log(`Compressed tokens: ${result.metrics.compressedTokens}`);
});

test('ContextCompressor - Custom options', async () => {
  const compressor = new ContextCompressor(null, {
    targetCompressionRatio: 0.7, // More aggressive
    similarityThreshold: 0.8      // Lower threshold for dedup
  });

  const result = await compressor.compress(SAMPLE_CONTEXTS.realWorld, 'hybrid');

  assert.ok(result.metrics.compressionRatio > 0, 'Respects custom options');

  console.log('\nCustom Options Test:');
  console.log(`Target ratio: 0.7 (70% compression)`);
  console.log(`Achieved ratio: ${(result.metrics.compressionRatio * 100).toFixed(1)}%`);
});

test('ContextCompressor - Performance test', async () => {
  const compressor = new ContextCompressor();

  // Generate large context
  const largeContext = SAMPLE_CONTEXTS.realWorld.repeat(50);

  const startTime = Date.now();
  const result = await compressor.compress(largeContext, 'hybrid');
  const endTime = Date.now();

  const compressionTime = endTime - startTime;

  assert.ok(compressionTime < 5000, 'Completes within 5 seconds for large context');
  assert.ok(result.metrics.compressionRatio > 0, 'Successfully compresses large context');

  console.log('\nPerformance Test:');
  console.log(`Large context tokens: ${result.metrics.originalTokens}`);
  console.log(`Compression time: ${compressionTime}ms`);
  console.log(`Compression ratio: ${(result.metrics.compressionRatio * 100).toFixed(1)}%`);
});

// Summary test - demonstrates full workflow
test('ContextCompressor - Full workflow demo', async () => {
  const compressor = new ContextCompressor(null, {
    targetCompressionRatio: 0.5
  });

  const demoContext = `User: Hi, I'm working on a React application.
Assistant: Great! How can I help you with your React application?
User: I need to optimize my component rendering.
Assistant: You can use React.memo to prevent unnecessary re-renders.
User: How do I use React.memo?
Assistant: Wrap your component with React.memo like this:

\`\`\`javascript
const MyComponent = React.memo(({ data }) => {
  return <div>{data}</div>;
});
\`\`\`

User: Thanks! That's helpful.
Assistant: You're welcome! Let me know if you need more help.
User: I also prefer using TypeScript.
Assistant: TypeScript is great for type safety.
User: I need to optimize my component rendering.
Assistant: You mentioned that earlier. I suggested React.memo.`;

  const result = await compressor.compress(demoContext, 'hybrid');

  console.log('\n========================================');
  console.log('FULL WORKFLOW DEMONSTRATION');
  console.log('========================================');
  console.log('\nOriginal Context:');
  console.log('---');
  console.log(result.original);
  console.log('\nCompressed Context:');
  console.log('---');
  console.log(result.compressed);
  console.log('\n========================================');
  console.log('COMPRESSION METRICS:');
  console.log('========================================');
  console.log(`Strategy: ${result.strategy}`);
  console.log(`Original tokens: ${result.metrics.originalTokens}`);
  console.log(`Compressed tokens: ${result.metrics.compressedTokens}`);
  console.log(`Tokens removed: ${result.metrics.tokensRemoved}`);
  console.log(`Compression ratio: ${(result.metrics.compressionRatio * 100).toFixed(1)}%`);
  console.log(`Quality score: ${result.metrics.qualityScore.toFixed(3)}`);
  console.log(`Compression time: ${result.metrics.compressionTime}ms`);
  console.log('========================================\n');

  assert.ok(result.metrics.compressionRatio >= 0.3, 'Achieves at least 30% compression');
  assert.ok(result.metrics.qualityScore > 0.4, 'Maintains quality above 0.4');
});
