# ContextCompressor - Quick Start Guide

## Installation

```javascript
import { ContextCompressor } from './compressor.js';
```

## Basic Usage (30 seconds)

```javascript
// 1. Create compressor
const compressor = new ContextCompressor();

// 2. Compress context
const result = await compressor.compress(yourContext, 'hybrid');

// 3. Use compressed context
console.log(result.compressed);
console.log(`Saved ${result.metrics.tokensRemoved} tokens`);
```

## Common Scenarios

### Scenario 1: Compress Conversation History

```javascript
const compressor = new ContextCompressor();

// Your conversation history
const conversation = `
User: Tell me about React hooks.
Assistant: React hooks are functions that...
User: Can you give an example?
Assistant: Sure! Here's an example...
`;

const result = await compressor.compress(conversation, 'hybrid');

// Use compressed version in API call
const apiResponse = await callAPI({
  context: result.compressed  // Smaller payload, lower cost!
});
```

### Scenario 2: Remove Duplicate Information

```javascript
const context = `
User: I prefer TypeScript.
User: I like TypeScript better than JavaScript.
User: My preference is TypeScript.
`;

const result = await compressor.compress(context, 'deduplication');
// Result: "User: I prefer TypeScript."
```

### Scenario 3: Clean Up Documentation

```javascript
const documentation = `
---
Copyright (c) 2024
License: MIT
---

Important API documentation here...

---
Auto-generated by tool
---
`;

const result = await compressor.compress(documentation, 'template');
// Result: "Important API documentation here..."
```

### Scenario 4: Condense Verbose Explanations

```javascript
const verbose = `
React is a library. It's used for building UIs.
React uses components. Components are reusable.
State management is important. Redux helps with state.
`;

const result = await compressor.compress(verbose, 'summarization', {
  targetCompressionRatio: 0.5  // Keep 50% of content
});
```

## Strategies Cheat Sheet

| Strategy | Use When | Compression | Quality |
|----------|----------|-------------|---------|
| `'hybrid'` ⭐ | Default choice | 40-60% | Good |
| `'deduplication'` | Repeated info | 10-20% | Excellent |
| `'pruning'` | Conversational fluff | 10-30% | Good |
| `'summarization'` | Verbose text | 40-60% | Fair |
| `'template'` | Boilerplate/docs | 5-20% | Excellent |

## Configuration Options

```javascript
const compressor = new ContextCompressor(null, {
  // How much to compress (0.0 to 1.0)
  targetCompressionRatio: 0.5,    // 50% compression

  // Similarity threshold for deduplication
  similarityThreshold: 0.9,        // 0.9 = very similar required

  // Preserve code blocks
  preserveCodeBlocks: true,

  // Preserve important sections
  preserveImportantSections: true
});
```

## Understanding Metrics

```javascript
const result = await compressor.compress(context);

// Access metrics
result.metrics.originalTokens      // Original size
result.metrics.compressedTokens    // Compressed size
result.metrics.tokensRemoved       // Tokens saved
result.metrics.compressionRatio    // 0.0-1.0 (0.5 = 50% compression)
result.metrics.percentageReduction // 0-100 (50 = 50% reduction)
result.metrics.qualityScore        // 0.0-1.0 (higher = better)
result.metrics.compressionTime     // Milliseconds
```

## Quality Score Guide

- **0.8-1.0:** Excellent (minimal information loss)
- **0.6-0.8:** Good (acceptable loss)
- **0.4-0.6:** Fair (noticeable loss)
- **0.0-0.4:** Poor (significant loss)

## Common Patterns

### Pattern 1: Compress Before API Call

```javascript
const compressor = new ContextCompressor();

async function callAI(context) {
  // Compress first
  const result = await compressor.compress(context, 'hybrid');

  // Use compressed context
  return await fetch('/api/ai', {
    method: 'POST',
    body: JSON.stringify({ context: result.compressed })
  });
}
```

### Pattern 2: Monitor Quality

```javascript
const result = await compressor.compress(context);

if (result.metrics.qualityScore < 0.5) {
  console.warn('Quality too low, using less aggressive compression');

  // Retry with lower compression
  result = await compressor.compress(context, 'deduplication');
}
```

### Pattern 3: Token Estimation Only

```javascript
const compressor = new ContextCompressor();

// Just estimate tokens (no compression)
const tokens = compressor.estimateTokens(myText);
console.log(`Estimated tokens: ${tokens}`);
```

### Pattern 4: Custom Pruning Patterns

```javascript
const customPatterns = [
  /^DEBUG:/i,
  /^VERBOSE:/i,
  /^Temporary note:/i
];

const result = await compressor.compress(context, 'pruning', {
  patterns: customPatterns
});
```

## Troubleshooting

### Not Enough Compression

```javascript
// Try more aggressive settings
const compressor = new ContextCompressor(null, {
  targetCompressionRatio: 0.7,    // 70% compression
  similarityThreshold: 0.85       // Lower threshold
});

const result = await compressor.compress(context, 'hybrid');
```

### Quality Too Low

```javascript
// Try less aggressive settings
const compressor = new ContextCompressor(null, {
  targetCompressionRatio: 0.4,    // 40% compression
  preserveCodeBlocks: true,
  preserveImportantSections: true
});
```

### Code Blocks Being Removed

```javascript
// Ensure code preservation is enabled
const compressor = new ContextCompressor(null, {
  preserveCodeBlocks: true  // Always preserve code
});
```

## Testing

```bash
# Run test suite
node --test src/compressor.test.js

# Run examples
node src/compressor.example.js
```

## Advanced Usage

See full documentation:
- **API Reference:** `src/COMPRESSOR_API.md`
- **Examples:** `src/compressor.example.js`
- **Implementation Guide:** `src/README_COMPRESSOR.md`

## Performance Tips

1. **Use Hybrid for Most Cases:** Best balance of compression and quality
2. **Cache Results:** Don't recompress the same context
3. **Monitor Quality:** Check quality score and adjust settings
4. **Preserve Code:** Always enable code block preservation for technical content
5. **Test Your Use Case:** Different content types may need different strategies

## One-Liners

```javascript
// Quick compression
const compressed = (await new ContextCompressor().compress(text, 'hybrid')).compressed;

// Quick token estimate
const tokens = new ContextCompressor().estimateTokens(text);

// Quick metrics
const metrics = new ContextCompressor().calculateMetrics(original, compressed);
```

## Need Help?

1. Check examples: `node src/compressor.example.js`
2. Read API docs: `src/COMPRESSOR_API.md`
3. Run tests to see behavior: `node --test src/compressor.test.js`

---

**Ready to compress? Start with the hybrid strategy and adjust from there!**

```javascript
const compressor = new ContextCompressor();
const result = await compressor.compress(yourContext, 'hybrid');
console.log(`Saved ${result.metrics.tokensRemoved} tokens!`);
```
