---
name: openclaw-config
description: Edit and validate OpenClaw Gateway config (openclaw.json / JSON5). Use when adding/changing config keys (gateway.*, agents.*, models.*, channels.*, tools.*, skills.*, plugins.*, $include) or diagnosing openclaw doctor/config validation errors, to avoid schema mismatches that prevent the Gateway from starting or weaken security policies.
---

# OpenClaw Config

## Overview

Safely edit `~/.openclaw/openclaw.json` (or the path set by `OPENCLAW_CONFIG_PATH`) using a schema-first workflow. Validate before and after changes to avoid invalid keys/types that can break startup or change security behavior.

## Workflow (Safe Edit)

1. **Identify the active config path**

- Precedence: `OPENCLAW_CONFIG_PATH` > `OPENCLAW_STATE_DIR/openclaw.json` > `~/.openclaw/openclaw.json`
- The config file is **JSON5** (comments + trailing commas allowed).

2. **Get an authoritative schema (do not guess keys)**

- If the Gateway is running: use `openclaw gateway call config.schema --params '{}'` to fetch a JSON Schema matching the running version.
- Otherwise: use `openclaw/openclaw` source-of-truth, primarily:
  - `src/config/zod-schema.ts` (`OpenClawSchema` root keys like `gateway`/`skills`/`plugins`)
  - `src/config/zod-schema.*.ts` (submodules: channels/providers/models/agents/tools)
  - `docs/gateway/configuration.md` (repo docs + examples)

3. **Apply changes with the smallest safe surface**

- Prefer small edits: `openclaw config get|set|unset` (dot path or bracket notation).
- If the Gateway is online and you want "write + validate + restart" in one step: use RPC `config.patch` (merge patch) or `config.apply` (replaces the entire config; use carefully).
- For complex setups, split config with `$include` (see below).

4. **Validate strictly**

- Run `openclaw doctor`, then fix issues using the reported `path` + `message`.
- Do not run `openclaw doctor --fix/--yes` without explicit user consent (it writes to config/state files).

## Guardrails (Avoid Schema Bugs)

- **Most objects are strict** (`.strict()`): unknown keys usually fail validation and the Gateway will refuse to start.
- `channels` is `.passthrough()`: extension channels (matrix/zalo/nostr, etc.) can add custom keys, but most provider configs remain strict.
- `env` is `.catchall(z.string())`: you can put string env vars directly under `env`, and you can also use `env.vars`.
- **Secrets**: prefer environment variables/credential files. Avoid committing long-lived tokens/API keys into `openclaw.json`.

## $include (Modular Config)

`$include` is resolved before schema validation and lets you split config across JSON5 files:

- Supports `"$include": "./base.json5"` or `"$include": ["./a.json5", "./b.json5"]`
- Relative paths are resolved against the directory of the current config file.
- Deep-merge rules (per implementation):
  - objects: merge recursively
  - arrays: **concatenate** (not replace)
  - primitives: later value wins
- If sibling keys exist alongside `$include`, sibling keys override included values.
- Limits: max depth 10; circular includes are detected and rejected.

## Common Recipes (Examples)

1. Set default workspace

```bash
openclaw config set agents.defaults.workspace '"~/.openclaw/workspace"' --json
openclaw doctor
```

2. Change Gateway port

```bash
openclaw config set gateway.port 18789 --json
openclaw doctor
```

3. Split config (example)

```json5
// ~/.openclaw/openclaw.json
{
  "$include": ["./gateway.json5", "./channels/telegram.json5"],
}
```

4. Telegram open DMs (must explicitly allow senders)

> Schema constraint: when `dmPolicy="open"`, `allowFrom` must include `"*"`.

```bash
openclaw config set channels.telegram.dmPolicy '"open"' --json
openclaw config set channels.telegram.allowFrom '["*"]' --json
openclaw doctor
```

5. Discord token (config or env fallback)

```bash
# Option A: write to config
openclaw config set channels.discord.token '"YOUR_DISCORD_BOT_TOKEN"' --json

# Option B: env var fallback (still recommend a channels.discord section exists)
# export DISCORD_BOT_TOKEN="..."

openclaw doctor
```

6. Enable web_search (Brave / Perplexity)

```bash
openclaw config set tools.web.search.enabled true --json
openclaw config set tools.web.search.provider '"brave"' --json

# Recommended: provide the key via env var (or write tools.web.search.apiKey)
# export BRAVE_API_KEY="..."

openclaw doctor
```

## Resources

Load these when you need a field index or source locations:

- `references/openclaw-config-fields.md` (root key index + key field lists with sources)
- `references/schema-sources.md` (how to locate schema + constraints in openclaw repo)
- `scripts/openclaw-config-check.sh` (print config path + run doctor)
