---
name: openclaw-youtube
description: "YouTube SERP Scout for agents. Search top-ranking videos, channels, and trends for content research and competitor tracking."
homepage: https://openclaw.ai
metadata: {"openclaw":{"emoji":"📺","requires":{"bins":["curl","python3"],"env":["AISA_API_KEY"]},"primaryEnv":"AISA_API_KEY"}}
---

# OpenClaw YouTube 📺

**YouTube SERP Scout for autonomous agents. Powered by AIsa.**

One API key. Rank discovery. Content research. Competitor tracking.

## 🔥 What Can You Do?

### Content Research
```
"Find top-ranking videos about 'AI agents tutorial' to see what's working"
```

### Competitor Tracking
```
"Search for videos from competitor channels about 'machine learning'"
```

### Trend Discovery
```
"What are the top YouTube videos about 'GPT-5' right now?"
```

### Topic Analysis
```
"Find popular videos on 'autonomous driving' to understand audience interest"
```

### Channel Discovery
```
"Search for channels creating content about 'crypto trading'"
```

## Quick Start

```bash
export AISA_API_KEY="your-key"
```

---

## Core Capabilities

### Basic YouTube Search

```bash
# Search for videos
curl "https://api.aisa.one/apis/v1/youtube/search?engine=youtube&q=AI+agents+tutorial" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

### Search with Country Filter

```bash
# Search in specific country (US)
curl "https://api.aisa.one/apis/v1/youtube/search?engine=youtube&q=machine+learning&gl=us" \
  -H "Authorization: Bearer $AISA_API_KEY"

# Search in Japan
curl "https://api.aisa.one/apis/v1/youtube/search?engine=youtube&q=AI&gl=jp&hl=ja" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

### Search with Language Filter

```bash
# Search with interface language
curl "https://api.aisa.one/apis/v1/youtube/search?engine=youtube&q=python+tutorial&hl=en" \
  -H "Authorization: Bearer $AISA_API_KEY"

# Chinese interface
curl "https://api.aisa.one/apis/v1/youtube/search?engine=youtube&q=编程教程&hl=zh-CN&gl=cn" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

### Pagination with Filter Token

```bash
# Use sp parameter for pagination or advanced filters
curl "https://api.aisa.one/apis/v1/youtube/search?engine=youtube&q=AI&sp=<filter_token>" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

---

## Python Client

```bash
# Basic search
python3 {baseDir}/scripts/youtube_client.py search --query "AI agents tutorial"

# Search with country
python3 {baseDir}/scripts/youtube_client.py search --query "machine learning" --country us

# Search with language
python3 {baseDir}/scripts/youtube_client.py search --query "python tutorial" --lang en

# Full options
python3 {baseDir}/scripts/youtube_client.py search --query "GPT-5 news" --country us --lang en

# Competitor research
python3 {baseDir}/scripts/youtube_client.py search --query "OpenAI tutorial"

# Trend discovery
python3 {baseDir}/scripts/youtube_client.py search --query "AI trends 2025"
```

---

## Use Cases

### 1. Content Gap Analysis

Find what content is ranking well to identify gaps in your strategy:

```python
# Search for top videos in your niche
results = client.search("AI automation tutorial")
# Analyze titles, views, and channels to find opportunities
```

### 2. Competitor Monitoring

Track what competitors are publishing:

```python
# Search for competitor brand + topic
results = client.search("OpenAI GPT tutorial")
# Monitor ranking changes over time
```

### 3. Keyword Research

Discover what topics are trending:

```python
# Search broad topics to see what's popular
results = client.search("artificial intelligence 2025")
# Extract common keywords from top-ranking titles
```

### 4. Audience Research

Understand what your target audience watches:

```python
# Search in specific regions
results = client.search("coding tutorial", country="jp", lang="ja")
# Analyze regional content preferences
```

### 5. SEO Analysis

Analyze how videos rank for specific keywords:

```python
# Track ranking positions for target keywords
keywords = ["AI tutorial", "machine learning basics", "Python AI"]
for kw in keywords:
    results = client.search(kw)
    # Record top 10 videos and their channels
```

---

## API Endpoint Reference

| Endpoint | Method | Description |
|----------|--------|-------------|
| `/youtube/search` | GET | Search YouTube SERP |

## Request Parameters

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| engine | string | Yes | Must be `youtube` |
| q | string | Yes | Search query |
| gl | string | No | Country code (e.g., `us`, `jp`, `uk`, `cn`) |
| hl | string | No | Interface language (e.g., `en`, `ja`, `zh-CN`) |
| sp | string | No | YouTube filter token for pagination/filters |

## Response Format

```json
{
  "search_metadata": {
    "id": "search_id",
    "status": "Success",
    "created_at": "2025-01-15T12:00:00Z",
    "request_time_taken": 1.23,
    "total_time_taken": 1.45
  },
  "search_results": [
    {
      "video_id": "abc123xyz",
      "title": "Complete AI Agents Tutorial 2025",
      "link": "https://www.youtube.com/watch?v=abc123xyz",
      "channel_name": "AI Academy",
      "channel_link": "https://www.youtube.com/@aiacademy",
      "description": "Learn how to build AI agents from scratch...",
      "views": "125K views",
      "published_date": "2 weeks ago",
      "duration": "45:30",
      "thumbnail": "https://i.ytimg.com/vi/abc123xyz/hqdefault.jpg"
    }
  ]
}
```

---

## Country Codes (gl)

| Code | Country |
|------|---------|
| us | United States |
| uk | United Kingdom |
| jp | Japan |
| cn | China |
| de | Germany |
| fr | France |
| kr | South Korea |
| in | India |
| br | Brazil |
| au | Australia |

## Language Codes (hl)

| Code | Language |
|------|----------|
| en | English |
| ja | Japanese |
| zh-CN | Chinese (Simplified) |
| zh-TW | Chinese (Traditional) |
| ko | Korean |
| de | German |
| fr | French |
| es | Spanish |
| pt | Portuguese |
| ru | Russian |

---

## Pricing

| API | Cost |
|-----|------|
| YouTube search | ~$0.002 |

Every response includes `usage.cost` and `usage.credits_remaining`.

---

## Get Started

1. Sign up at [aisa.one](https://aisa.one)
2. Get your API key
3. Add credits (pay-as-you-go)
4. Set environment variable: `export AISA_API_KEY="your-key"`

## Full API Reference

See [API Reference](https://aisa.mintlify.app/api-reference/introduction) for complete endpoint documentation.
