---
name: us-stock-analyst
description: "Professional US stock analysis with financial data, news, social sentiment, and multi-model AI. Comprehensive reports at $0.02-0.10 per analysis."
homepage: https://openclaw.ai
metadata: {"openclaw":{"emoji":"📈","requires":{"bins":["curl","python3"],"env":["AISA_API_KEY"]},"primaryEnv":"AISA_API_KEY"}}
---

# US Stock Analyst 📈

**Professional equity analysis powered by AIsa's unified API platform.**

One API key. Complete market intelligence. AI-powered insights.

## 🔥 What Can You Do?

### Investment Research
```
"Analyze NVDA: financial metrics, analyst estimates, insider trades, 
news sentiment, and AI-powered valuation"
```

### Portfolio Monitoring
```
"Track my portfolio (AAPL, MSFT, GOOGL): daily updates on metrics, 
news, and sentiment changes"
```

### Earnings Analysis
```
"Full Tesla Q4 earnings analysis: results vs estimates, guidance, 
price reaction, analyst updates"
```

### Competitor Analysis
```
"Compare AMD vs NVDA: financials, growth, valuation, market sentiment"
```

### Screening & Discovery
```
"Find tech stocks with P/E < 30, revenue growth > 20%, 
and positive insider activity"
```

## Quick Start

```bash
export AISA_API_KEY="your-key"
```

---

## Core Capabilities

### 📊 Financial Data (MarketPulse APIs)

**Real-time Financial Metrics**
```bash
curl "https://api.aisa.one/apis/v1/financial/financial-metrics/snapshot?ticker=AAPL" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

Returns: Market cap, P/E ratio, revenue, EPS, profit margin, ROE, debt/equity, and more.

**Historical Stock Prices**
```bash
# Daily prices for last 30 days
curl "https://api.aisa.one/apis/v1/financial/prices?ticker=AAPL&start_date=2025-01-01&end_date=2025-01-31&interval=day&interval_multiplier=1" \
  -H "Authorization: Bearer $AISA_API_KEY"

# 5-minute intraday data
curl "https://api.aisa.one/apis/v1/financial/prices?ticker=AAPL&start_date=2025-02-07&end_date=2025-02-07&interval=minute&interval_multiplier=5" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

**Financial Statements**
```bash
# All statements (income, balance, cash flow)
curl "https://api.aisa.one/apis/v1/financial/financial_statements/all?ticker=AAPL" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

**Analyst Estimates**
```bash
# EPS forecasts and ratings
curl "https://api.aisa.one/apis/v1/financial/analyst/eps?ticker=AAPL&period=annual" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

**Insider Trading**
```bash
# Track insider buy/sell activity
curl "https://api.aisa.one/apis/v1/financial/insider/trades?ticker=AAPL" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

**Institutional Ownership**
```bash
# See who owns the stock
curl "https://api.aisa.one/apis/v1/financial/institutional/ownership?ticker=AAPL" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

**SEC Filings**
```bash
# Access 10-K, 10-Q, 8-K filings
curl "https://api.aisa.one/apis/v1/financial/sec/filings?ticker=AAPL" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

---

### 📰 News & Research

**Company News**
```bash
curl "https://api.aisa.one/apis/v1/financial/news?ticker=AAPL&limit=10" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

**Web Search (News & Articles)**
```bash
curl -X POST "https://api.aisa.one/apis/v1/scholar/search/web?query=AAPL+stock+analysis&max_num_results=10" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

**Academic Research**
```bash
curl -X POST "https://api.aisa.one/apis/v1/scholar/search/scholar?query=semiconductor+industry+analysis&max_num_results=5" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

---

### 🐦 Social Sentiment

**Twitter Search**
```bash
curl "https://api.aisa.one/apis/v1/twitter/tweet/advanced_search?query=\$AAPL&queryType=Latest" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

---

### 📺 Video Content

**YouTube Search (Earnings Calls, Analysis)**
```bash
curl "https://api.aisa.one/apis/v1/youtube/search?engine=youtube&q=AAPL+earnings+call&gl=us&hl=en" \
  -H "Authorization: Bearer $AISA_API_KEY"
```

---

### 🤖 AI Analysis (Multi-Model)

**LLM Gateway (OpenAI Compatible)**
```bash
curl -X POST "https://api.aisa.one/v1/chat/completions" \
  -H "Authorization: Bearer $AISA_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "model": "gpt-4",
    "messages": [
      {
        "role": "system",
        "content": "You are a professional equity analyst."
      },
      {
        "role": "user",
        "content": "Analyze Apple stock investment prospects"
      }
    ],
    "temperature": 0.3,
    "max_tokens": 2000
  }'
```

**Supported Models:**
- GPT-4, GPT-4 Turbo (OpenAI)
- Claude 3 Opus, Sonnet, Haiku (Anthropic)
- Gemini 1.5 Pro, Ultra (Google)
- Qwen Max, Plus (Alibaba)
- DeepSeek V2 (DeepSeek)
- Grok (xAI)

---

## Python Client

```bash
# Basic analysis
python3 {baseDir}/scripts/stock_analyst.py analyze --ticker AAPL

# Standard analysis with multiple models
python3 {baseDir}/scripts/stock_analyst.py analyze --ticker NVDA --depth standard --models gpt-4 claude-3-opus

# Deep analysis (all data sources)
python3 {baseDir}/scripts/stock_analyst.py analyze --ticker TSLA --depth deep

# Quick screening
python3 {baseDir}/scripts/stock_analyst.py analyze --ticker MSFT --depth quick

# Save report to file
python3 {baseDir}/scripts/stock_analyst.py analyze --ticker GOOGL --output report.json
```

---

## Analysis Depth Options

| Mode | Time | Cost | Data Sources |
|------|------|------|--------------|
| **quick** | ~10s | $0.01-0.02 | Metrics, News, Twitter, Basic AI |
| **standard** | ~20s | $0.02-0.05 | + Analyst Estimates, Insider Trades, YouTube |
| **deep** | ~30s | $0.05-0.10 | + Statements, Institutional, SEC, Research |

---

## API Reference

| Category | Endpoint | Method | Cost |
|----------|----------|--------|------|
| **Financial Metrics** | `/financial/financial-metrics/snapshot` | GET | $0.002 |
| **Stock Prices** | `/financial/prices` | GET | $0.001 |
| **News** | `/financial/news` | GET | $0.001 |
| **Statements** | `/financial/financial_statements/*` | GET | $0.002 |
| **Analyst Estimates** | `/financial/analyst/eps` | GET | $0.002 |
| **Insider Trades** | `/financial/insider/trades` | GET | $0.001 |
| **Institutional** | `/financial/institutional/ownership` | GET | $0.001 |
| **SEC Filings** | `/financial/sec/filings` | GET | $0.001 |
| **Web Search** | `/scholar/search/web` | POST | $0.001 |
| **Scholar Search** | `/scholar/search/scholar` | POST | $0.002 |
| **Twitter** | `/twitter/tweet/advanced_search` | GET | $0.0004 |
| **YouTube** | `/youtube/search` | GET | $0.002 |
| **LLM** | `/v1/chat/completions` | POST | Token-based |

Every response includes `usage.cost` and `usage.credits_remaining`.

---

## Example Output

```json
{
  "ticker": "NVDA",
  "analysis_date": "2025-02-07T10:30:00Z",
  
  "investment_summary": "NVIDIA maintains dominant position in AI chip market with strong data center revenue growth. Recent Blackwell launch positions company for continued expansion...",
  
  "key_metrics": {
    "market_cap": 1780500000000,
    "pe_ratio": 68.5,
    "revenue": 60922000000,
    "revenue_growth": 1.26,
    "profit_margin": 0.489,
    "roe": 1.152
  },
  
  "sentiment_analysis": {
    "sentiment": "bullish",
    "confidence": "high",
    "key_themes": ["AI dominance", "Data center growth", "Blackwell launch"],
    "summary": "Overwhelmingly positive sentiment following Q4 earnings beat"
  },
  
  "valuation": {
    "assessment": "fairly_valued",
    "price_target_12m": 850.00,
    "reasoning": "Premium valuation justified by AI market leadership and strong growth trajectory"
  },
  
  "data_sources": {
    "Financial Metrics": "Available",
    "Stock News": 10,
    "Analyst Estimates": "Available",
    "Insider Trades": 15,
    "Twitter": "Available",
    "YouTube": 5
  }
}
```

---

## Pricing

**Analysis Costs:**
- Quick: $0.01-0.02 per stock
- Standard: $0.02-0.05 per stock
- Deep: $0.05-0.10 per stock

**Comparison:**
- Bloomberg Terminal: $2,000/month
- FactSet: $1,000/month
- Traditional Analyst Report: $50-500 each
- **AIsa Stock Analyst: $0.02-0.10 each** ✨

**Cost Breakdown:**
```
Standard Analysis ($0.02-0.05):
├── Financial Metrics: $0.002
├── Stock Prices: $0.001
├── Company News: $0.001
├── Analyst Estimates: $0.002
├── Insider Trades: $0.001
├── Twitter: $0.0004
├── YouTube: $0.002
└── LLM Analysis: $0.01-0.04
```

---

## Use Cases

### 1. Investment Research
Screen and analyze stocks before investing:
```python
analyst.analyze_stock("NVDA", depth="deep")
```

### 2. Portfolio Monitoring
Daily updates on your holdings:
```python
for ticker in ["AAPL", "MSFT", "GOOGL"]:
    report = analyst.analyze_stock(ticker, depth="quick")
```

### 3. Earnings Season
Comprehensive earnings analysis:
```python
analyst.analyze_stock("TSLA", depth="standard")
# Check estimates, actual results, guidance, reaction
```

### 4. Insider Tracking
Monitor insider activity:
```python
report = analyst.analyze_stock("META", depth="standard")
print(report['raw_data']['insider_trades'])
```

### 5. Sentiment Analysis
Track market perception:
```python
report = analyst.analyze_stock("COIN", depth="standard")
print(report['sentiment_analysis'])
```

---

## Compliance

**Disclaimer (Always Included):**
> This analysis is for informational purposes only and should not be 
> considered personalized investment advice. Please conduct your own 
> research and consult with licensed financial advisors before making 
> investment decisions.

**Regulatory Compliance:**
- SEC Rule 15c2-1 (not investment advice)
- FINRA regulations (informational only)
- GDPR data privacy compliant

---

## Get Started

1. Sign up at [aisa.one](https://aisa.one)
2. Get your API key
3. Add credits (pay-as-you-go, minimum $5)
4. Set environment variable: `export AISA_API_KEY="your-key"`
5. Run analysis: `python scripts/stock_analyst.py analyze --ticker AAPL`

---

## Full API Documentation

- **API Reference**: https://aisa.mintlify.app/api-reference/introduction
- **Complete Docs**: https://aisa.mintlify.app/llms.txt
- **Support**: support@aisa.one
- **Discord**: https://discord.gg/aisa

---

## About AIsa

**AIsa** - Unified API infrastructure for AI agents.

Single API Key. Pay-Per-Use. Agent-Native.

- Website: https://aisa.one
- Documentation: https://aisa.mintlify.app
