#!/usr/bin/env python3
"""
OpenClaw Twitter - AIsa API Client
Twitter/X data and automation for autonomous agents.

Usage:
    python twitter_client.py user-info --username <username>
    python twitter_client.py tweets --username <username>
    python twitter_client.py search --query <query> [--type Latest|Top]
    python twitter_client.py trends [--woeid <woeid>]
    python twitter_client.py user-search --keyword <keyword>
    python twitter_client.py followers --username <username>
    python twitter_client.py followings --username <username>
    python twitter_client.py login --username <u> --email <e> --password <p> --proxy <proxy>
    python twitter_client.py post --username <u> --text <text>
    python twitter_client.py like --username <u> --tweet-id <id>
    python twitter_client.py retweet --username <u> --tweet-id <id>
"""

import argparse
import json
import os
import sys
import urllib.request
import urllib.parse
import urllib.error
from typing import Any, Dict, Optional


class TwitterClient:
    """OpenClaw Twitter - Twitter/X API Client."""
    
    BASE_URL = "https://api.aisa.one/apis/v1"
    
    def __init__(self, api_key: Optional[str] = None):
        """Initialize the client with an API key."""
        self.api_key = api_key or os.environ.get("AISA_API_KEY")
        if not self.api_key:
            raise ValueError(
                "AISA_API_KEY is required. Set it via environment variable or pass to constructor."
            )
    
    def _request(
        self, 
        method: str, 
        endpoint: str, 
        params: Optional[Dict[str, Any]] = None,
        data: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """Make an HTTP request to the AIsa API."""
        url = f"{self.BASE_URL}{endpoint}"
        
        if params:
            query_string = urllib.parse.urlencode(
                {k: v for k, v in params.items() if v is not None}
            )
            url = f"{url}?{query_string}"
        
        headers = {
            "Authorization": f"Bearer {self.api_key}",
            "Content-Type": "application/json",
            "User-Agent": "OpenClaw-Twitter/1.0"
        }
        
        request_data = None
        if data:
            request_data = json.dumps(data).encode("utf-8")
        
        if method == "POST" and request_data is None:
            request_data = b"{}"
        
        req = urllib.request.Request(url, data=request_data, headers=headers, method=method)
        
        try:
            with urllib.request.urlopen(req, timeout=60) as response:
                return json.loads(response.read().decode("utf-8"))
        except urllib.error.HTTPError as e:
            error_body = e.read().decode("utf-8")
            try:
                return json.loads(error_body)
            except json.JSONDecodeError:
                return {"success": False, "error": {"code": str(e.code), "message": error_body}}
        except urllib.error.URLError as e:
            return {"success": False, "error": {"code": "NETWORK_ERROR", "message": str(e.reason)}}
    
    # ==================== Read APIs ====================
    
    def user_info(self, username: str) -> Dict[str, Any]:
        """Get Twitter user information by username."""
        return self._request("GET", "/twitter/user/info", params={"userName": username})
    
    def user_tweets(self, username: str) -> Dict[str, Any]:
        """Get tweets from a specific user."""
        return self._request("GET", "/twitter/user/user_last_tweet", params={"userName": username})
    
    def search(self, query: str, query_type: str = "Latest") -> Dict[str, Any]:
        """Search for tweets matching a query."""
        return self._request("GET", "/twitter/tweet/advanced_search", params={
            "query": query,
            "queryType": query_type
        })
    
    def tweet_detail(self, tweet_ids: str) -> Dict[str, Any]:
        """Get detailed information about tweets by IDs."""
        return self._request("GET", "/twitter/tweet/tweetById", params={"tweet_ids": tweet_ids})
    
    def trends(self, woeid: int = 1) -> Dict[str, Any]:
        """Get current Twitter trending topics by WOEID (1 = worldwide)."""
        return self._request("GET", "/twitter/trends", params={"woeid": woeid})
    
    def user_search(self, keyword: str) -> Dict[str, Any]:
        """Search for Twitter users by keyword."""
        return self._request("GET", "/twitter/user/search_user", params={"keyword": keyword})
    
    def followers(self, username: str) -> Dict[str, Any]:
        """Get user followers."""
        return self._request("GET", "/twitter/user/user_followers", params={"userName": username})
    
    def followings(self, username: str) -> Dict[str, Any]:
        """Get user followings."""
        return self._request("GET", "/twitter/user/user_followings", params={"userName": username})
    
    # ==================== Write APIs (V3 - requires login) ====================
    
    def login(self, username: str, email: str, password: str, proxy: str, totp_code: str = None) -> Dict[str, Any]:
        """Login to Twitter account."""
        data = {
            "user_name": username,
            "email": email,
            "password": password,
            "proxy": proxy
        }
        if totp_code:
            data["totp_code"] = totp_code
        return self._request("POST", "/twitter/user_login_v3", data=data)
    
    def get_account(self, username: str) -> Dict[str, Any]:
        """Get logged-in account details."""
        return self._request("GET", "/twitter/get_my_x_account_detail_v3", params={"user_name": username})
    
    def send_tweet(self, username: str, text: str, media_base64: str = None, media_type: str = None) -> Dict[str, Any]:
        """Send a tweet."""
        data = {"user_name": username, "text": text}
        if media_base64:
            data["media_data_base64"] = media_base64
        if media_type:
            data["media_type"] = media_type
        return self._request("POST", "/twitter/send_tweet_v3", data=data)
    
    def like(self, username: str, tweet_id: str) -> Dict[str, Any]:
        """Like a tweet."""
        return self._request("POST", "/twitter/like_tweet_v3", data={
            "user_name": username,
            "tweet_id": tweet_id
        })
    
    def retweet(self, username: str, tweet_id: str) -> Dict[str, Any]:
        """Retweet a tweet."""
        return self._request("POST", "/twitter/retweet_v3", data={
            "user_name": username,
            "tweet_id": tweet_id
        })


def main():
    """Main CLI entry point."""
    parser = argparse.ArgumentParser(
        description="OpenClaw Twitter - Twitter/X data and automation",
        formatter_class=argparse.RawDescriptionHelpFormatter
    )
    
    subparsers = parser.add_subparsers(dest="command", help="Command")
    
    # user-info
    user_info = subparsers.add_parser("user-info", help="Get user information")
    user_info.add_argument("--username", "-u", required=True, help="Twitter username")
    
    # tweets
    tweets = subparsers.add_parser("tweets", help="Get user's tweets")
    tweets.add_argument("--username", "-u", required=True, help="Twitter username")
    
    # search
    search = subparsers.add_parser("search", help="Search tweets")
    search.add_argument("--query", "-q", required=True, help="Search query")
    search.add_argument("--type", "-t", choices=["Latest", "Top"], default="Latest", help="Query type")
    
    # detail
    detail = subparsers.add_parser("detail", help="Get tweets by IDs")
    detail.add_argument("--tweet-ids", "-t", required=True, help="Tweet IDs (comma-separated)")
    
    # trends
    trends = subparsers.add_parser("trends", help="Get trending topics")
    trends.add_argument("--woeid", "-w", type=int, default=1, help="WOEID (1=worldwide)")
    
    # user-search
    user_search = subparsers.add_parser("user-search", help="Search users")
    user_search.add_argument("--keyword", "-k", required=True, help="Search keyword")
    
    # followers
    followers = subparsers.add_parser("followers", help="Get user followers")
    followers.add_argument("--username", "-u", required=True, help="Twitter username")
    
    # followings
    followings = subparsers.add_parser("followings", help="Get user followings")
    followings.add_argument("--username", "-u", required=True, help="Twitter username")
    
    # login
    login = subparsers.add_parser("login", help="Login to Twitter account")
    login.add_argument("--username", "-u", required=True, help="Twitter username")
    login.add_argument("--email", "-e", required=True, help="Account email")
    login.add_argument("--password", "-p", required=True, help="Account password")
    login.add_argument("--proxy", required=True, help="Proxy URL")
    login.add_argument("--totp", help="TOTP 2FA code")
    
    # account
    account = subparsers.add_parser("account", help="Check account status")
    account.add_argument("--username", "-u", required=True, help="Twitter username")
    
    # post
    post = subparsers.add_parser("post", help="Send a tweet")
    post.add_argument("--username", "-u", required=True, help="Twitter username")
    post.add_argument("--text", "-t", required=True, help="Tweet text")
    post.add_argument("--media", help="Base64 encoded media")
    post.add_argument("--media-type", choices=["image/jpeg", "image/png", "image/gif", "video/mp4"])
    
    # like
    like = subparsers.add_parser("like", help="Like a tweet")
    like.add_argument("--username", "-u", required=True, help="Twitter username")
    like.add_argument("--tweet-id", "-t", required=True, help="Tweet ID")
    
    # retweet
    retweet = subparsers.add_parser("retweet", help="Retweet a tweet")
    retweet.add_argument("--username", "-u", required=True, help="Twitter username")
    retweet.add_argument("--tweet-id", "-t", required=True, help="Tweet ID")
    
    args = parser.parse_args()
    
    if not args.command:
        parser.print_help()
        sys.exit(1)
    
    try:
        client = TwitterClient()
    except ValueError as e:
        print(json.dumps({"success": False, "error": {"code": "AUTH_ERROR", "message": str(e)}}))
        sys.exit(1)
    
    result = None
    
    if args.command == "user-info":
        result = client.user_info(args.username)
    elif args.command == "tweets":
        result = client.user_tweets(args.username)
    elif args.command == "search":
        result = client.search(args.query, args.type)
    elif args.command == "detail":
        result = client.tweet_detail(args.tweet_ids)
    elif args.command == "trends":
        result = client.trends(args.woeid)
    elif args.command == "user-search":
        result = client.user_search(args.keyword)
    elif args.command == "followers":
        result = client.followers(args.username)
    elif args.command == "followings":
        result = client.followings(args.username)
    elif args.command == "login":
        result = client.login(args.username, args.email, args.password, args.proxy, args.totp)
    elif args.command == "account":
        result = client.get_account(args.username)
    elif args.command == "post":
        result = client.send_tweet(args.username, args.text, args.media, args.media_type)
    elif args.command == "like":
        result = client.like(args.username, args.tweet_id)
    elif args.command == "retweet":
        result = client.retweet(args.username, args.tweet_id)
    
    if result:
        output = json.dumps(result, indent=2, ensure_ascii=False)
        try:
            print(output)
        except UnicodeEncodeError:
            print(json.dumps(result, indent=2, ensure_ascii=True))
        sys.exit(0 if result.get("success", True) else 1)


if __name__ == "__main__":
    main()
