# Cron Optimization (General Patterns)

Cron jobs are best when they are **deterministic, small, and testable**. The biggest recurring costs come from:
- long prompts that re-explain the world every run
- tool calls that return large payloads
- jobs that deliver messages even when nothing changed
- re-running expensive work due to non-idempotent steps

This guide is **model-agnostic**: it avoids naming specific models and focuses on repeatable patterns.

---

## 1) Principle: Cron should be “script-first”

**Goal:** the cron payload should be:
- a short instruction (or one command)
- a single script execution
- a small, bounded output

Prefer:
- `scripts/<job>.py` or `scripts/<job>.mjs`
- inputs/outputs stored in `memory/` or `state/` files

---

## 2) Keep prompts short and stable

Patterns:
- Put long documentation in `references/` files or comments in the script.
- Put configuration knobs in a JSON file (read by script), not in the prompt.
- Avoid embedding large templates unless absolutely necessary.

Rule of thumb:
- If the prompt needs > ~300 tokens of “how to run”, convert it into a script.

---

## 3) Prefer “alert-only delivery”

Most scheduled jobs should not message the user every time.

Recommended pattern:
- Job always writes its output to `memory/<artifact>.md` or `memory/<artifact>.json`.
- Job returns:
  - `NO_REPLY` when nothing is new / actionable
  - a short alert message only when something changed (new signal, error, anomaly)

If you need periodic reporting, separate into two jobs:
- **collector** (frequent, deliver=false, script-first)
- **reporter** (less frequent, deliver=true, summarizes recent artifacts)

---

## 4) Throttles and budgets (anti-surprise)

Add explicit throttles to reduce waste:
- **time-based throttle**: only run heavy steps every N runs
- **change-based throttle**: compute a hash of inputs; skip if unchanged
- **result-size cap**: never output more than X lines / X chars

Budget guardrails (recommended when costs matter):
- If cost spikes (or error repeats), downgrade behavior:
  - reduce scan scope
  - reduce frequency
  - disable delivery and log locally
  - fall back to minimal mode and alert once

---

## 5) Model assignment (tiered, not named)

Model-agnostic guidance:
- Default cron jobs should use the **cheapest tier** that reliably completes the task.
- Upgrade to a higher tier only if:
  - the task repeatedly fails, or
  - accuracy materially matters (e.g., financial decisions, complex synthesis)

---

## 6) Idempotency and crash safety

Cron jobs should be safe to re-run.

Patterns:
- Keep a cursor/state file (e.g., `memory/<job>-state.json`).
- Append-only logs (JSONL) for auditability.
- Avoid side effects unless a human explicitly approves.

---

## 7) Guardrails (human control)

- Do not create/edit/remove cron definitions unless explicitly requested.
- Show the exact cron change and rollback plan before applying.
- If optimization requires reducing coverage or frequency, present options and let the user choose.

Suggested options:
- Profile A: alert-only, minimal scope (cheapest)
- Profile B: balanced (moderate scope)
- Profile C: full scan + periodic report (most coverage)
