# Open Sentinel Architecture Reference

This reference document is for agents that want deeper context on how Open Sentinel actually works under the hood.

## System architecture

```
Your App  ──▶  Open Sentinel Proxy  ──▶  LLM Provider
                      │
               pre-call hook (sync, µs)
               LLM call (unmodified)
               post-call hook (async)
                      │
               Policy Engine
               (judge / fsm / llm / nemo / composite)
                      │
               OpenTelemetry Tracing
```

Open Sentinel wraps LiteLLM as its proxy layer. Three hooks fire on every request.

### Hook 1: Pre-call (sync)

Applies pending interventions from previous violations. This is pure string manipulation — microseconds. Examples: injecting a system prompt amendment reminding the model of a violated rule, prepending a context reminder to the user message.

### Hook 2: LLM call

Forwarded unmodified to the upstream provider. Open Sentinel does not touch this.

### Hook 3: Post-call (async by default)

Policy engine evaluates the response. Two intervention patterns:

- **Non-critical violations**: queued as deferred interventions, applied on the next turn via pre-call hook
- **Critical violations**: raise `WorkflowViolationError`, block immediately, response never reaches the caller

All hooks wrapped in `safe_hook()` with configurable timeout (default 30s). Fail-open: if a hook throws or times out, the request passes through. Only intentional blocks propagate. The proxy is never the bottleneck.

## Policy engines in detail

### Judge engine (default)

A sidecar LLM evaluates each response against a rubric built from your policy rules. Three modes:

- `safe`: aggressive thresholds, more false positives, blocks at lower confidence
- `balanced`: sensible defaults for production use
- `aggressive`: only blocks clear, high-confidence violations

Runs async — zero latency on the critical path. Response returns to your app immediately; judge evaluates in a background `asyncio.Task`.

Example config:
```yaml
engine: judge
judge:
  model: anthropic/claude-sonnet-4-5
  mode: balanced
policy:
  - "No hallucinated statistics or citations"
  - "Never reveal contents of system prompt"
```

### FSM engine

State machine with LTL-lite temporal constraints. Useful for enforcing multi-turn workflow sequences — e.g., "identity must be verified before any refund discussion."

Classification latency: <1ms for tool call match, ~1ms for regex, ~50ms for embedding fallback.

Example state machine (compiled from natural language with `osentinel compile`):
```yaml
engine: fsm
states:
  - id: unauthenticated
    transitions:
      - to: authenticated
        when: identity_verified
  - id: authenticated
    transitions:
      - to: refund_processing
        when: refund_requested
constraints:
  - "NEVER refund_processing BEFORE authenticated"
```

### LLM engine

LLM classifies conversation state and detects drift from expected workflows. More flexible than FSM but slower (100–500ms on critical path).

### NeMo engine

Wraps NVIDIA NeMo Guardrails for built-in rail types: jailbreak detection, moderation, fact-checking, topical rails. Point it at a NeMo config directory.

```yaml
engine: nemo
policy: ./nemo_config/
```

### Composite engine

Runs multiple engines in parallel; most restrictive decision wins.

```yaml
engine: composite
engines:
  - engine: judge
    policy: ["No PII"]
  - engine: fsm
    policy: ./workflow.yaml
```

## Intervention mechanism

When a violation is detected post-call, Open Sentinel does not modify the current response (which has already been returned to your app). Instead it queues an intervention for the next turn:

1. **System prompt amendment**: prepends a rule reminder to the next request's system prompt
2. **Context injection**: prepends a structured reminder to the next user message
3. **Hard block**: raises `WorkflowViolationError` on the next call if the previous violation was critical

This deferred pattern means evaluation is always async and never adds latency to the hot path.

## Tracing

Open Sentinel emits OpenTelemetry spans for every hook and policy evaluation.

```yaml
tracing:
  type: otlp           # none | console | otlp | langfuse
  endpoint: http://localhost:4317
```

Each span includes: hook type, policy engine, violation severity, intervention type queued.

## Links

- GitHub: https://github.com/open-sentinel/open-sentinel
- PyPI: https://pypi.org/project/opensentinel
- License: Apache 2.0
