import {
  promptAccountId,
  DEFAULT_ACCOUNT_ID,
  normalizeAccountId,
  type ChannelOnboardingAdapter,
  OpenClawConfig,
} from "openclaw/plugin-sdk";
import type { CoreConfig } from "./types.js";
import {
  listOlvidAccountIds,
  resolveDefaultOlvidAccountId,
  resolveOlvidAccount,
} from "./accounts.js";

const channel = "olvid" as const;

export const olvidOnboardingAdapter: ChannelOnboardingAdapter = {
  channel,
  getStatus: async ({ cfg }) => {
    const configured = listOlvidAccountIds(cfg as CoreConfig).some((accountId: string) => {
      const account = resolveOlvidAccount({ cfg: cfg as CoreConfig, accountId });
      return Boolean(account.clientKey && account.daemonUrl);
    });
    return {
      channel,
      configured,
      statusLines: [`Olvid: ${configured ? "configured" : "needs setup"}`],
      selectionHint: configured ? "configured" : "privacy first",
      quickstartScore: configured ? 1 : 5,
    };
  },
  configure: async ({ cfg, prompter, accountOverrides, shouldPromptAccountIds }) => {
    const olvidOverride = accountOverrides["olvid"]?.trim();
    const defaultAccountId: string = resolveDefaultOlvidAccountId(cfg as CoreConfig);
    let accountId = olvidOverride ? normalizeAccountId(olvidOverride) : defaultAccountId;

    if (shouldPromptAccountIds && !olvidOverride) {
      accountId = await promptAccountId({
        cfg,
        prompter,
        label: "Olvid",
        currentId: accountId,
        listAccountIds: listOlvidAccountIds as (core: OpenClawConfig) => string[],
        defaultAccountId: defaultAccountId,
      });
    }

    let next: CoreConfig = cfg as CoreConfig;
    const resolvedAccount = resolveOlvidAccount({
      cfg: next,
      accountId: accountId,
    });
    const allowEnv = accountId === DEFAULT_ACCOUNT_ID;

    /*
     ** determine / ask user for daemon target
     */
    let daemonUrl = resolvedAccount.daemonUrl;
    if (allowEnv && !daemonUrl && Boolean(process.env.OLVID_DAEMON_TARGET?.trim())) {
      const keepEnv = await prompter.confirm({
        message: "OLVID_DAEMON_TARGET detected. Use env var?",
        initialValue: true,
      });
      if (keepEnv) {
        daemonUrl = process.env.OLVID_DAEMON_TARGET?.trim();
      } else {
        daemonUrl = String(
          await prompter.text({
            message:
              "Enter url to access your Olvid daemon instance (e.g., http://localhost:50051)",
            validate: (value: string) => (value?.trim() ? undefined : "Required"),
          }),
        );
      }
    } else if (daemonUrl) {
      const keep = await prompter.confirm({
        message: "Daemon target already configured. Keep it?",
        initialValue: true,
      });
      if (!keep) {
        daemonUrl = String(
          await prompter.text({
            message:
              "Enter url to access your Olvid daemon instance (e.g., http://localhost:50051)",
            validate: (value: string) => (value?.trim() ? undefined : "Required"),
          }),
        );
      }
    } else {
      daemonUrl = String(
        await prompter.text({
          message: "Enter url to access your Olvid daemon instance (e.g., http://localhost:50051)",
          validate: (value: string) => (value?.trim() ? undefined : "Required"),
        }),
      );
    }

    /*
     ** determine / ask user for client key
     */
    let clientKey = resolvedAccount.clientKey;
    if (allowEnv && !clientKey && Boolean(process.env.OLVID_CLIENT_KEY?.trim())) {
      const keepEnv = await prompter.confirm({
        message: "OLVID_CLIENT_KEY detected. Use env var?",
        initialValue: true,
      });
      if (keepEnv) {
        clientKey = process.env.OLVID_CLIENT_KEY?.trim();
      } else {
        clientKey = String(
          await prompter.text({
            message: "Enter your client key",
            validate: (value: string) => (value?.trim() ? undefined : "Required"),
          }),
        );
      }
    } else if (clientKey) {
      const keep = await prompter.confirm({
        message: "Client key already configured. Keep it?",
        initialValue: true,
      });
      if (!keep) {
        clientKey = String(
          await prompter.text({
            message: "Enter your client key",
            validate: (value: string) => (value?.trim() ? undefined : "Required"),
          }),
        );
      }
    } else {
      clientKey = String(
        await prompter.text({
          message: "Enter your client key",
          validate: (value: string) => (value?.trim() ? undefined : "Required"),
        }),
      );
    }

    /*
     ** save in config
     */
    if (accountId === DEFAULT_ACCOUNT_ID) {
      if (accountId === DEFAULT_ACCOUNT_ID) {
        next = {
          ...next,
          channels: {
            ...next.channels,
            olvid: {
              ...next.channels?.["olvid"],
              enabled: true,
              daemonUrl,
              clientKey,
            },
          },
        };
      } else {
        next = {
          ...next,
          channels: {
            ...next.channels,
            olvid: {
              ...next.channels?.["olvid"],
              enabled: true,
              accounts: {
                ...next.channels?.["olvid"]?.accounts,
                [accountId]: {
                  ...next.channels?.["olvid"]?.accounts?.[accountId],
                  enabled: next.channels?.["olvid"]?.accounts?.[accountId]?.enabled ?? true,
                  daemonUrl: daemonUrl,
                  clientKey,
                },
              },
            },
          },
        };
      }
    }

    return { cfg: next, accountId: accountId };
  },
};
