import { DEFAULT_ACCOUNT_ID, normalizeAccountId } from "openclaw/plugin-sdk";
import type { CoreConfig, OlvidAccountConfig } from "./types.js";

export type OlvidClientKeySource = "env" | "config" | "none";
export type OlvidDaemonUrlSource = "env" | "config" | "none";

export type ResolvedOlvidAccount = {
  accountId: string;
  enabled: boolean;
  name?: string;
  clientKey?: string;
  daemonUrl?: string;
  clientKeySource: OlvidClientKeySource;
  daemonUrlSource: OlvidDaemonUrlSource;
  config: OlvidAccountConfig;
};

function listConfiguredAccountIds(cfg: CoreConfig): string[] {
  const accounts = cfg.channels?.olvid?.accounts;
  if (!accounts || typeof accounts !== "object") {
    return [];
  }
  return Object.keys(accounts).filter(Boolean);
}

export function listOlvidAccountIds(cfg: CoreConfig): string[] {
  const ids = listConfiguredAccountIds(cfg);
  if (ids.length === 0) {
    return [DEFAULT_ACCOUNT_ID];
  }
  return ids.sort((a, b) => a.localeCompare(b));
}

export function resolveDefaultOlvidAccountId(cfg: CoreConfig): string {
  const ids = listOlvidAccountIds(cfg);
  if (ids.includes(DEFAULT_ACCOUNT_ID)) {
    return DEFAULT_ACCOUNT_ID;
  }
  return ids[0] ?? DEFAULT_ACCOUNT_ID;
}

function resolveAccountConfig(cfg: CoreConfig, accountId: string): OlvidAccountConfig | undefined {
  const accounts = cfg.channels?.olvid?.accounts;
  if (!accounts || typeof accounts !== "object") {
    return undefined;
  }
  return accounts[accountId] as ResolvedOlvidAccount | undefined;
}

function mergeOlvidAccountConfig(cfg: CoreConfig, accountId: string): OlvidAccountConfig {
  const { accounts: _ignored, ...base } = (cfg.channels?.olvid ?? {}) as OlvidAccountConfig & {
    accounts?: unknown;
  };
  const account = resolveAccountConfig(cfg, accountId) ?? {};
  return { ...base, ...account };
}

export function resolveOlvidAccount(params: {
  cfg: CoreConfig;
  accountId?: string | null;
}): ResolvedOlvidAccount {
  const accountId = normalizeAccountId(params.accountId);
  const baseEnabled = params.cfg.channels?.olvid?.enabled !== false;
  const merged = mergeOlvidAccountConfig(params.cfg, accountId);
  const accountEnabled = merged.enabled !== false;
  const enabled = baseEnabled && accountEnabled;

  const allowEnv = accountId === DEFAULT_ACCOUNT_ID;
  const envClientKey = allowEnv ? process.env.OLVID_CLIENT_KEY?.trim() : undefined;
  const envDaemonUrl = allowEnv ? process.env.OLVID_DAEMON_TARGET?.trim() : undefined;
  const configClientKey = merged.clientKey?.trim();
  const configDaemonUrl = merged.daemonUrl?.trim();
  const clientKey = configClientKey || envClientKey;
  const daemonUrl = configDaemonUrl || envDaemonUrl;

  const clientKeySource: OlvidClientKeySource = configClientKey
    ? "config"
    : envClientKey
      ? "env"
      : "none";
  const daemonUrlSource: OlvidDaemonUrlSource = configDaemonUrl
    ? "config"
    : envDaemonUrl
      ? "env"
      : "none";

  return {
    accountId,
    enabled,
    name: merged.name?.trim() || undefined,
    clientKey,
    daemonUrl,
    clientKeySource: clientKeySource,
    daemonUrlSource: daemonUrlSource,
    config: merged,
  };
}
