# 🔪 oktk - LLM Token Optimizer

### Reduce AI API Costs by 60-90% | GPT-4 & Claude Token Saver

> **By Buba Draugelis** 🇱🇹

Compresses CLI command outputs before sending to your LLM, dramatically cutting token usage and API costs.

[![OpenClaw Skill](https://img.shields.io/badge/OpenClaw-Skill-blue)](https://clawhub.com)
[![License: MIT](https://img.shields.io/badge/License-MIT-green.svg)](LICENSE)
[![Node.js](https://img.shields.io/badge/Node.js-18+-brightgreen)](https://nodejs.org)

---

## 🤔 The Problem

When you use an AI assistant to run terminal commands, **the entire output goes into the LLM context**:

```bash
$ git status
# Returns 60+ lines of text
# = ~800 tokens
# Your AI reads ALL of it
# You PAY for ALL of it
```

This wastes:
- 💸 **Money** — tokens cost real dollars
- 📊 **Context window** — less space for actual conversation
- ⏱️ **Time** — more tokens = slower responses

---

## ✅ The Solution

**oktk** compresses command outputs before they reach the LLM:

```
┌─────────────┐        ┌─────────────┐        ┌─────────────┐
│   Command   │   ──►  │    oktk     │   ──►  │     LLM     │
│  (800 tok)  │        │  compress   │        │  (80 tok)   │
└─────────────┘        └─────────────┘        └─────────────┘
                              │
                        90% SAVED
```

**Same information. 90% fewer tokens.**

---

## 📊 When Does It Work?

oktk automatically compresses these commands:

| Command | What oktk extracts | Token Savings |
|---------|-------------------|:-------------:|
| `git status` | Branch, ahead/behind, file counts | **90%** |
| `git log` | Hash + message + author (1 line each) | **85%** |
| `git diff` | Summary: files changed, +/- lines | **80%** |
| `npm test` | ✅ passed / ❌ failed + counts | **98%** |
| `cargo test` | Same as npm test | **98%** |
| `pytest` | Pass/fail summary | **95%** |
| `ls -la` | Grouped by type, sizes only | **83%** |
| `curl` | Status + headers + truncated body | **97%** |
| `grep` | Match count + first results | **80%** |

---

## 🔍 Concrete Example

### Before oktk — sent to LLM (800 tokens):

```
On branch main
Your branch is ahead of 'origin/main' by 3 commits.
  (use "git push" to publish your local commits)

Changes not staged for commit:
  (use "git add <file>..." to update what will be committed)
  (use "git restore <file>..." to discard changes in working directory)
        modified:   src/components/Button.jsx
        modified:   src/components/Header.jsx
        modified:   src/components/Modal.jsx
        modified:   src/utils/format.js
        modified:   src/utils/validate.js
        modified:   package.json

Untracked files:
  (use "git add <file>..." to include in what will be committed)
        src/components/Footer.jsx
        src/components/Sidebar.jsx
        tests/

no changes added to commit (use "git add" and/or "git commit -a")
```

### After oktk — sent to LLM (80 tokens):

```
📍 main
↑ Ahead 3 commits
✏️  Modified: 6
❓ Untracked: 3
```

**The AI gets the same useful information. You pay 90% less.**

---

## ⚙️ How It Works

```
1. You run:     git status
                    │
2. Command executes, produces 800 tokens of output
                    │
3. oktk detects:   "This is a git command"
                    │
4. GitFilter:      Extracts branch, status, file counts
                    │
5. Result:         80 tokens sent to LLM
                    │
6. Cache:          Same command next time = instant
```

### Technical Flow

```
Command Output
      │
      ▼
┌─────────────────┐
│  Filter Router  │ ← Detects: git? npm? ls? curl?
└────────┬────────┘
         │
    ┌────┴────┬────────┬─────────┐
    ▼         ▼        ▼         ▼
┌───────┐ ┌──────┐ ┌───────┐ ┌──────┐
│  Git  │ │ Test │ │ Files │ │ curl │  ← Specialized filters
└───┬───┘ └──┬───┘ └───┬───┘ └──┬───┘
    │        │         │        │
    └────────┴────┬────┴────────┘
                  ▼
           ┌──────────┐
           │  Output  │ ← Compressed result
           └──────────┘
```

---

## 🛡️ Safety — Never Breaks

oktk has **3 fallback layers**:

```
1. Specialized Filter  →  Try GitFilter, TestFilter, etc.
         │ fails?
         ▼
2. PassthroughFilter   →  Basic safe compression
         │ fails?
         ▼
3. Raw Output          →  Return original (same as no oktk)
```

| Guarantee | How |
|-----------|-----|
| ✅ Never crashes | 3-layer fallback |
| ✅ Never loses data | `--raw` flag always works |
| ✅ Preserves errors | Error messages never filtered |
| ✅ Hides secrets | Auto-redacts API keys, tokens |

---

## 🚀 Quick Start

### Basic Usage

```bash
# Compress git status
node skills/oktk/scripts/oktk.js git status

# Compress git log
node skills/oktk/scripts/oktk.js git log -n 10

# See raw output (bypass compression)
node skills/oktk/scripts/oktk.js git status --raw

# Check your savings
node skills/oktk/scripts/oktk.js gain

# List available filters
node skills/oktk/scripts/oktk.js filters

# Clear cache
node skills/oktk/scripts/oktk.js cache --clear
```

### See Your Savings

```bash
$ node skills/oktk/scripts/oktk.js gain

📊 Token Savings (All time)
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

Commands filtered: 1,247
Tokens saved:      456,789 (78%)

By filter:
  git     234,567 (82%)
  test    156,789 (95%)
  files    43,212 (81%)
  curl     22,221 (94%)

💰 At $0.01/1K tokens = $4.57 saved this week
```

---

## 📁 Project Structure

```
skills/oktk/
├── SKILL.md                 # ClawHub metadata
├── README.md                # This file
├── scripts/
│   ├── oktk.js              # Main CLI
│   ├── cache.js             # Hash-based caching
│   ├── analytics.js         # Savings tracking
│   └── filters/
│       ├── BaseFilter.js    # Base class
│       ├── GitFilter.js     # git commands
│       ├── TestFilter.js    # npm/cargo/pytest
│       ├── FilesFilter.js   # ls/find/tree
│       ├── NetworkFilter.js # curl/wget
│       ├── SearchFilter.js  # grep/ripgrep
│       └── PassthroughFilter.js
├── test/
│   └── test.js              # 24 tests
└── examples/
    └── *.js
```

---

## ⚙️ Configuration

### Environment Variables

```bash
export OKTK_DISABLE=true      # Turn off completely
export OKTK_CACHE_TTL=3600    # Cache lifetime (seconds)
export OKTK_DEBUG=1           # Show debug info
```

### Emergency Off Switch

```bash
# Method 1: Environment
export OKTK_DISABLE=true

# Method 2: File
touch ~/.oktk/EMERGENCY

# Method 3: Per-command
node skills/oktk/scripts/oktk.js git status --raw
```

---

## 🧪 Testing

```bash
# Run all 24 tests
node skills/oktk/test/test.js

# Run examples
node skills/oktk/examples/git-status-example.js
```

---

## 🔧 Add Your Own Filter

```javascript
// scripts/filters/MyFilter.js
const BaseFilter = require('./BaseFilter');

class MyFilter extends BaseFilter {
  async apply(output, context = {}) {
    if (!this.canFilter(output)) return output;
    
    // Your compression logic
    const lines = output.split('\n');
    const summary = `Found ${lines.length} lines`;
    
    return summary;
  }
}

module.exports = MyFilter;
```

Add to `oktk.js`:
```javascript
const MyFilter = require('./filters/MyFilter');
// ...
this.filters = [
  [/^mycommand\b/i, MyFilter],
  // ... existing
];
```

---

## 📈 Performance

| Metric | Value |
|--------|-------|
| Filter speed | < 10ms |
| Cache hit rate | ~60% |
| Memory | < 50MB |

---

## 📜 License

MIT License

---

## 🙏 Credits

- Inspired by [rtk](https://github.com/rtk-ai/rtk)
- Built for [OpenClaw](https://openclaw.ai)

---

<div align="center">

**Made with ❤️ in Lithuania 🇱🇹**

*Saving tokens, saving money, one command at a time.*

</div>
