---
name: officeclaw
description: Connect to personal Microsoft accounts via Microsoft Graph API to manage email, calendar events, and tasks. Use this skill when the user needs to read/write Outlook mail, manage calendar appointments, or handle Microsoft To Do tasks.
license: Apache-2.0
homepage: https://github.com/danielithomas/officeclaw
user-invocable: true
compatibility: Requires Python 3.9+, network access to graph.microsoft.com, and one-time OAuth setup
metadata:
  author: Daniel Thomas
  version: "1.0.1"
  openclaw:
    requires:
      anyBins: ["python", "python3", "officeclaw"]
      env: []
    os: ["darwin", "linux", "win32"]
---

# OfficeClaw: Microsoft Graph API Integration

Connect your OpenClaw agent to personal Microsoft accounts (Outlook.com, Hotmail, Live) to manage email, calendar, and tasks through the Microsoft Graph API.

## Installation

Install from PyPI:

```bash
pip install officeclaw
```

Or with uv:

```bash
uv pip install officeclaw
```

Verify installation:

```bash
officeclaw --version
```

## Setup (One-Time)

> **Quick start:** OfficeClaw ships with a default app registration — just run `officeclaw auth login` and go. No Azure setup needed.
>
> **Advanced:** Want full control? Create your own Azure App Registration (free, ~5 minutes) and set `OFFICECLAW_CLIENT_ID` in your `.env`. See [Microsoft's guide](https://learn.microsoft.com/en-us/entra/identity-platform/quickstart-register-app) or follow the steps below.

### 1. Create an Azure App Registration

1. Go to [entra.microsoft.com](https://entra.microsoft.com) → App registrations → New registration
2. Name: `officeclaw` (or anything you like)
3. Supported account types: **Personal Microsoft accounts only**
4. Redirect URI: leave blank (not needed for device code flow)
5. Click **Register**
6. Copy the **Application (client) ID** — this is your `OFFICECLAW_CLIENT_ID`
7. Go to **Authentication** → Advanced settings → **Allow public client flows** → **Yes** → Save
8. Go to **API permissions** → Add permission → Microsoft Graph → Delegated permissions. Choose based on your needs:

**Read-only (safest):**
- `Mail.Read`, `Calendars.Read`, `Tasks.ReadWrite`*

**Full access (all features including send/delete):**
- `Mail.Read`, `Mail.ReadWrite`, `Mail.Send`
- `Calendars.Read`, `Calendars.ReadWrite`
- `Tasks.ReadWrite`

*\*Tasks.ReadWrite is the minimum available scope for Microsoft To Do — there is no read-only option.*

> **Least privilege:** Only grant the permissions you actually need. If you only want to read emails and calendar, skip `Mail.ReadWrite`, `Mail.Send`, and `Calendars.ReadWrite`. OfficeClaw will gracefully error on commands that require missing permissions.

### 2. Configure Environment

Create a `.env` file in your skill directory:

```bash
OFFICECLAW_CLIENT_ID=your-client-id-here

# Capability gates (disabled by default for safety)
# OFFICECLAW_ENABLE_SEND=true    # Allow sending/replying/forwarding emails
# OFFICECLAW_ENABLE_DELETE=true   # Allow deleting emails, events, and tasks
```

No client secret needed for device code flow. Write operations (send, delete) are **disabled by default** — enable only what you need.

### 3. Authenticate

```bash
officeclaw auth login
```

This displays a URL and code. Open the URL in a browser, enter the code, and sign in with your Microsoft account. Tokens are stored securely in `~/.officeclaw/token_cache.json` (permissions 600).

## When to Use This Skill

Activate this skill when the user needs to:

### Email Operations
- **Read emails**: "Show me my latest emails", "Find emails from john@example.com"
- **Send emails**: "Send an email to...", "Reply to the last email from..."
- **Manage inbox**: "Mark emails as read", "Archive old emails", "Delete emails"

### Calendar Operations
- **View events**: "What's on my calendar today?", "Show meetings this week"
- **Create events**: "Schedule a meeting with...", "Add dentist appointment on Friday"
- **Update events**: "Move the 2pm meeting to 3pm", "Cancel tomorrow's standup"

### Task Management
- **List tasks**: "What's on my to-do list?", "Show incomplete tasks"
- **Create tasks**: "Add 'buy groceries' to my tasks", "Create a task to review report"
- **Complete tasks**: "Mark 'finish proposal' as done", "Complete all shopping tasks"

## Available Commands

### Authentication

```bash
officeclaw auth login       # Authenticate via device code flow
officeclaw auth status      # Check authentication status
officeclaw auth logout      # Clear stored tokens
```

### Mail Commands

```bash
officeclaw mail list --limit 10                # List recent messages
officeclaw mail list --unread                   # List unread messages only
officeclaw mail get <message-id>               # Get specific message
officeclaw mail send --to user@example.com --subject "Hello" --body "Message text"
officeclaw mail send --to user@example.com --subject "Report" --body "Attached" --attachment report.pdf
officeclaw mail search --query "from:boss@example.com"
officeclaw mail archive <message-id>           # Archive a message
officeclaw mail mark-read <message-id>         # Mark as read
officeclaw --json mail list                    # JSON output for parsing
```

### Calendar Commands

```bash
officeclaw calendar list --start 2026-02-01 --end 2026-02-28
officeclaw calendar create \
  --subject "Team Meeting" \
  --start "2026-02-15T10:00:00" \
  --end "2026-02-15T11:00:00" \
  --location "Conference Room"
officeclaw calendar get <event-id>
officeclaw calendar update <event-id> --subject "Updated Meeting"
officeclaw calendar delete <event-id>
officeclaw --json calendar list --start 2026-02-01 --end 2026-02-28
```

### Task Commands

```bash
officeclaw tasks list-lists                              # List task lists
officeclaw tasks list --list-id <list-id>                # List tasks
officeclaw tasks list --list-id <list-id> --status active  # Active tasks only
officeclaw tasks create --list-id <list-id> --title "Complete report" --due-date "2026-02-20"
officeclaw tasks complete --list-id <list-id> --task-id <task-id>
officeclaw tasks reopen --list-id <list-id> --task-id <task-id>
```

## Output Format

Use `--json` flag for structured JSON output:

```bash
officeclaw --json mail list
```

Returns:
```json
{
  "status": "success",
  "data": [
    {
      "id": "AAMkADEzN...",
      "subject": "Meeting Notes",
      "from": {"emailAddress": {"address": "sender@example.com"}},
      "receivedDateTime": "2026-02-12T10:30:00Z",
      "isRead": false
    }
  ]
}
```

## Error Handling

Common errors and solutions:

| Error | Cause | Solution |
|-------|-------|----------|
| `AuthenticationError` | Not logged in or token expired | Run `officeclaw auth login` |
| `AccessDenied` | Missing permissions | Re-authenticate with required scopes |
| `ResourceNotFound` | Invalid ID | Verify the ID exists |
| `RateLimitError` | Too many API calls | Wait 60 seconds and retry |

## Guidelines for Agents

When using this skill:

1. **Confirm destructive actions**: Ask before deleting or sending
2. **Summarize results**: Don't show raw JSON, provide summaries
3. **Handle errors gracefully**: Guide user through re-authentication
4. **Respect privacy**: Don't log email content
5. **Use JSON mode**: For programmatic parsing, use `--json` flag
6. **Batch operations**: Process multiple items efficiently

## Security & Privacy

- **Write operations disabled by default**: Send, reply, forward, and delete are all blocked unless explicitly enabled via `OFFICECLAW_ENABLE_SEND` and `OFFICECLAW_ENABLE_DELETE` environment variables. This prevents accidental or unauthorised write actions.
- **No client secret required**: Uses device code flow (public client) by default
- **Least-privilege permissions**: You choose which Graph API scopes to grant — read-only is sufficient for most use cases. See the setup guide above.
- **Tokens stored securely**: `~/.officeclaw/token_cache.json` with 600 file permissions
- **No data storage**: OfficeClaw passes data through, never stores email/calendar content
- **No telemetry**: No usage data collected
- **Your own Azure app**: Each user creates their own Azure app registration with their own client ID — no shared credentials

## Troubleshooting

If the skill isn't working:

1. **Check authentication**: Run `officeclaw auth status`
2. **Re-authenticate**: Run `officeclaw auth login`
3. **Verify network**: Ensure `graph.microsoft.com` is reachable
4. **Check environment**: Verify `OFFICECLAW_CLIENT_ID` is set in `.env`

## References

- [OfficeClaw on GitHub](https://github.com/danielithomas/officeclaw)
- [OfficeClaw on PyPI](https://pypi.org/project/officeclaw/)
- [Microsoft Graph API](https://docs.microsoft.com/graph/)
- [OpenClaw](https://docs.openclaw.ai)
