from src.connectors.odoo_client import OdooClient
from datetime import datetime, timedelta
import collections
from src.logic.openclaw_intelligence import OpenClawIntelligence
import logging

logger = logging.getLogger(__name__)

class IntelligenceEngine:
    def __init__(self, client: OdooClient):
        self.client = client
        self._ai_error = None
        try:
            from src.logic.openclaw_intelligence import OpenClawIntelligence
            self.ai = OpenClawIntelligence()
        except Exception as e:
            self.ai = None
            self._ai_error = str(e)
            logger.warning(f"OpenClawIntelligence not available: {e}")

    def get_ai_anomaly_report(self):
        """Use OpenClaw intelligence to perform advanced anomaly detection on recent moves."""
        if not self.ai:
            error_msg = f"OpenClaw intelligence not configured: {self._ai_error}" if self._ai_error else "OpenClaw intelligence not configured."
            return {"error": error_msg, "fallback": "Use detect_anomalies() for rules-based analysis"}
        
        moves = self.client.search_read(
            'account.move',
            domain=[('state', '=', 'posted')],
            fields=['name', 'partner_id', 'amount_total', 'invoice_date', 'move_type', 'ref'],
            limit=50
        )
        try:
            return self.ai.analyze_anomalies(moves)
        except Exception as e:
            return f"AI Analysis failed: {str(e)}"

    def ask(self, query: str):
        """Perform a natural language query using Odoo data as context."""
        if not self.ai:
            error_msg = f"OpenClaw intelligence not configured: {self._ai_error}" if self._ai_error else "OpenClaw intelligence not configured."
            return {"error": error_msg, "fallback": "Use structured reports for deterministic analysis"}

        # Strategy: Fetch a broad set of recent data as context
        context = {
            "recent_moves": self.client.search_read(
                'account.move', 
                domain=[('state', '=', 'posted')], 
                fields=['name', 'partner_id', 'amount_total', 'invoice_date', 'move_type', 'ref'],
                limit=20
            ),
            "cash_flow": self.client.search_read(
                'account.account', 
                domain=[('account_type', '=', 'asset_cash')], 
                fields=['name', 'code', 'current_balance']
            )
        }
        return self.ai.natural_language_query(query, context)

    def get_vat_report(self, date_from, date_to):
        """
        Calculate VAT liability (Output vs Input) for a given period.
        Based on account.move.line tax_tag_ids or accounts.
        """
        # Fetch journal items for the period that have tax tags or are in tax accounts
        # In Odoo, tax lines usually have a tax_line_id or tax_tag_ids.
        # We'll look for lines with tax_tag_ids.
        
        domain = [
            ('date', '>=', date_from),
            ('date', '<=', date_to),
            ('parent_state', '=', 'posted'),
            ('tax_line_id', '!=', False)
        ]
        
        tax_lines = self.client.search_read(
            'account.move.line',
            domain=domain,
            fields=['debit', 'credit', 'tax_line_id', 'tax_tag_ids', 'move_type']
        )
        
        output_vat = 0.0
        input_vat = 0.0
        
        for line in tax_lines:
            # move_type in account.move: 'out_invoice', 'out_refund' -> Output VAT
            # move_type in account.move: 'in_invoice', 'in_refund' -> Input VAT
            # Note: account.move.line has move_type as well.
            
            if line['move_type'] in ['out_invoice', 'out_refund']:
                # For sales, credit is usually positive VAT, debit is refund
                output_vat += (line['credit'] - line['debit'])
            elif line['move_type'] in ['in_invoice', 'in_refund']:
                # For purchases, debit is usually positive VAT, credit is refund
                input_vat += (line['debit'] - line['credit'])
        
        return {
            'period': f"{date_from} to {date_to}",
            'output_vat': output_vat,
            'input_vat': input_vat,
            'net_vat_liability': output_vat - input_vat,
            'currency': self._get_company_currency()
        }

    def get_trend_analysis(self, months=6):
        """
        Compare MoM revenue and spending.
        """
        today = datetime.now()
        trends = []
        
        for i in range(months):
            start_date = (today.replace(day=1) - timedelta(days=i*30)).replace(day=1)
            # Find the end of that month
            next_month = (start_date + timedelta(days=32)).replace(day=1)
            end_date = next_month - timedelta(days=1)
            
            start_str = start_date.strftime('%Y-%m-%d')
            end_str = end_date.strftime('%Y-%m-%d')
            
            # Fetch Revenue (out_invoice)
            rev_domain = [
                ('move_type', '=', 'out_invoice'),
                ('state', '=', 'posted'),
                ('invoice_date', '>=', start_str),
                ('invoice_date', '<=', end_str)
            ]
            revenues = self.client.search_read('account.move', domain=rev_domain, fields=['amount_total'])
            total_rev = sum(r['amount_total'] for r in revenues)
            
            # Fetch Spending (in_invoice)
            exp_domain = [
                ('move_type', '=', 'in_invoice'),
                ('state', '=', 'posted'),
                ('invoice_date', '>=', start_str),
                ('invoice_date', '<=', end_str)
            ]
            expenses = self.client.search_read('account.move', domain=exp_domain, fields=['amount_total'])
            total_exp = sum(e['amount_total'] for e in expenses)
            
            trends.append({
                'month': start_date.strftime('%Y-%m'),
                'revenue': total_rev,
                'spending': total_exp
            })
            
        trends.reverse() # Chronological order
        return trends

    def detect_anomalies(self):
        """
        Basic check for unusual transactions.
        """
        anomalies = []
        
        # 1. Duplicate Bills (same partner, same reference, different id)
        # We'll check recent vendor bills
        recent_bills = self.client.search_read(
            'account.move',
            domain=[('move_type', '=', 'in_invoice'), ('state', '!=', 'cancel')],
            fields=['ref', 'partner_id', 'amount_total', 'invoice_date'],
            limit=500
        )
        
        seen_refs = collections.defaultdict(list)
        for bill in recent_bills:
            if bill['ref'] and bill['partner_id']:
                key = (bill['partner_id'][0], bill['ref'])
                seen_refs[key].append(bill)
        
        for key, bills in seen_refs.items():
            if len(bills) > 1:
                anomalies.append({
                    'type': 'potential_duplicate',
                    'details': f"Multiple bills found for partner {bills[0]['partner_id'][1]} with reference {key[1]}",
                    'entities': [b['id'] for b in bills]
                })

        # 2. Missing Tax Codes on posted invoices
        missing_tax = self.client.search_read(
            'account.move',
            domain=[
                ('move_type', 'in', ['out_invoice', 'in_invoice']),
                ('state', '=', 'posted'),
                ('amount_tax', '=', 0),
                ('amount_total', '>', 0)
            ],
            fields=['name', 'partner_id', 'amount_total'],
            limit=100
        )
        for move in missing_tax:
            anomalies.append({
                'type': 'missing_tax',
                'details': f"Posted transaction {move['name']} has zero tax amount.",
                'entity_id': move['id']
            })

        # 3. Outliers in payment amounts (very simple: > 5x average of last 100 payments)
        payments = self.client.search_read(
            'account.payment',
            domain=[('state', '=', 'posted')],
            fields=['amount', 'name'],
            limit=100,
            order='id desc'
        )
        if payments:
            avg_payment = sum(p['amount'] for p in payments) / len(payments)
            for p in payments[:10]: # Check most recent 10 against the avg of 100
                if p['amount'] > avg_payment * 5:
                    anomalies.append({
                        'type': 'payment_outlier',
                        'details': f"Payment {p['name']} is significantly higher than average ({p['amount']} vs avg {avg_payment:.2f})",
                        'entity_id': p['id']
                    })

        return anomalies

    def _get_company_currency(self):
        company = self.client.search_read('res.company', limit=1, fields=['currency_id'])
        if company:
            return company[0]['currency_id'][1]
        return "Unknown"
